<?php
/**
 * CodeIgniter Migration for Community Card Fields
 * File: application/migrations/003_add_community_card_fields.php
 */

defined('BASEPATH') OR exit('No direct script access allowed');

class Migration_Add_community_card_fields extends CI_Migration {

    public function up() {
        // Add community card fields to patients table
        $fields = [
            'coverage_type' => [
                'type' => 'ENUM',
                'constraint' => ['none', 'insurance', 'community_card'],
                'default' => 'none',
                'null' => TRUE
            ],
            'community_card_number' => [
                'type' => 'VARCHAR',
                'constraint' => 20,
                'null' => TRUE
            ],
            'community_card_issue_date' => [
                'type' => 'DATE',
                'null' => TRUE
            ],
            'community_card_expiry' => [
                'type' => 'DATE',
                'null' => TRUE
            ],
            'community_card_status' => [
                'type' => 'VARCHAR',
                'constraint' => 50,
                'null' => TRUE
            ],
            'community_card_coverage' => [
                'type' => 'ENUM',
                'constraint' => ['basic', 'plus', 'premium'],
                'default' => 'basic',
                'null' => TRUE
            ]
        ];

        $this->dbforge->add_column('patients', $fields);

        // Add indexes for better performance
        $this->db->query('CREATE INDEX idx_patients_coverage_type ON patients (coverage_type)');
        $this->db->query('CREATE INDEX idx_patients_community_card_number ON patients (community_card_number)');
        $this->db->query('CREATE INDEX idx_patients_community_card_expiry ON patients (community_card_expiry)');

        // Create Community Card Registry table for verification and management
        $this->dbforge->add_field([
            'id' => [
                'type' => 'INT',
                'constraint' => 11,
                'unsigned' => TRUE,
                'auto_increment' => TRUE
            ],
            'card_number' => [
                'type' => 'VARCHAR',
                'constraint' => 20,
            ],
            'holder_name' => [
                'type' => 'VARCHAR',
                'constraint' => 255,
            ],
            'holder_id_number' => [
                'type' => 'VARCHAR',
                'constraint' => 50,
                'null' => TRUE
            ],
            'holder_phone' => [
                'type' => 'VARCHAR',
                'constraint' => 20,
                'null' => TRUE
            ],
            'coverage_level' => [
                'type' => 'ENUM',
                'constraint' => ['basic', 'plus', 'premium'],
                'default' => 'basic'
            ],
            'issue_date' => [
                'type' => 'DATE',
            ],
            'expiry_date' => [
                'type' => 'DATE',
            ],
            'status' => [
                'type' => 'ENUM',
                'constraint' => ['active', 'expired', 'suspended', 'cancelled'],
                'default' => 'active'
            ],
            'renewal_count' => [
                'type' => 'INT',
                'constraint' => 11,
                'default' => 0
            ],
            'last_renewed' => [
                'type' => 'DATE',
                'null' => TRUE
            ],
            'premium_paid' => [
                'type' => 'DECIMAL',
                'constraint' => '10,2',
                'default' => 0.00
            ],
            'payment_method' => [
                'type' => 'VARCHAR',
                'constraint' => 50,
                'null' => TRUE
            ],
            'issuing_branch' => [
                'type' => 'VARCHAR',
                'constraint' => 100,
                'null' => TRUE
            ],
            'notes' => [
                'type' => 'TEXT',
                'null' => TRUE
            ],
            'created_at' => [
                'type' => 'TIMESTAMP',
                'default' => 'CURRENT_TIMESTAMP'
            ],
            'updated_at' => [
                'type' => 'TIMESTAMP',
                'default' => 'CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP'
            ],
            'created_by' => [
                'type' => 'INT',
                'constraint' => 11,
                'null' => TRUE
            ],
            'updated_by' => [
                'type' => 'INT',
                'constraint' => 11,
                'null' => TRUE
            ]
        ]);
        
        $this->dbforge->add_key('id', TRUE);
        $this->dbforge->add_key(['card_number'], FALSE, TRUE); // Unique key
        $this->dbforge->add_key('holder_id_number');
        $this->dbforge->add_key('status');
        $this->dbforge->add_key('expiry_date');
        $this->dbforge->create_table('community_card_registry');

        // Insert sample community cards for testing
        $sample_cards = [
            [
                'card_number' => 'CHC-2024-0001-0001',
                'holder_name' => 'John Doe',
                'holder_id_number' => '12345678',
                'holder_phone' => '+254700000001',
                'coverage_level' => 'basic',
                'issue_date' => date('Y-m-d', strtotime('-6 months')),
                'expiry_date' => date('Y-m-d', strtotime('+6 months')),
                'status' => 'active',
                'premium_paid' => 2500.00,
                'payment_method' => 'MPESA',
                'issuing_branch' => 'City Health Nairobi'
            ],
            [
                'card_number' => 'CHC-2024-0001-0002',
                'holder_name' => 'Jane Smith',
                'holder_id_number' => '87654321',
                'holder_phone' => '+254700000002',
                'coverage_level' => 'plus',
                'issue_date' => date('Y-m-d', strtotime('-3 months')),
                'expiry_date' => date('Y-m-d', strtotime('+9 months')),
                'status' => 'active',
                'premium_paid' => 4500.00,
                'payment_method' => 'Bank Transfer',
                'issuing_branch' => 'City Health Mombasa'
            ],
            [
                'card_number' => 'CHC-2023-0001-0003',
                'holder_name' => 'Robert Johnson',
                'holder_id_number' => '11223344',
                'holder_phone' => '+254700000003',
                'coverage_level' => 'basic',
                'issue_date' => date('Y-m-d', strtotime('-18 months')),
                'expiry_date' => date('Y-m-d', strtotime('-6 months')),
                'status' => 'expired',
                'premium_paid' => 2500.00,
                'payment_method' => 'Cash',
                'issuing_branch' => 'City Health Kisumu'
            ]
        ];

        $this->db->insert_batch('community_card_registry', $sample_cards);

        // Create Community Card Usage Log table
        $this->dbforge->add_field([
            'id' => [
                'type' => 'INT',
                'constraint' => 11,
                'unsigned' => TRUE,
                'auto_increment' => TRUE
            ],
            'card_number' => [
                'type' => 'VARCHAR',
                'constraint' => 20,
            ],
            'patient_id' => [
                'type' => 'INT',
                'constraint' => 11,
                'unsigned' => TRUE
            ],
            'service_type' => [
                'type' => 'VARCHAR',
                'constraint' => 100,
            ],
            'original_amount' => [
                'type' => 'DECIMAL',
                'constraint' => '10,2',
            ],
            'coverage_percentage' => [
                'type' => 'DECIMAL',
                'constraint' => '5,2',
            ],
            'covered_amount' => [
                'type' => 'DECIMAL',
                'constraint' => '10,2',
            ],
            'copay_amount' => [
                'type' => 'DECIMAL',
                'constraint' => '10,2',
            ],
            'service_date' => [
                'type' => 'DATETIME',
                'default' => 'CURRENT_TIMESTAMP'
            ],
            'notes' => [
                'type' => 'TEXT',
                'null' => TRUE
            ],
            'created_at' => [
                'type' => 'TIMESTAMP',
                'default' => 'CURRENT_TIMESTAMP'
            ]
        ]);
        
        $this->dbforge->add_key('id', TRUE);
        $this->dbforge->add_key('card_number');
        $this->dbforge->add_key('patient_id');
        $this->dbforge->add_key('service_date');
        $this->dbforge->create_table('community_card_usage_log');

        // Add foreign key constraints (if supported)
        if ($this->db->dbdriver !== 'sqlite3') {
            $this->db->query('ALTER TABLE community_card_usage_log ADD CONSTRAINT fk_community_card_usage_patient 
                              FOREIGN KEY (patient_id) REFERENCES patients(id) 
                              ON DELETE CASCADE ON UPDATE CASCADE');
        }
    }

    public function down() {
        // Drop foreign key constraints first (if they exist)
        if ($this->db->dbdriver !== 'sqlite3') {
            $this->db->query('ALTER TABLE community_card_usage_log DROP FOREIGN KEY IF EXISTS fk_community_card_usage_patient');
        }

        // Drop the new tables
        $this->dbforge->drop_table('community_card_usage_log', TRUE);
        $this->dbforge->drop_table('community_card_registry', TRUE);

        // Drop indexes from patients table
        $this->db->query('DROP INDEX IF EXISTS idx_patients_coverage_type ON patients');
        $this->db->query('DROP INDEX IF EXISTS idx_patients_community_card_number ON patients');
        $this->db->query('DROP INDEX IF EXISTS idx_patients_community_card_expiry ON patients');

        // Remove columns from patients table
        $columns_to_drop = [
            'coverage_type',
            'community_card_number',
            'community_card_issue_date',
            'community_card_expiry',
            'community_card_status',
            'community_card_coverage'
        ];

        foreach ($columns_to_drop as $column) {
            $this->dbforge->drop_column('patients', $column);
        }
    }
}

?>