<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Queue Management System Database Schema
 * Migration file to create all necessary tables for the queue management system
 */
class Migration_Create_queue_system_tables extends CI_Migration {

    public function up() {
        
        // =============================================
        // 1. PATIENT_QUEUE TABLE (Main Queue Records)
        // =============================================
        if (!$this->db->table_exists('patient_queue')) {
            $this->dbforge->add_field([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'auto_increment' => TRUE
                ],
                'patient_id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'null' => FALSE
                ],
                'queue_number' => [
                    'type' => 'VARCHAR',
                    'constraint' => 20,
                    'null' => FALSE
                ],
                'department' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE
                ],
                'status' => [
                    'type' => 'ENUM',
                    'constraint' => ['waiting', 'in_progress', 'completed', 'skipped', 'transferred'],
                    'default' => 'waiting'
                ],
                'priority' => [
                    'type' => 'ENUM',
                    'constraint' => ['low', 'normal', 'high', 'urgent'],
                    'default' => 'normal'
                ],
                'queue_type' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'default' => 'walk_in'
                ],
                'workflow_type' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'default' => 'general'
                ],
                'estimated_wait_time' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => TRUE,
                    'comment' => 'Estimated wait time in minutes'
                ],
                'actual_wait_time' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => TRUE,
                    'comment' => 'Actual wait time in minutes'
                ],
                'called_at' => [
                    'type' => 'DATETIME',
                    'null' => TRUE
                ],
                'completed_at' => [
                    'type' => 'DATETIME',
                    'null' => TRUE
                ],
                'served_by' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'null' => TRUE,
                    'comment' => 'Staff ID who served the patient'
                ],
                'notes' => [
                    'type' => 'TEXT',
                    'null' => TRUE
                ],
                'next_queue' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => TRUE,
                    'comment' => 'Next department if transferred'
                ],
                'created_at' => [
                    'type' => 'DATETIME',
                    'null' => FALSE,
                    'default' => 'CURRENT_TIMESTAMP'
                ],
                'updated_at' => [
                    'type' => 'DATETIME',
                    'null' => TRUE,
                    'on_update' => 'CURRENT_TIMESTAMP'
                ]
            ]);
            
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key(['patient_id', 'department']);
            $this->dbforge->add_key(['department', 'status']);
            $this->dbforge->add_key(['queue_number', 'created_at']);
            $this->dbforge->create_table('patient_queue');
            
            // Add foreign key constraints
            $this->db->query('ALTER TABLE patient_queue ADD CONSTRAINT fk_patient_queue_patient 
                             FOREIGN KEY (patient_id) REFERENCES patients(id) ON DELETE CASCADE');
            $this->db->query('ALTER TABLE patient_queue ADD CONSTRAINT fk_patient_queue_staff 
                             FOREIGN KEY (served_by) REFERENCES staff(id) ON DELETE SET NULL');
        }

        // =============================================
        // 2. QUEUE_CONFIG TABLE (Department Settings)
        // =============================================
        if (!$this->db->table_exists('queue_config')) {
            $this->dbforge->add_field([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'auto_increment' => TRUE
                ],
                'department' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE,
                    'unique' => TRUE
                ],
                'queue_prefix' => [
                    'type' => 'VARCHAR',
                    'constraint' => 10,
                    'null' => FALSE,
                    'default' => 'Q'
                ],
                'counter_current' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'default' => 1
                ],
                'counter_reset' => [
                    'type' => 'ENUM',
                    'constraint' => ['daily', 'weekly', 'monthly', 'never'],
                    'default' => 'daily'
                ],
                'average_service_time' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'default' => 15,
                    'comment' => 'Average service time in minutes'
                ],
                'max_queue_size' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'default' => 100
                ],
                'auto_call_enabled' => [
                    'type' => 'BOOLEAN',
                    'default' => FALSE
                ],
                'sound_enabled' => [
                    'type' => 'BOOLEAN',
                    'default' => TRUE
                ],
                'display_enabled' => [
                    'type' => 'BOOLEAN',
                    'default' => TRUE
                ],
                'is_active' => [
                    'type' => 'BOOLEAN',
                    'default' => TRUE
                ],
                'created_at' => [
                    'type' => 'DATETIME',
                    'null' => FALSE,
                    'default' => 'CURRENT_TIMESTAMP'
                ],
                'updated_at' => [
                    'type' => 'DATETIME',
                    'null' => TRUE,
                    'on_update' => 'CURRENT_TIMESTAMP'
                ]
            ]);
            
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key('department', TRUE);
            $this->dbforge->create_table('queue_config');
        }

        // =============================================
        // 3. QUEUE_DISPLAY_SETTINGS TABLE
        // =============================================
        if (!$this->db->table_exists('queue_display_settings')) {
            $this->dbforge->add_field([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'auto_increment' => TRUE
                ],
                'department' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE
                ],
                'display_name' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => FALSE
                ],
                'theme_color' => [
                    'type' => 'VARCHAR',
                    'constraint' => 20,
                    'default' => '#007bff'
                ],
                'icon_class' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'default' => 'fa-users'
                ],
                'display_order' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'default' => 0
                ],
                'is_visible' => [
                    'type' => 'BOOLEAN',
                    'default' => TRUE
                ]
            ]);
            
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key('display_order');
            $this->dbforge->create_table('queue_display_settings');
        }

        // =============================================
        // 4. QUEUE_FLOW_RULES TABLE (Auto-routing)
        // =============================================
        if (!$this->db->table_exists('queue_flow_rules')) {
            $this->dbforge->add_field([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'auto_increment' => TRUE
                ],
                'rule_name' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => FALSE
                ],
                'from_department' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE
                ],
                'to_department' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE
                ],
                'condition_type' => [
                    'type' => 'ENUM',
                    'constraint' => ['automatic', 'manual', 'conditional'],
                    'default' => 'manual'
                ],
                'condition_data' => [
                    'type' => 'JSON',
                    'null' => TRUE,
                    'comment' => 'JSON conditions for routing'
                ],
                'workflow_type' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'default' => 'general'
                ],
                'priority_mapping' => [
                    'type' => 'VARCHAR',
                    'constraint' => 20,
                    'default' => 'maintain',
                    'comment' => 'maintain, increase, decrease, or specific priority'
                ],
                'is_active' => [
                    'type' => 'BOOLEAN',
                    'default' => TRUE
                ],
                'created_at' => [
                    'type' => 'DATETIME',
                    'null' => FALSE,
                    'default' => 'CURRENT_TIMESTAMP'
                ]
            ]);
            
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key(['from_department', 'condition_type']);
            $this->dbforge->create_table('queue_flow_rules');
        }

        // =============================================
        // 5. QUEUE_NOTIFICATIONS TABLE
        // =============================================
        if (!$this->db->table_exists('queue_notifications')) {
            $this->dbforge->add_field([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'auto_increment' => TRUE
                ],
                'department' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE
                ],
                'notification_type' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => FALSE
                ],
                'patient_id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'null' => TRUE
                ],
                'queue_id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'null' => TRUE
                ],
                'message' => [
                    'type' => 'TEXT',
                    'null' => FALSE
                ],
                'metadata' => [
                    'type' => 'JSON',
                    'null' => TRUE
                ],
                'is_read' => [
                    'type' => 'BOOLEAN',
                    'default' => FALSE
                ],
                'expires_at' => [
                    'type' => 'DATETIME',
                    'null' => TRUE
                ],
                'created_at' => [
                    'type' => 'DATETIME',
                    'null' => FALSE,
                    'default' => 'CURRENT_TIMESTAMP'
                ]
            ]);
            
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key(['department', 'is_read']);
            $this->dbforge->add_key('expires_at');
            $this->dbforge->create_table('queue_notifications');
        }

        // =============================================
        // INSERT DEFAULT CONFIGURATION DATA
        // =============================================
        $this->insertDefaultQueueConfig();
        $this->insertDefaultDisplaySettings();
        $this->insertDefaultFlowRules();
    }

    public function down() {
        // Drop tables in reverse order to handle foreign key constraints
        $this->dbforge->drop_table('queue_notifications', TRUE);
        $this->dbforge->drop_table('queue_flow_rules', TRUE);
        $this->dbforge->drop_table('queue_display_settings', TRUE);
        $this->dbforge->drop_table('queue_config', TRUE);
        $this->dbforge->drop_table('patient_queue', TRUE);
    }

    /**
     * Insert default queue configuration for all departments
     */
    private function insertDefaultQueueConfig() {
        $departments = [
            ['department' => 'registration', 'queue_prefix' => 'REG', 'avg_service_time' => 10],
            ['department' => 'triage', 'queue_prefix' => 'TRI', 'avg_service_time' => 15],
            ['department' => 'consultation', 'queue_prefix' => 'CON', 'avg_service_time' => 20],
            ['department' => 'pharmacy', 'queue_prefix' => 'PHM', 'avg_service_time' => 8],
            ['department' => 'laboratory', 'queue_prefix' => 'LAB', 'avg_service_time' => 12],
            ['department' => 'radiology', 'queue_prefix' => 'RAD', 'avg_service_time' => 25],
            ['department' => 'phlebotomy', 'queue_prefix' => 'PHL', 'avg_service_time' => 5],
            ['department' => 'nursing', 'queue_prefix' => 'NUR', 'avg_service_time' => 18],
            ['department' => 'operation_theatre', 'queue_prefix' => 'OPT', 'avg_service_time' => 90],
            ['department' => 'antenatal', 'queue_prefix' => 'ANC', 'avg_service_time' => 25],
            ['department' => 'pathology', 'queue_prefix' => 'PAT', 'avg_service_time' => 15],
            ['department' => 'cardiology', 'queue_prefix' => 'CAR', 'avg_service_time' => 30],
            ['department' => 'neurology', 'queue_prefix' => 'NEU', 'avg_service_time' => 25],
            ['department' => 'orthopedics', 'queue_prefix' => 'ORT', 'avg_service_time' => 20],
            ['department' => 'emergency', 'queue_prefix' => 'EMR', 'avg_service_time' => 10]
        ];

        foreach ($departments as $dept) {
            $config = [
                'department' => $dept['department'],
                'queue_prefix' => $dept['queue_prefix'],
                'average_service_time' => $dept['avg_service_time'],
                'counter_current' => 1,
                'counter_reset' => 'daily',
                'max_queue_size' => 100,
                'sound_enabled' => 1,
                'display_enabled' => 1,
                'is_active' => 1
            ];
            
            $this->db->insert('queue_config', $config);
        }
    }

    /**
     * Insert default display settings for departments
     */
    private function insertDefaultDisplaySettings() {
        $display_settings = [
            ['department' => 'registration', 'display_name' => 'Registration', 'theme_color' => '#28a745', 'icon_class' => 'fa-user-plus', 'display_order' => 1],
            ['department' => 'triage', 'display_name' => 'Triage', 'theme_color' => '#17a2b8', 'icon_class' => 'fa-stethoscope', 'display_order' => 2],
            ['department' => 'consultation', 'display_name' => 'Consultation', 'theme_color' => '#007bff', 'icon_class' => 'fa-user-md', 'display_order' => 3],
            ['department' => 'pharmacy', 'display_name' => 'Pharmacy', 'theme_color' => '#fd7e14', 'icon_class' => 'fa-pills', 'display_order' => 4],
            ['department' => 'laboratory', 'display_name' => 'Laboratory', 'theme_color' => '#6f42c1', 'icon_class' => 'fa-vial', 'display_order' => 5],
            ['department' => 'radiology', 'display_name' => 'Radiology', 'theme_color' => '#20c997', 'icon_class' => 'fa-x-ray', 'display_order' => 6],
            ['department' => 'phlebotomy', 'display_name' => 'Phlebotomy', 'theme_color' => '#e83e8c', 'icon_class' => 'fa-syringe', 'display_order' => 7],
            ['department' => 'nursing', 'display_name' => 'Nursing', 'theme_color' => '#17a2b8', 'icon_class' => 'fa-user-nurse', 'display_order' => 8],
            ['department' => 'operation_theatre', 'display_name' => 'Operation Theatre', 'theme_color' => '#ffc107', 'icon_class' => 'fa-procedures', 'display_order' => 9],
            ['department' => 'antenatal', 'display_name' => 'Antenatal Care', 'theme_color' => '#28a745', 'icon_class' => 'fa-baby', 'display_order' => 10],
            ['department' => 'pathology', 'display_name' => 'Pathology', 'theme_color' => '#6f42c1', 'icon_class' => 'fa-microscope', 'display_order' => 11],
            ['department' => 'emergency', 'display_name' => 'Emergency', 'theme_color' => '#dc3545', 'icon_class' => 'fa-ambulance', 'display_order' => 12]
        ];

        foreach ($display_settings as $setting) {
            $this->db->insert('queue_display_settings', $setting);
        }
    }

    /**
     * Insert default flow rules for common patient workflows
     */
    private function insertDefaultFlowRules() {
        $flow_rules = [
            [
                'rule_name' => 'Registration to Triage',
                'from_department' => 'registration',
                'to_department' => 'triage',
                'condition_type' => 'automatic',
                'workflow_type' => 'general',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Triage to Consultation',
                'from_department' => 'triage',
                'to_department' => 'consultation',
                'condition_type' => 'manual',
                'workflow_type' => 'general',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Consultation to Laboratory',
                'from_department' => 'consultation',
                'to_department' => 'laboratory',
                'condition_type' => 'manual',
                'workflow_type' => 'lab_tests',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Consultation to Pharmacy',
                'from_department' => 'consultation',
                'to_department' => 'pharmacy',
                'condition_type' => 'manual',
                'workflow_type' => 'prescription',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Laboratory to Phlebotomy',
                'from_department' => 'laboratory',
                'to_department' => 'phlebotomy',
                'condition_type' => 'manual',
                'workflow_type' => 'sample_collection',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Phlebotomy to Laboratory',
                'from_department' => 'phlebotomy',
                'to_department' => 'laboratory',
                'condition_type' => 'automatic',
                'workflow_type' => 'lab_processing',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Consultation to Nursing',
                'from_department' => 'consultation',
                'to_department' => 'nursing',
                'condition_type' => 'manual',
                'workflow_type' => 'nursing_care',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Consultation to Operation Theatre',
                'from_department' => 'consultation',
                'to_department' => 'operation_theatre',
                'condition_type' => 'manual',
                'workflow_type' => 'surgery',
                'priority_mapping' => 'increase'
            ],
            [
                'rule_name' => 'Operation Theatre to Nursing',
                'from_department' => 'operation_theatre',
                'to_department' => 'nursing',
                'condition_type' => 'automatic',
                'workflow_type' => 'post_operative',
                'priority_mapping' => 'high'
            ],
            [
                'rule_name' => 'Consultation to Antenatal',
                'from_department' => 'consultation',
                'to_department' => 'antenatal',
                'condition_type' => 'manual',
                'workflow_type' => 'antenatal_care',
                'priority_mapping' => 'maintain'
            ],
            [
                'rule_name' => 'Laboratory to Pathology',
                'from_department' => 'laboratory',
                'to_department' => 'pathology',
                'condition_type' => 'manual',
                'workflow_type' => 'pathology_review',
                'priority_mapping' => 'maintain'
            ]
        ];

        foreach ($flow_rules as $rule) {
            $this->db->insert('queue_flow_rules', $rule);
        }
    }
}