<?php defined('BASEPATH') OR exit('No direct script access allowed');

class Mpesa_lib {
    
    private $CI;
    private $config;
    
    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->config->load('mpesa');
        $this->config = $this->CI->config->item('mpesa');
    }
    
    /**
     * Generate Access Token
     */
    public function generateAccessToken() {
        $credentials = base64_encode($this->config['consumer_key'] . ':' . $this->config['consumer_secret']);
        $url = $this->config['urls'][$this->config['environment']]['token_url'];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Authorization: Basic ' . $credentials));
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpcode == 200) {
            $result = json_decode($response);
            return $result->access_token;
        }
        
        return false;
    }
    
    /**
     * Generate STK Push Password
     */
    public function generatePassword($timestamp = null) {
        if (!$timestamp) {
            $timestamp = date('YmdHis');
        }
        
        $password = base64_encode($this->config['business_shortcode'] . $this->config['pass_key'] . $timestamp);
        return array(
            'password' => $password,
            'timestamp' => $timestamp
        );
    }
    
    /**
     * Initiate STK Push
     */
    public function stkPush($phone_number, $amount, $account_reference = null, $transaction_desc = null) {
        $access_token = $this->generateAccessToken();
        
        if (!$access_token) {
            return array('status' => 'error', 'message' => 'Failed to generate access token');
        }
        
        // Format phone number
        $phone_number = $this->formatPhoneNumber($phone_number);
        
        // Generate password
        $password_data = $this->generatePassword();
        
        // Default values
        $account_reference = $account_reference ?: $this->config['account_reference'];
        $transaction_desc = $transaction_desc ?: $this->config['transaction_desc'];
        
        $curl_post_data = array(
            'BusinessShortCode' => $this->config['business_shortcode'],
            'Password' => $password_data['password'],
            'Timestamp' => $password_data['timestamp'],
            'TransactionType' => $this->config['transaction_type'],
            'Amount' => $amount,
            'PartyA' => $phone_number,
            'PartyB' => $this->config['business_shortcode'],
            'PhoneNumber' => $phone_number,
            'CallBackURL' => $this->config['callback_url'],
            'AccountReference' => $account_reference,
            'TransactionDesc' => $transaction_desc
        );
        
        $data_string = json_encode($curl_post_data);
        $url = $this->config['urls'][$this->config['environment']]['stk_push_url'];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/json',
            'Authorization: Bearer ' . $access_token
        ));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpcode == 200) {
            return json_decode($response, true);
        }
        
        return array('status' => 'error', 'message' => 'STK Push failed', 'response' => $response);
    }
    
    /**
     * Query STK Push Status
     */
    public function stkQuery($checkout_request_id) {
        $access_token = $this->generateAccessToken();
        
        if (!$access_token) {
            return array('status' => 'error', 'message' => 'Failed to generate access token');
        }
        
        $password_data = $this->generatePassword();
        
        $curl_post_data = array(
            'BusinessShortCode' => $this->config['business_shortcode'],
            'Password' => $password_data['password'],
            'Timestamp' => $password_data['timestamp'],
            'CheckoutRequestID' => $checkout_request_id
        );
        
        $data_string = json_encode($curl_post_data);
        $url = $this->config['urls'][$this->config['environment']]['stk_query_url'];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/json',
            'Authorization: Bearer ' . $access_token
        ));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpcode == 200) {
            return json_decode($response, true);
        }
        
        return array('status' => 'error', 'message' => 'STK Query failed', 'response' => $response);
    }
    
    /**
     * Format phone number to 254XXXXXXXXX
     */
    private function formatPhoneNumber($phone) {
        // Remove any non-digit characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Handle different formats
        if (strlen($phone) == 10 && substr($phone, 0, 1) == '0') {
            // Convert 0712345678 to 254712345678
            $phone = '254' . substr($phone, 1);
        } elseif (strlen($phone) == 9) {
            // Convert 712345678 to 254712345678
            $phone = '254' . $phone;
        } elseif (strlen($phone) == 12 && substr($phone, 0, 3) == '254') {
            // Already in correct format
            return $phone;
        } elseif (strlen($phone) == 13 && substr($phone, 0, 4) == '+254') {
            // Convert +254712345678 to 254712345678
            $phone = substr($phone, 1);
        }
        
        return $phone;
    }
}