<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Procurement Helper
 * File: application/helpers/procurement_helper.php
 * 
 * Contains utility functions for the procurement module
 */

// =============================================
// STATUS AND FORMATTING FUNCTIONS
// =============================================

/**
 * Get status badge HTML for requisitions
 */
if (!function_exists('get_requisition_status_badge')) {
    function get_requisition_status_badge($status) {
        $badges = [
            'draft' => '<span class="label label-default">Draft</span>',
            'pending_approval' => '<span class="label label-warning">Pending Approval</span>',
            'approved' => '<span class="label label-success">Approved</span>',
            'rejected' => '<span class="label label-danger">Rejected</span>',
            'partially_fulfilled' => '<span class="label label-info">Partially Fulfilled</span>',
            'fulfilled' => '<span class="label label-success">Fulfilled</span>',
            'cancelled' => '<span class="label label-default">Cancelled</span>'
        ];
        
        return isset($badges[$status]) ? $badges[$status] : '<span class="label label-default">' . ucfirst($status) . '</span>';
    }
}

/**
 * Get status badge HTML for purchase orders
 */
if (!function_exists('get_po_status_badge')) {
    function get_po_status_badge($status) {
        $badges = [
            'draft' => '<span class="label label-default">Draft</span>',
            'sent' => '<span class="label label-info">Sent</span>',
            'confirmed' => '<span class="label label-warning">Confirmed</span>',
            'partially_received' => '<span class="label label-primary">Partially Received</span>',
            'fully_received' => '<span class="label label-success">Fully Received</span>',
            'cancelled' => '<span class="label label-danger">Cancelled</span>',
            'dispute' => '<span class="label label-danger">Dispute</span>'
        ];
        
        return isset($badges[$status]) ? $badges[$status] : '<span class="label label-default">' . ucfirst($status) . '</span>';
    }
}

/**
 * Get priority badge HTML
 */
if (!function_exists('get_priority_badge')) {
    function get_priority_badge($priority) {
        $badges = [
            'low' => '<span class="label label-default">Low</span>',
            'medium' => '<span class="label label-info">Medium</span>',
            'high' => '<span class="label label-warning">High</span>',
            'urgent' => '<span class="label label-danger">Urgent</span>'
        ];
        
        return isset($badges[$priority]) ? $badges[$priority] : '<span class="label label-default">' . ucfirst($priority) . '</span>';
    }
}

// =============================================
// CURRENCY AND NUMBER FORMATTING
// =============================================

/**
 * Format currency amount
 */
if (!function_exists('format_currency')) {
    function format_currency($amount, $currency = 'KES') {
        return $currency . ' ' . number_format((float)$amount, 2);
    }
}

/**
 * Format percentage
 */
if (!function_exists('format_percentage')) {
    function format_percentage($value, $decimals = 1) {
        return number_format((float)$value, $decimals) . '%';
    }
}

// =============================================
// DATE FORMATTING FUNCTIONS
// =============================================

/**
 * Format date for display
 */
if (!function_exists('format_procurement_date')) {
    function format_procurement_date($date, $format = 'M d, Y') {
        if (empty($date) || $date == '0000-00-00' || $date == '0000-00-00 00:00:00') {
            return '-';
        }
        return date($format, strtotime($date));
    }
}

/**
 * Get days difference from today
 */
if (!function_exists('get_days_difference')) {
    function get_days_difference($date) {
        if (empty($date) || $date == '0000-00-00') {
            return null;
        }
        
        $today = new DateTime();
        $target = new DateTime($date);
        $diff = $today->diff($target);
        
        return $diff->invert ? -$diff->days : $diff->days;
    }
}

/**
 * Check if date is overdue
 */
if (!function_exists('is_overdue')) {
    function is_overdue($date) {
        return get_days_difference($date) < 0;
    }
}

// =============================================
// VALIDATION FUNCTIONS
// =============================================

/**
 * Validate requisition data
 */
if (!function_exists('validate_requisition_data')) {
    function validate_requisition_data($data) {
        $errors = [];
        
        if (empty($data['department_id'])) {
            $errors[] = 'Department is required';
        }
        
        if (empty($data['required_date'])) {
            $errors[] = 'Required date is required';
        } elseif (strtotime($data['required_date']) < strtotime(date('Y-m-d'))) {
            $errors[] = 'Required date cannot be in the past';
        }
        
        if (empty($data['purpose'])) {
            $errors[] = 'Purpose is required';
        }
        
        if (empty($data['items']) || !is_array($data['items'])) {
            $errors[] = 'At least one item is required';
        }
        
        return $errors;
    }
}

/**
 * Validate supplier data
 */
if (!function_exists('validate_supplier_data')) {
    function validate_supplier_data($data) {
        $errors = [];
        
        if (empty($data['supplier_name'])) {
            $errors[] = 'Supplier name is required';
        }
        
        if (empty($data['supplier_code'])) {
            $errors[] = 'Supplier code is required';
        }
        
        if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Invalid email format';
        }
        
        return $errors;
    }
}

// =============================================
// CALCULATION FUNCTIONS
// =============================================

/**
 * Calculate requisition total
 */
if (!function_exists('calculate_requisition_total')) {
    function calculate_requisition_total($items) {
        $total = 0;
        
        if (is_array($items)) {
            foreach ($items as $item) {
                $quantity = isset($item['quantity']) ? (int)$item['quantity'] : 0;
                $cost = isset($item['unit_cost']) ? (float)$item['unit_cost'] : 0;
                $total += $quantity * $cost;
            }
        }
        
        return $total;
    }
}

/**
 * Calculate tax amount
 */
if (!function_exists('calculate_tax')) {
    function calculate_tax($amount, $tax_rate = 16) {
        return ($amount * $tax_rate) / 100;
    }
}

/**
 * Calculate purchase order total with tax
 */
if (!function_exists('calculate_po_total')) {
    function calculate_po_total($subtotal, $tax_rate = 16, $discount = 0) {
        $tax_amount = calculate_tax($subtotal, $tax_rate);
        return $subtotal + $tax_amount - $discount;
    }
}

// =============================================
// UTILITY FUNCTIONS
// =============================================

/**
 * Generate random string for codes
 */
if (!function_exists('generate_code')) {
    function generate_code($prefix = '', $length = 8) {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $code = $prefix;
        
        for ($i = 0; $i < $length; $i++) {
            $code .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        return $code;
    }
}

/**
 * Sanitize filename for uploads
 */
if (!function_exists('sanitize_filename')) {
    function sanitize_filename($filename) {
        // Remove special characters and spaces
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
        // Remove multiple underscores
        $filename = preg_replace('/_+/', '_', $filename);
        // Remove leading/trailing underscores
        $filename = trim($filename, '_');
        
        return $filename;
    }
}

/**
 * Get file extension
 */
if (!function_exists('get_file_extension')) {
    function get_file_extension($filename) {
        return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    }
}

// =============================================
// NOTIFICATION FUNCTIONS
// =============================================

/**
 * Create notification message
 */
if (!function_exists('create_notification_message')) {
    function create_notification_message($type, $data) {
        $messages = [
            'requisition_created' => "New requisition #{requisition_no} created by {user_name}",
            'requisition_approved' => "Requisition #{requisition_no} has been approved",
            'requisition_rejected' => "Requisition #{requisition_no} has been rejected",
            'po_created' => "Purchase Order #{lpo_number} created for supplier {supplier_name}",
            'po_sent' => "Purchase Order #{lpo_number} sent to supplier",
            'goods_received' => "Goods received for PO #{lpo_number} - GRN #{grn_number}",
            'low_stock' => "Item {item_name} is running low (Current: {current_stock}, Reorder: {reorder_level})"
        ];
        
        if (isset($messages[$type])) {
            $message = $messages[$type];
            
            // Replace placeholders with actual data
            foreach ($data as $key => $value) {
                $message = str_replace('{' . $key . '}', $value, $message);
            }
            
            return $message;
        }
        
        return "Procurement notification";
    }
}

// =============================================
// PERMISSION CHECKING FUNCTIONS
// =============================================

/**
 * Check if user can approve requisitions
 */
if (!function_exists('can_approve_requisitions')) {
    function can_approve_requisitions($user_id = null) {
        // Implementation depends on your permission system
        // This is a placeholder - replace with your actual permission check
        $CI =& get_instance();
        
        if (!$user_id) {
            $user_id = $CI->session->userdata('staff_id');
        }
        
        // Check user role or specific permissions
        // Return true if user can approve requisitions
        return true; // Placeholder
    }
}

/**
 * Check if user can create purchase orders
 */
if (!function_exists('can_create_purchase_orders')) {
    function can_create_purchase_orders($user_id = null) {
        // Similar to above - implement based on your permission system
        return true; // Placeholder
    }
}

// =============================================
// REPORTING FUNCTIONS
// =============================================

/**
 * Get procurement statistics
 */
if (!function_exists('get_procurement_stats')) {
    function get_procurement_stats($period = 'month') {
        $CI =& get_instance();
        
        // Date range based on period
        switch ($period) {
            case 'week':
                $start_date = date('Y-m-d', strtotime('-1 week'));
                break;
            case 'month':
                $start_date = date('Y-m-01');
                break;
            case 'quarter':
                $start_date = date('Y-m-d', strtotime('-3 months'));
                break;
            case 'year':
                $start_date = date('Y-01-01');
                break;
            default:
                $start_date = date('Y-m-01');
        }
        
        $end_date = date('Y-m-d');
        
        // Get statistics (this would typically come from your models)
        return [
            'total_requisitions' => 0,
            'approved_requisitions' => 0,
            'total_pos' => 0,
            'total_spend' => 0,
            'period' => $period,
            'start_date' => $start_date,
            'end_date' => $end_date
        ];
    }
}

// =============================================
// EXPORT FUNCTIONS
// =============================================

/**
 * Prepare data for CSV export
 */
if (!function_exists('prepare_csv_data')) {
    function prepare_csv_data($data, $headers) {
        $csv_data = [];
        $csv_data[] = $headers;
        
        foreach ($data as $row) {
            $csv_row = [];
            foreach ($headers as $header) {
                $key = strtolower(str_replace(' ', '_', $header));
                $csv_row[] = isset($row[$key]) ? $row[$key] : '';
            }
            $csv_data[] = $csv_row;
        }
        
        return $csv_data;
    }
}

/**
 * Generate CSV content
 */
if (!function_exists('generate_csv_content')) {
    function generate_csv_content($data) {
        $output = '';
        
        foreach ($data as $row) {
            $output .= '"' . implode('","', $row) . '"' . "\n";
        }
        
        return $output;
    }
}

// =============================================
// CONFIGURATION FUNCTIONS
// =============================================

/**
 * Get procurement configuration
 */
if (!function_exists('get_procurement_config')) {
    function get_procurement_config($key = null) {
        $config = [
            'default_currency' => 'KES',
            'tax_rate' => 16,
            'approval_limit' => 50000,
            'requisition_prefix' => 'REQ',
            'lpo_prefix' => 'LPO',
            'grn_prefix' => 'GRN',
            'low_stock_threshold' => 10,
            'email_notifications' => true,
            'auto_approve_limit' => 10000
        ];
        
        if ($key) {
            return isset($config[$key]) ? $config[$key] : null;
        }
        
        return $config;
    }
}

/**
 * Get next sequence number
 */
if (!function_exists('get_next_sequence')) {
    function get_next_sequence($type, $year = null, $month = null) {
        $CI =& get_instance();
        
        if (!$year) $year = date('Y');
        if (!$month) $month = date('m');
        
        $table_map = [
            'requisition' => 'purchase_requisitions',
            'lpo' => 'purchase_orders',
            'grn' => 'goods_received_notes'
        ];
        
        if (!isset($table_map[$type])) {
            return 1;
        }
        
        $CI->db->select('COUNT(*) + 1 as next_seq');
        $CI->db->from($table_map[$type]);
        $CI->db->where('YEAR(created_at)', $year);
        $CI->db->where('MONTH(created_at)', $month);
        
        $query = $CI->db->get();
        $result = $query->row();
        
        return $result ? $result->next_seq : 1;
    }
}

// =============================================
// HELPER FOR MENU ACTIVE STATE
// =============================================

/**
 * Check if procurement menu should be active
 */
if (!function_exists('is_procurement_menu_active')) {
    function is_procurement_menu_active($submenu = '') {
        $CI =& get_instance();
        
        $current_controller = $CI->router->class;
        $current_method = $CI->router->method;
        
        if ($current_controller !== 'procurement') {
            return false;
        }
        
        if (empty($submenu)) {
            return true;
        }
        
        // Check specific submenu
        $submenu_map = [
            'dashboard' => ['index'],
            'requisitions' => ['requisitions', 'saveRequisition', 'viewRequisition'],
            'purchase_orders' => ['purchaseOrders', 'createLPO', 'viewLPO'],
            'goods_received' => ['goodsReceived', 'createGRN', 'viewGRN'],
            'suppliers' => ['suppliers', 'saveSupplier', 'viewSupplier'],
            'items' => ['items', 'saveItem', 'viewItem'],
            'reports' => ['reports', 'procurementReport']
        ];
        
        return isset($submenu_map[$submenu]) && in_array($current_method, $submenu_map[$submenu]);
    }
}

/* End of file procurement_helper.php */
/* Location: ./application/helpers/procurement_helper.php */