<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Wellness extends Admin_Controller {

    function __construct() {
        parent::__construct();
        $this->load->model('wellness_model');
        $this->load->model('staff_model');
        $this->load->library('datatables');
    }

    public function index() {
        $this->session->set_userdata('top_menu', 'Wellness');
        $this->session->set_userdata('sub_menu', 'wellness/index');

        $data['title'] = 'Wellness Module';
        $data['locations'] = $this->wellness_model->getWellnessLocations();
        $data['service_catalog'] = $this->wellness_model->getServiceCatalog();
        
       /* try {
            $data['staff'] = $this->staff_model->getStaffbyrole() ?: array();
        } catch (Exception $e) {
            $data['staff'] = array();
        }*/

        $this->load->view('layout/header', $data);
        $this->load->view('admin/wellness/index', $data);
        $this->load->view('layout/footer', $data);
    }

    public function create_visit() {
        if ($this->input->method() == 'post') {
            $this->form_validation->set_rules('patient_name', 'Patient Name', 'required');
            $this->form_validation->set_rules('location', 'Location', 'required');
            $this->form_validation->set_rules('gender', 'Gender', 'required');

            if ($this->form_validation->run()) {
                try {
                    $visit_data = array(
                        'location' => $this->input->post('location'),
                        'patient_name' => $this->input->post('patient_name'),
                        'age' => $this->input->post('age'),
                        'gender' => $this->input->post('gender'),
                        'phone_number' => $this->input->post('phone_number'),
                        'national_id' => $this->input->post('national_id'),
                        'county' => $this->input->post('county'),
                        'sub_county' => $this->input->post('sub_county'),
                        'village' => $this->input->post('village'),
                        'emergency_contact' => $this->input->post('emergency_contact'),
                        'insurance_type' => $this->input->post('insurance_type'),
                        'insurance_number' => $this->input->post('insurance_number'),
                        'chief_complaint' => $this->input->post('chief_complaint'),
                        'served_by' => $this->input->post('served_by'),
                        'notes' => $this->input->post('notes')
                    );

                    $visit_id = $this->wellness_model->createWellnessVisit($visit_data);

                    $this->session->set_flashdata('msg', '<div class="alert alert-success">Wellness visit created successfully!</div>');
                    redirect('admin/wellness/visit_details/' . $visit_id);

                } catch (Exception $e) {
                    $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error creating visit: ' . $e->getMessage() . '</div>');
                }
            }
        }

        redirect('admin/wellness');
    }

    public function visit_details($visit_id) {
        $data['title'] = 'Wellness Visit Details';
        $data['visit'] = $this->wellness_model->getWellnessVisit($visit_id);
        $data['services'] = $this->wellness_model->getWellnessServices($visit_id);
        $data['service_catalog'] = $this->wellness_model->getServiceCatalog();
        $data['staff'] = $this->staff_model->getStaffbyrole() ?: array();

        if (!$data['visit']) {
            show_404();
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/wellness/visit_details', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add_service() {
        if ($this->input->method() == 'post') {
            $visit_id = $this->input->post('visit_id');
            
            $service_data = array(
                'service_type' => $this->input->post('service_type'),
                'service_name' => $this->input->post('service_name'),
                'service_code' => $this->input->post('service_code'),
                'quantity' => $this->input->post('quantity') ?: 1,
                'unit_price' => $this->input->post('unit_price'),
                'discount_percentage' => $this->input->post('discount_percentage') ?: 0,
                'tax_percentage' => $this->input->post('tax_percentage') ?: 0,
                'provider_staff_id' => $this->input->post('provider_staff_id'),
                'notes' => $this->input->post('notes'),
                'status' => $this->input->post('status') ?: 'pending'
            );

            try {
                $this->wellness_model->addWellnessService($visit_id, $service_data);
                $this->session->set_flashdata('msg', '<div class="alert alert-success">Service added successfully!</div>');
            } catch (Exception $e) {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error adding service: ' . $e->getMessage() . '</div>');
            }
        }

        redirect('admin/wellness/visit_details/' . $visit_id);
    }

    public function process_payment() {
        if ($this->input->method() == 'post') {
            $visit_id = $this->input->post('visit_id');
            
            $payment_data = array(
                'amount' => $this->input->post('amount'),
                'payment_method' => $this->input->post('payment_method'),
                'receipt_number' => $this->input->post('receipt_number')
            );

            try {
                $receipt = $this->wellness_model->processPayment($visit_id, $payment_data);
                $this->session->set_flashdata('msg', '<div class="alert alert-success">Payment processed successfully! Receipt: ' . $receipt . '</div>');
            } catch (Exception $e) {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error processing payment: ' . $e->getMessage() . '</div>');
            }
        }

        redirect('admin/wellness/visit_details/' . $visit_id);
    }

/**
 * Reports page
 */
public function reports() {
    $this->session->set_userdata('top_menu', 'Wellness');
    $this->session->set_userdata('sub_menu', 'wellness/reports');

    $data['title'] = 'Wellness Reports';
    
    $filters = array();
    if ($this->input->get('location')) {
        $filters['location'] = $this->input->get('location');
    }
    if ($this->input->get('date_from')) {
        $filters['date_from'] = $this->input->get('date_from');
    }
    if ($this->input->get('date_to')) {
        $filters['date_to'] = $this->input->get('date_to');
    }

    $data['report_data'] = $this->wellness_model->getWellnessReport($filters);
    $data['locations'] = $this->wellness_model->getWellnessLocations();
    $data['filters'] = $filters;

    $this->load->view('layout/header', $data);
    $this->load->view('admin/wellness/reports', $data);
    $this->load->view('layout/footer', $data);
}

/**
 * Locations management page
 */
public function locations() {
    $this->session->set_userdata('top_menu', 'Wellness');
    $this->session->set_userdata('sub_menu', 'wellness/locations');

    $data['title'] = 'Wellness Locations';
    $data['locations'] = $this->wellness_model->getWellnessLocations();

    $this->load->view('layout/header', $data);
    $this->load->view('admin/wellness/locations', $data);
    $this->load->view('layout/footer', $data);
}

/**
 * Service catalog management page
 */
public function service_catalog() {
    $this->session->set_userdata('top_menu', 'Wellness');
    $this->session->set_userdata('sub_menu', 'wellness/service_catalog');

    $data['title'] = 'Service Catalog';
    $data['service_catalog'] = $this->wellness_model->getServiceCatalog();

    $this->load->view('layout/header', $data);
    $this->load->view('admin/wellness/service_catalog', $data);
    $this->load->view('layout/footer', $data);
}

/**
 * Add new location
 */
public function add_location() {
    if ($this->input->method() == 'post') {
        $this->form_validation->set_rules('location_name', 'Location Name', 'required');
        $this->form_validation->set_rules('location_code', 'Location Code', 'required|is_unique[wellness_locations.location_code]');
        $this->form_validation->set_rules('county', 'County', 'required');

        if ($this->form_validation->run()) {
            try {
                $location_data = array(
                    'location_name' => $this->input->post('location_name'),
                    'location_code' => strtoupper($this->input->post('location_code')),
                    'county' => $this->input->post('county'),
                    'sub_county' => $this->input->post('sub_county'),
                    'address' => $this->input->post('address'),
                    'gps_coordinates' => $this->input->post('gps_coordinates'),
                    'contact_person' => $this->input->post('contact_person'),
                    'contact_phone' => $this->input->post('contact_phone'),
                    'is_active' => 1
                );

                $this->db->insert('wellness_locations', $location_data);

                $this->session->set_flashdata('msg', '<div class="alert alert-success">Location added successfully!</div>');
            } catch (Exception $e) {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error adding location: ' . $e->getMessage() . '</div>');
            }
        } else {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">' . validation_errors() . '</div>');
        }
    }

    redirect('admin/wellness/locations');
}

/**
 * Update location
 */
public function update_location() {
    if ($this->input->method() == 'post') {
        $location_id = $this->input->post('location_id');
        
        $location_data = array(
            'location_name' => $this->input->post('location_name'),
            'county' => $this->input->post('county'),
            'sub_county' => $this->input->post('sub_county'),
            'address' => $this->input->post('address'),
            'gps_coordinates' => $this->input->post('gps_coordinates'),
            'contact_person' => $this->input->post('contact_person'),
            'contact_phone' => $this->input->post('contact_phone'),
            'updated_at' => date('Y-m-d H:i:s')
        );

        try {
            $this->db->where('id', $location_id);
            $this->db->update('wellness_locations', $location_data);

            $this->session->set_flashdata('msg', '<div class="alert alert-success">Location updated successfully!</div>');
        } catch (Exception $e) {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error updating location: ' . $e->getMessage() . '</div>');
        }
    }

    redirect('admin/wellness/locations');
}

/**
 * Get location details for editing
 */
public function get_location() {
    header('Content-Type: application/json');
    
    $location_id = $this->input->get('location_id');
    
    $this->db->where('id', $location_id);
    $location = $this->db->get('wellness_locations')->row();
    
    if ($location) {
        echo json_encode(array('success' => true, 'location' => $location));
    } else {
        echo json_encode(array('success' => false));
    }
}

/**
 * Toggle location status
 */
public function toggle_location_status() {
    header('Content-Type: application/json');
    
    $location_id = $this->input->post('location_id');
    $status = $this->input->post('status');
    
    $this->db->where('id', $location_id);
    $result = $this->db->update('wellness_locations', array('is_active' => $status));
    
    echo json_encode(array('success' => $result));
}

/**
 * Delete location
 */
public function delete_location() {
    header('Content-Type: application/json');
    
    $location_id = $this->input->post('location_id');
    
    // Check if location has associated visits
    $this->db->where('location_id', $location_id);
    $visit_count = $this->db->count_all_results('wellness_visits');
    
    if ($visit_count > 0) {
        echo json_encode(array('success' => false, 'message' => 'Cannot delete location with associated visits'));
        return;
    }
    
    $this->db->where('id', $location_id);
    $result = $this->db->delete('wellness_locations');
    
    echo json_encode(array('success' => $result));
}

/**
 * Generate location code
 */
public function generate_location_code() {
    header('Content-Type: application/json');
    
    $county = $this->input->get('county');
    
    // Get county prefix (first 3 letters)
    $prefix = strtoupper(substr(str_replace(' ', '', $county), 0, 3));
    
    // Get next number for this county
    $this->db->select('location_code');
    $this->db->like('location_code', $prefix . '-', 'after');
    $this->db->order_by('location_code', 'DESC');
    $this->db->limit(1);
    $query = $this->db->get('wellness_locations');
    
    $next_number = 1;
    if ($query->num_rows() > 0) {
        $last_code = $query->row()->location_code;
        $number_part = substr($last_code, -3);
        $next_number = intval($number_part) + 1;
    }
    
    $code = $prefix . '-' . str_pad($next_number, 3, '0', STR_PAD_LEFT);
    
    echo json_encode(array('success' => true, 'code' => $code));
}

/**
 * Get monthly visit statistics
 */
public function get_monthly_visit_stats() {
    header('Content-Type: application/json');
    
    $current_month = date('Y-m');
    
    $this->db->where('DATE_FORMAT(visit_date, "%Y-%m")', $current_month);
    $monthly_visits = $this->db->count_all_results('wellness_visits');
    
    echo json_encode(array('monthly_visits' => $monthly_visits));
}

/**
 * Add service to catalog
 */
public function add_service_catalog() {
    if ($this->input->method() == 'post') {
        $this->form_validation->set_rules('service_name', 'Service Name', 'required');
        $this->form_validation->set_rules('service_code', 'Service Code', 'required|is_unique[wellness_service_catalog.service_code]');
        $this->form_validation->set_rules('service_type', 'Service Type', 'required');
        $this->form_validation->set_rules('standard_price', 'Standard Price', 'required|numeric');

        if ($this->form_validation->run()) {
            try {
                $service_data = array(
                    'service_name' => $this->input->post('service_name'),
                    'service_code' => strtoupper($this->input->post('service_code')),
                    'service_type' => $this->input->post('service_type'),
                    'category' => $this->input->post('category'),
                    'description' => $this->input->post('description'),
                    'standard_price' => $this->input->post('standard_price'),
                    'minimum_price' => $this->input->post('minimum_price'),
                    'duration_minutes' => $this->input->post('duration_minutes') ?: 30,
                    'requires_specialist' => $this->input->post('requires_specialist') ?: 0,
                    'is_active' => 1
                );

                $this->db->insert('wellness_service_catalog', $service_data);

                $this->session->set_flashdata('msg', '<div class="alert alert-success">Service added successfully!</div>');
            } catch (Exception $e) {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error adding service: ' . $e->getMessage() . '</div>');
            }
        } else {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">' . validation_errors() . '</div>');
        }
    }

    redirect('admin/wellness/service_catalog');
}

/**
 * Update service catalog
 */
public function update_service_catalog() {
    if ($this->input->method() == 'post') {
        $service_id = $this->input->post('service_id');
        
        $service_data = array(
            'service_name' => $this->input->post('service_name'),
            'service_type' => $this->input->post('service_type'),
            'category' => $this->input->post('category'),
            'description' => $this->input->post('description'),
            'standard_price' => $this->input->post('standard_price'),
            'minimum_price' => $this->input->post('minimum_price'),
            'duration_minutes' => $this->input->post('duration_minutes'),
            'requires_specialist' => $this->input->post('requires_specialist'),
            'updated_at' => date('Y-m-d H:i:s')
        );

        try {
            $this->db->where('id', $service_id);
            $this->db->update('wellness_service_catalog', $service_data);

            $this->session->set_flashdata('msg', '<div class="alert alert-success">Service updated successfully!</div>');
        } catch (Exception $e) {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error updating service: ' . $e->getMessage() . '</div>');
        }
    }

    redirect('admin/wellness/service_catalog');
}

/**
 * Get service details for editing
 */
public function get_service() {
    header('Content-Type: application/json');
    
    $service_id = $this->input->get('service_id');
    
    $this->db->where('id', $service_id);
    $service = $this->db->get('wellness_service_catalog')->row();
    
    if ($service) {
        echo json_encode(array('success' => true, 'service' => $service));
    } else {
        echo json_encode(array('success' => false));
    }
}

/**
 * Toggle service status
 */
public function toggle_service_status() {
    header('Content-Type: application/json');
    
    $service_id = $this->input->post('service_id');
    $status = $this->input->post('status');
    
    $this->db->where('id', $service_id);
    $result = $this->db->update('wellness_service_catalog', array('is_active' => $status));
    
    echo json_encode(array('success' => $result));
}

/**
 * Delete service from catalog
 */
public function delete_service_catalog() {
    header('Content-Type: application/json');
    
    $service_id = $this->input->post('service_id');
    
    // Check if service has been used in visits
    $this->db->where('service_id', $service_id);
    $usage_count = $this->db->count_all_results('wellness_services');
    
    if ($usage_count > 0) {
        echo json_encode(array('success' => false, 'message' => 'Cannot delete service that has been used in visits'));
        return;
    }
    
    $this->db->where('id', $service_id);
    $result = $this->db->delete('wellness_service_catalog');
    
    echo json_encode(array('success' => $result));
}

/**
 * Generate service code
 */
public function generate_service_code() {
    header('Content-Type: application/json');
    
    $service_type = $this->input->get('service_type');
    
    // Service type prefixes
    $prefixes = array(
        'consultation' => 'CONS',
        'dental' => 'DENT',
        'laboratory' => 'LAB',
        'pharmacy' => 'PHARM',
        'radiology' => 'RAD',
        'optical' => 'OPT',
        'specialized' => 'SPEC'
    );
    
    $prefix = isset($prefixes[$service_type]) ? $prefixes[$service_type] : 'GEN';
    
    // Get next number for this service type
    $pattern = 'WV-' . $prefix . '-%';
    $this->db->select('service_code');
    $this->db->like('service_code', $pattern);
    $this->db->order_by('service_code', 'DESC');
    $this->db->limit(1);
    $query = $this->db->get('wellness_service_catalog');
    
    $next_number = 1;
    if ($query->num_rows() > 0) {
        $last_code = $query->row()->service_code;
        $number_part = substr($last_code, -3);
        $next_number = intval($number_part) + 1;
    }
    
    $code = 'WV-' . $prefix . '-' . str_pad($next_number, 3, '0', STR_PAD_LEFT);
    
    echo json_encode(array('success' => true, 'code' => $code));
}

/**
 * Bulk activate services
 */
public function bulk_activate_services() {
    header('Content-Type: application/json');
    
    $result = $this->db->update('wellness_service_catalog', array('is_active' => 1));
    
    echo json_encode(array('success' => $result));
}

/**
 * Bulk deactivate services
 */
public function bulk_deactivate_services() {
    header('Content-Type: application/json');
    
    $result = $this->db->update('wellness_service_catalog', array('is_active' => 0));
    
    echo json_encode(array('success' => $result));
}

/**
 * Export service catalog
 */
public function export_service_catalog() {
    $this->load->library('excel'); // You'll need to install/configure an Excel library
    
    $services = $this->wellness_model->getServiceCatalog();
    
    // Create Excel file
    $filename = 'wellness_service_catalog_' . date('Y-m-d') . '.csv';
    
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    
    $output = fopen('php://output', 'w');
    
    // Add CSV headers
    fputcsv($output, array(
        'Service Code',
        'Service Name', 
        'Service Type',
        'Category',
        'Description',
        'Standard Price',
        'Minimum Price',
        'Duration (minutes)',
        'Requires Specialist',
        'Status'
    ));
    
    // Add data rows
    foreach ($services as $service) {
        fputcsv($output, array(
            $service->service_code,
            $service->service_name,
            $service->service_type,
            $service->category,
            $service->description,
            $service->standard_price,
            $service->minimum_price,
            $service->duration_minutes,
            $service->requires_specialist ? 'Yes' : 'No',
            $service->is_active ? 'Active' : 'Inactive'
        ));
    }
    
    fclose($output);
    exit;
}

/**
 * Import service catalog
 */
public function import_service_catalog() {
    header('Content-Type: application/json');
    
    if (!isset($_FILES['import_file'])) {
        echo json_encode(array('success' => false, 'message' => 'No file uploaded'));
        return;
    }
    
    $file = $_FILES['import_file'];
    $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    
    if (!in_array($file_ext, array('csv', 'xlsx'))) {
        echo json_encode(array('success' => false, 'message' => 'Only CSV and Excel files are allowed'));
        return;
    }
    
    try {
        $imported_count = 0;
        
        if ($file_ext == 'csv') {
            $handle = fopen($file['tmp_name'], 'r');
            
            // Skip header row
            fgetcsv($handle);
            
            while (($data = fgetcsv($handle)) !== FALSE) {
                if (count($data) >= 6) { // Minimum required columns
                    $service_data = array(
                        'service_code' => $data[0],
                        'service_name' => $data[1],
                        'service_type' => $data[2],
                        'category' => $data[3],
                        'description' => $data[4],
                        'standard_price' => floatval($data[5]),
                        'minimum_price' => isset($data[6]) ? floatval($data[6]) : null,
                        'duration_minutes' => isset($data[7]) ? intval($data[7]) : 30,
                        'requires_specialist' => isset($data[8]) ? ($data[8] == 'Yes' ? 1 : 0) : 0,
                        'is_active' => 1
                    );
                    
                    // Check if service code already exists
                    $this->db->where('service_code', $service_data['service_code']);
                    if ($this->db->count_all_results('wellness_service_catalog') == 0) {
                        $this->db->insert('wellness_service_catalog', $service_data);
                        $imported_count++;
                    }
                }
            }
            
            fclose($handle);
        }
        
        echo json_encode(array('success' => true, 'imported_count' => $imported_count));
        
    } catch (Exception $e) {
        echo json_encode(array('success' => false, 'message' => $e->getMessage()));
    }
}

/**
 * Update visit details
 */
public function update_visit() {
    if ($this->input->method() == 'post') {
        $visit_id = $this->input->post('visit_id');
        
        $visit_data = array(
            'patient_name' => $this->input->post('patient_name'),
            'age' => $this->input->post('age'),
            'gender' => $this->input->post('gender'),
            'phone_number' => $this->input->post('phone_number'),
            'national_id' => $this->input->post('national_id'),
            'chief_complaint' => $this->input->post('chief_complaint'),
            'notes' => $this->input->post('notes'),
            'updated_at' => date('Y-m-d H:i:s')
        );

        try {
            $this->db->where('id', $visit_id);
            $this->db->update('wellness_visits', $visit_data);

            $this->session->set_flashdata('msg', '<div class="alert alert-success">Visit updated successfully!</div>');
        } catch (Exception $e) {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error updating visit: ' . $e->getMessage() . '</div>');
        }
    }

    $visit_id = $this->input->post('visit_id');
    redirect('admin/wellness/visit_details/' . $visit_id);
}

/**
 * Export reports to Excel
 */
public function export_excel() {
    $filters = array();
    if ($this->input->get('location')) {
        $filters['location'] = $this->input->get('location');
    }
    if ($this->input->get('date_from')) {
        $filters['date_from'] = $this->input->get('date_from');
    }
    if ($this->input->get('date_to')) {
        $filters['date_to'] = $this->input->get('date_to');
    }
    
    $report_data = $this->wellness_model->getWellnessReport($filters);
    
    $filename = 'wellness_report_' . date('Y-m-d_H-i-s') . '.csv';
    
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    
    $output = fopen('php://output', 'w');
    
    // Add CSV headers
    fputcsv($output, array(
        'Date',
        'Location',
        'Total Visits',
        'Consultations',
        'Dental Services',
        'Laboratory Services',
        'Pharmacy Services',
        'Radiology Services',
        'Optical Services',
        'Total Revenue',
        'Total Payments',
        'Outstanding Balance',
        'Collection Rate %'
    ));
    
    // Add data rows
    foreach ($report_data as $row) {
        $collection_rate = $row->total_revenue > 0 ? round(($row->total_payments / $row->total_revenue) * 100, 1) : 0;
        
        fputcsv($output, array(
            date('Y-m-d', strtotime($row->visit_date)),
            $row->location,
            $row->total_visits,
            $row->consultations,
            $row->dental_services,
            $row->lab_services,
            $row->pharmacy_services,
            $row->radiology_services,
            $row->optical_services,
            number_format($row->total_revenue, 2),
            number_format($row->total_payments, 2),
            number_format($row->total_outstanding, 2),
            $collection_rate
        ));
    }
    
    fclose($output);
    exit;
}

/**
 * Export reports to PDF
 */
public function export_pdf() {
    // This would require a PDF library like TCPDF or DOMPDF
    // For now, redirect to a printable HTML version
    
    $filters = array();
    if ($this->input->get('location')) {
        $filters['location'] = $this->input->get('location');
    }
    if ($this->input->get('date_from')) {
        $filters['date_from'] = $this->input->get('date_from');
    }
    if ($this->input->get('date_to')) {
        $filters['date_to'] = $this->input->get('date_to');
    }
    
    $data['report_data'] = $this->wellness_model->getWellnessReport($filters);
    $data['filters'] = $filters;
    $data['title'] = 'Wellness Report - ' . date('Y-m-d');
    
    $this->load->view('admin/wellness/report_pdf', $data);
}

/**
 * Print visit summary
 */
public function print_visit_summary($visit_id) {
    $data['visit'] = $this->wellness_model->getWellnessVisit($visit_id);
    $data['services'] = $this->wellness_model->getWellnessServices($visit_id);
    $data['title'] = 'Visit Summary - ' . $data['visit']->visit_reference;
    
    if (!$data['visit']) {
        show_404();
    }
    
    $this->load->view('admin/wellness/visit_summary_print', $data);
}

/**
 * Generate receipt
 */
public function generate_receipt($visit_id) {
    $data['visit'] = $this->wellness_model->getWellnessVisit($visit_id);
    $data['services'] = $this->wellness_model->getWellnessServices($visit_id);
    $data['title'] = 'Receipt - ' . $data['visit']->receipt_number;
    
    if (!$data['visit'] || $data['visit']->amount_paid <= 0) {
        show_error('No payment found for this visit');
    }
    
    $this->load->view('admin/wellness/receipt_print', $data);
}

/**
 * Delete service from visit
 */
public function delete_service() {
    header('Content-Type: application/json');
    
    $service_id = $this->input->post('service_id');
    
    // Get service details before deletion
    $this->db->where('id', $service_id);
    $service = $this->db->get('wellness_services')->row();
    
    if ($service) {
        $this->db->where('id', $service_id);
        $result = $this->db->delete('wellness_services');
        
        if ($result) {
            // Update visit total
            $this->updateVisitTotal($service->visit_id);
            echo json_encode(array('success' => true));
        } else {
            echo json_encode(array('success' => false, 'message' => 'Database error'));
        }
    } else {
        echo json_encode(array('success' => false, 'message' => 'Service not found'));
    }
}

/**
 * Get visit timeline
 */
public function get_visit_timeline() {
    header('Content-Type: application/json');
    
    $visit_id = $this->input->get('visit_id');
    
    // Get visit creation
    $this->db->where('id', $visit_id);
    $visit = $this->db->get('wellness_visits')->row();
    
    $timeline = array();
    
    if ($visit) {
        $timeline[] = array(
            'type' => 'visit_created',
            'title' => 'Visit Created',
            'description' => 'Wellness visit was created for ' . $visit->patient_name,
            'time' => date('H:i', strtotime($visit->created_at))
        );
        
        // Get services added
        $this->db->select('ws.*, CONCAT(s.name, " ", s.surname) as added_by_name');
        $this->db->from('wellness_services ws');
        $this->db->join('staff s', 's.id = ws.provider_staff_id', 'left');
        $this->db->where('ws.visit_id', $visit_id);
        $this->db->order_by('ws.created_at', 'ASC');
        $services = $this->db->get()->result();
        
        foreach ($services as $service) {
            $timeline[] = array(
                'type' => 'service_added',
                'title' => 'Service Added',
                'description' => $service->service_name . ' added' . ($service->added_by_name ? ' by ' . $service->added_by_name : ''),
                'time' => date('H:i', strtotime($service->created_at))
            );
            
            if ($service->status == 'completed' && $service->completed_at) {
                $timeline[] = array(
                    'type' => 'service_completed',
                    'title' => 'Service Completed',
                    'description' => $service->service_name . ' marked as completed',
                    'time' => date('H:i', strtotime($service->completed_at))
                );
            }
        }
        
        // Add payment events if any
        if ($visit->amount_paid > 0) {
            $timeline[] = array(
                'type' => 'payment_processed',
                'title' => 'Payment Processed',
                'description' => 'Payment of KSh ' . number_format($visit->amount_paid, 2) . ' received',
                'time' => date('H:i', strtotime($visit->updated_at))
            );
        }
    }
    
    echo json_encode($timeline);
}

/**
 * Get visit status for auto-refresh
 */
public function get_visit_status() {
    header('Content-Type: application/json');
    
    $visit_id = $this->input->get('visit_id');
    $visit = $this->wellness_model->getWellnessVisit($visit_id);
    
    if ($visit) {
        echo json_encode(array(
            'payment_status' => $visit->payment_status,
            'total_amount' => $visit->total_amount,
            'amount_paid' => $visit->amount_paid,
            'balance' => $visit->balance
        ));
    } else {
        echo json_encode(array('success' => false));
    }
}

/**
 * Private helper method to update visit totals
 */
private function updateVisitTotal($visit_id) {
    $this->db->select('SUM(final_amount) as total');
    $this->db->where('visit_id', $visit_id);
    $query = $this->db->get('wellness_services');
    $result = $query->row();
    
    $total_amount = $result->total ?? 0;
    
    // Get current paid amount
    $this->db->select('amount_paid');
    $this->db->where('id', $visit_id);
    $visit = $this->db->get('wellness_visits')->row();
    
    $amount_paid = $visit ? $visit->amount_paid : 0;
    $balance = $total_amount - $amount_paid;
    
    // Determine payment status
    if ($balance <= 0) {
        $payment_status = 'paid';
    } elseif ($amount_paid > 0) {
        $payment_status = 'partial';
    } else {
        $payment_status = 'pending';
    }
    
    $this->db->where('id', $visit_id);
    $this->db->update('wellness_visits', array(
        'total_amount' => $total_amount,
        'balance' => max(0, $balance),
        'payment_status' => $payment_status
    ));
}

    public function get_service_price() {
        header('Content-Type: application/json');
        
        $service_code = $this->input->get('service_code');
        
        $this->db->where('service_code', $service_code);
        $this->db->where('is_active', 1);
        $query = $this->db->get('wellness_service_catalog');
        $service = $query->row();
        
        if ($service) {
            echo json_encode(array(
                'success' => true,
                'price' => $service->standard_price,
                'name' => $service->service_name,
                'type' => $service->service_type
            ));
        } else {
            echo json_encode(array('success' => false));
        }
    }
}