<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Triage extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
            
            // Add queue model
             $this->load->model('Queue_model');
        } catch (Exception $e) {
            log_message('error', 'Triage controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'triage');

        // Prepare data for view
        $data['title'] = 'Patient Triage with Biometric Verification';
        
        // Get queue data for triage department
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('triage');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('triage');
        
        // Get staff members (nurses, doctors)
        try {
            $data['staff'] = $this->getTriageStaff();
        } catch (Exception $e) {
            log_message('error', 'Error loading staff: ' . $e->getMessage());
            $data['staff'] = [];
        }
        
        // Get gender list
        try {
            if (method_exists($this->customlib, 'getGender_Patient')) {
                $data['genderList'] = $this->customlib->getGender_Patient();
            } else {
                $data['genderList'] = array(
                    'male' => 'Male',
                    'female' => 'Female',
                    'other' => 'Other'
                );
            }
        } catch (Exception $e) {
            $data['genderList'] = array('male' => 'Male', 'female' => 'Female');
        }
        
         // FIXED: Add triage priorities for dropdown - THIS WAS MISSING!
        $data['triagePriorities'] = $this->getTriagePriorities();
        
        // Get departments for routing
        $data['departments'] = $this->getAvailableDepartments();
        
        // Load views
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/triage/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'View loading error: ' . $e->getMessage());
            show_error('Failed to load triage view: ' . $e->getMessage());
        }
    }
    
/**
     * Queue management for triage
     */
    public function queue() {
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'triage/queue');
        
        $data['title'] = 'Triage Queue Management';
        $data['department'] = 'triage';
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('triage');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('triage');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/triage/queue_management', $data);
        $this->load->view('layout/footer');
    }
    
        /**
 * AJAX: Get queue notifications for department
 * This method checks for actual waiting patients and returns them as notifications
 */
public function get_queue_notifications() {
    header('Content-Type: application/json');
    
    try {
        $department = $this->input->get('department', TRUE);
        $last_check = $this->input->get('last_check', TRUE);
        
        // Default to checking all departments if none specified
        if (empty($department) || $department === 'all') {
            $department = null;
        }
        
        // If no last_check provided, use current time minus 30 seconds
        if (empty($last_check)) {
            $last_check = time() - 30;
        }
        
        // Convert timestamp to datetime for database query
        $last_check_datetime = date('Y-m-d H:i:s', $last_check);
        
        // Query for new patients added to queue since last check
        $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->where('pq.created_at >', $last_check_datetime);
        
        // Filter by department if specified
        if ($department) {
            $this->db->where('pq.department', $department);
        }
        
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'notifications' => [],
                'timestamp' => time()
            ]);
            return;
        }
        
        $new_patients = $query->result_array();
        $notifications = [];
        
        // Convert database results to notification format
        foreach ($new_patients as $patient) {
            $notifications[] = [
                'patient_id' => $patient['patient_id'],
                'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                'queue_number' => $patient['queue_number'],
                'department' => $patient['department'],
                'priority' => $patient['priority'],
                'workflow_type' => $patient['queue_type'] ?: 'general',
                'time' => date('H:i:s', strtotime($patient['created_at'])),
                'created_at' => $patient['created_at'],
                'phone' => $patient['mobileno'] ?: '',
                'age' => $patient['age'] ?: '',
                'gender' => $patient['gender'] ?: '',
                'patient_unique_id' => $patient['patient_unique_id'] ?: ''
            ];
        }
        
        // Also check for priority patients (urgent/high priority)
        if (empty($new_patients)) {
            // Check for any urgent patients that might have been missed
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.status', 'waiting');
            $this->db->where_in('pq.priority', ['urgent', 'high']);
            
            if ($department) {
                $this->db->where('pq.department', $department);
            }
            
            $this->db->order_by('pq.created_at', 'DESC');
            $this->db->limit(5);
            
            $urgent_query = $this->db->get();
            $urgent_patients = $urgent_query->result_array();
            
            foreach ($urgent_patients as $patient) {
                $notifications[] = [
                    'patient_id' => $patient['patient_id'],
                    'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                    'queue_number' => $patient['queue_number'],
                    'department' => $patient['department'],
                    'priority' => $patient['priority'],
                    'workflow_type' => $patient['queue_type'] ?: 'urgent',
                    'time' => date('H:i:s', strtotime($patient['created_at'])),
                    'created_at' => $patient['created_at'],
                    'phone' => $patient['mobileno'] ?: '',
                    'age' => $patient['age'] ?: '',
                    'gender' => $patient['gender'] ?: '',
                    'patient_unique_id' => $patient['patient_unique_id'] ?: '',
                    'is_urgent' => true
                ];
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'notifications' => $notifications,
            'count' => count($notifications),
            'timestamp' => time(),
            'last_check' => $last_check,
            'department' => $department ?: 'all',
            'debug' => [
                'query_time' => $last_check_datetime,
                'current_time' => date('Y-m-d H:i:s'),
                'department_filter' => $department
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Exception: ' . $e->getMessage(),
            'notifications' => [],
            'timestamp' => time()
        ]);
    }
}

public function debug_queue_status() {
    header('Content-Type: application/json');
    
    try {
        // Get all waiting patients
        $this->db->select('pq.*, p.patient_name, p.mobileno');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(20);
        
        $query = $this->db->get();
        $waiting_patients = $query->result_array();
        
        // Get queue counts by department
        $this->db->select('department, COUNT(*) as count');
        $this->db->from('patient_queue');
        $this->db->where('status', 'waiting');
        $this->db->group_by('department');
        $dept_counts = $this->db->get()->result_array();
        
        echo json_encode([
            'status' => 'success',
            'waiting_patients' => $waiting_patients,
            'department_counts' => $dept_counts,
            'total_waiting' => count($waiting_patients),
            'timestamp' => time(),
            'current_time' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

    // =============================================
    // QUEUE OPERATIONS (AJAX)
    // =============================================

    /**
     * AJAX: Call next patient for triage
     */
    /*public function call_next_patient() {
        header('Content-Type: application/json');
        
        try {
            $nurse_id = $this->input->post('nurse_id') ?: $this->customlib->getStaffID();
            
            $result = $this->Queue_model->callNextPatient('triage', $nurse_id);
            
            if ($result['status'] === 'success') {
                $this->send_queue_notification('triage', 'patient_called', $result['patient']);
            }
            
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }*/

    // =============================================
    // PATIENT MANAGEMENT METHODS - FIXED
    // =============================================

    /**
     * FIXED: Get all patients - compatible with any patients table structure
     */
public function getAllPatients() {
    header('Content-Type: application/json');
    
    try {
        // Get all active patients with workflow and fingerprint status
        $this->db->select('
            id, 
            patient_name, 
            mobileno, 
            gender, 
            age, 
            patient_unique_id,
            image,
            patient_type,
            COALESCE(patient_workflow, patient_type, "general") as workflow,
            CASE 
                WHEN fingerprint_template IS NOT NULL AND TRIM(fingerprint_template) != "" 
                THEN 1 
                ELSE 0 
            END as has_fingerprint,
            LENGTH(fingerprint_template) as template_length,
            CASE 
                WHEN UPPER(COALESCE(patient_workflow, patient_type, "general")) = "GCC"
                THEN 1 
                ELSE 0 
            END as requires_verification,
            created_at
        ');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('patient_name', 'ASC');
        $this->db->limit(200);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'patients' => [],
                'count' => 0
            ]);
            return;
        }
        
        $patients = $query->result_array();
        
        // Count patients by workflow type for statistics
        $workflowStats = [
            'gcc' => 0,
            'general' => 0,
            'specialized' => 0,
            'with_fingerprints' => 0,
            'requiring_verification' => 0
        ];
        
        foreach ($patients as $patient) {
            $workflow = strtolower($patient['workflow']);
            if (isset($workflowStats[$workflow])) {
                $workflowStats[$workflow]++;
            }
            
            if ($patient['has_fingerprint'] == 1) {
                $workflowStats['with_fingerprints']++;
            }
            
            if ($patient['requires_verification'] == 1) {
                $workflowStats['requiring_verification']++;
            }
        }
        
        log_message('info', 'Triage: Loaded ' . count($patients) . ' patients - ' . json_encode($workflowStats));
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Found ' . count($patients) . ' patients',
            'patients' => $patients,
            'count' => count($patients),
            'workflow_stats' => $workflowStats
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Triage: Error loading patients: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'patients' => [],
            'count' => 0
        ]);
    }
}
    
    
     public function createTriageAppointment() {
        try {
            // Validate required inputs
            $patient_id = $this->input->post('patient_id');
            $workflow_type = $this->input->post('workflow_type');
            $chief_complaint = $this->input->post('chief_complaint');
            $priority = $this->input->post('priority') ?: 'normal';
            
            if (!$patient_id) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID is required'
                ]);
                return;
            }

            // Validate patient exists
            $patient = $this->patient_model->get($patient_id);
            if (!$patient) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found'
                ]);
                return;
            }

            // Get staff ID (current user)
            $staff_id = $this->session->userdata('admin_id') ?: 1;

            // Prepare triage data
            $triageData = [
                'patient_id' => $patient_id,
                'chief_complaint' => $chief_complaint ?: 'Workflow-based triage',
                'vital_signs' => json_encode([
                    'workflow_type' => $workflow_type,
                    'priority' => $priority,
                    'created_via' => 'workflow_routing'
                ]),
                'assessment' => "Auto-created for {$workflow_type} workflow",
                'priority' => $priority,
                'status' => 'pending',
                'verification_status' => 'unverified',
                'triage_by' => $staff_id,
                'created_at' => date('Y-m-d H:i:s'),
                'notes' => "Created automatically for {$workflow_type} patient workflow"
            ];

            // Create triage record
            $triage_id = $this->triage_model->createTriage($triageData);
            
            if ($triage_id) {
                // If specialized workflow, route to consultation immediately
                if ($workflow_type === 'specialized') {
                    $routing_result = $this->routeToConsultation($patient_id, $triage_id, 'high');
                }
                
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Triage appointment created successfully',
                    'triage_id' => $triage_id,
                    'patient_id' => $patient_id,
                    'workflow_type' => $workflow_type,
                    'routing_result' => isset($routing_result) ? $routing_result : null
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to create triage appointment'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Create triage appointment error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Error creating triage appointment: ' . $e->getMessage()
            ]);
        }
    }

    private function routeToConsultation($patient_id, $triage_id, $priority = 'normal') {
        try {
            // Load OPD model if available
            if (file_exists(APPPATH . 'models/Opd_model.php')) {
                $this->load->model('opd_model');
            }

            // Get available doctor
            $doctor_id = $this->getAvailableDoctor();

            // Create consultation/OPD appointment
            if ($this->db->table_exists('opd_details')) {
                $opdData = [
                    'patient_id' => $patient_id,
                    'case_type' => 'Triage Referral',
                    'casualty' => ($priority === 'urgent' || $priority === 'high') ? 'yes' : 'no',
                    'old_patient' => 'yes',
                    'symptoms' => 'Referred from Triage - Priority: ' . $priority,
                    'appointment_date' => date('Y-m-d'),
                    'appointment_time' => date('H:i:s'),
                    'cons_doctor' => $doctor_id,
                    'generated_by' => $this->session->userdata('admin_id') ?: 1,
                    'amount' => 0, // Will be updated based on charge
                    'payment_mode' => 'Cash',
                    'refference' => 'Triage-' . $triage_id
                ];

                $this->db->insert('opd_details', $opdData);
                $opd_id = $this->db->insert_id();

                if ($opd_id) {
                    return [
                        'status' => 'success',
                        'message' => 'Patient routed to consultation',
                        'opd_id' => $opd_id,
                        'doctor_id' => $doctor_id
                    ];
                }
            }

            return [
                'status' => 'partial',
                'message' => 'Triage created but consultation routing failed'
            ];

        } catch (Exception $e) {
            log_message('error', 'Consultation routing error: ' . $e->getMessage());
            return [
                'status' => 'error',
                'message' => 'Failed to route to consultation: ' . $e->getMessage()
            ];
        }
    }
    
    public function scheduleGCCServices() {
        try {
            $patient_id = $this->input->post('patient_id');
            $services = $this->input->post('services');
            $package = $this->input->post('package');
            $preferred_date = $this->input->post('preferred_date');

            if (!$patient_id || !$services) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID and services are required'
                ]);
                return;
            }

            $appointments = [];
            $service_list = is_array($services) ? $services : [$services];

            foreach ($service_list as $service) {
                $appointment = $this->createServiceAppointment($patient_id, $service, $preferred_date, $package);
                if ($appointment) {
                    $appointments[] = $appointment;
                }
            }

            echo json_encode([
                'status' => 'success',
                'message' => 'GCC services scheduled successfully',
                'appointments' => $appointments,
                'package' => $package
            ]);

        } catch (Exception $e) {
            log_message('error', 'GCC services scheduling error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Error scheduling GCC services: ' . $e->getMessage()
            ]);
        }
    }

    private function createServiceAppointment($patient_id, $service, $preferred_date, $package) {
        $appointment_date = $preferred_date ?: date('Y-m-d', strtotime('+1 day'));
        
        switch ($service) {
            case 'lab':
                return $this->createLabAppointment($patient_id, $appointment_date, $package);
            case 'radiology':
                return $this->createRadiologyAppointment($patient_id, $appointment_date, $package);
            case 'consultation':
                return $this->createConsultationAppointment($patient_id, $appointment_date);
            default:
                return [
                    'service' => $service,
                    'status' => 'scheduled',
                    'date' => $appointment_date,
                    'time' => '09:00 AM',
                    'department' => ucfirst($service)
                ];
        }
    }

    private function createLabAppointment($patient_id, $date, $package) {
        try {
            if ($this->db->table_exists('pathology_report')) {
                $lab_data = [
                    'patient_id' => $patient_id,
                    'test_name' => 'GCC Health Package - ' . ucfirst($package),
                    'short_name' => 'GCC-' . strtoupper(substr($package, 0, 3)),
                    'test_parameter_name' => json_encode($this->getGCCLabTests($package)),
                    'patient_charge' => $this->getGCCPackagePrice($package, 'lab'),
                    'doctor_name' => $this->getAvailableDoctor(),
                    'collection_date' => $date,
                    'reporting_date' => date('Y-m-d', strtotime($date . ' +2 days')),
                    'generated_by' => $this->session->userdata('admin_id') ?: 1
                ];

                $this->db->insert('pathology_report', $lab_data);
                return [
                    'service' => 'Laboratory',
                    'status' => 'scheduled',
                    'date' => $date,
                    'time' => '08:00 AM',
                    'department' => 'Laboratory',
                    'tests' => $this->getGCCLabTests($package),
                    'fasting_required' => true
                ];
            }
        } catch (Exception $e) {
            log_message('error', 'Lab appointment creation error: ' . $e->getMessage());
        }

        return [
            'service' => 'Laboratory',
            'status' => 'pending',
            'date' => $date,
            'time' => '08:00 AM',
            'department' => 'Laboratory'
        ];
    }

    private function createRadiologyAppointment($patient_id, $date, $package) {
        try {
            if ($this->db->table_exists('radiology_report')) {
                $radiology_data = [
                    'patient_id' => $patient_id,
                    'test_name' => 'GCC Radiology Package - ' . ucfirst($package),
                    'short_name' => 'RAD-GCC',
                    'test_parameter_name' => json_encode($this->getGCCRadiologyTests($package)),
                    'patient_charge' => $this->getGCCPackagePrice($package, 'radiology'),
                    'doctor_name' => $this->getAvailableDoctor(),
                    'test_date' => $date,
                    'generated_by' => $this->session->userdata('admin_id') ?: 1
                ];

                $this->db->insert('radiology_report', $radiology_data);
                return [
                    'service' => 'Radiology',
                    'status' => 'scheduled',
                    'date' => $date,
                    'time' => '10:00 AM',
                    'department' => 'Radiology',
                    'tests' => $this->getGCCRadiologyTests($package)
                ];
            }
        } catch (Exception $e) {
            log_message('error', 'Radiology appointment creation error: ' . $e->getMessage());
        }

        return [
            'service' => 'Radiology',
            'status' => 'pending',
            'date' => $date,
            'time' => '10:00 AM',
            'department' => 'Radiology'
        ];
    }

    private function createConsultationAppointment($patient_id, $date) {
        try {
            if ($this->db->table_exists('opd_details')) {
                $opd_data = [
                    'patient_id' => $patient_id,
                    'case_type' => 'GCC Consultation',
                    'appointment_date' => $date,
                    'appointment_time' => '11:00:00',
                    'cons_doctor' => $this->getAvailableDoctor(),
                    'generated_by' => $this->session->userdata('admin_id') ?: 1,
                    'amount' => 2000,
                    'payment_mode' => 'Cash'
                ];

                $this->db->insert('opd_details', $opd_data);
                return [
                    'service' => 'Consultation',
                    'status' => 'scheduled',
                    'date' => $date,
                    'time' => '11:00 AM',
                    'department' => 'OPD'
                ];
            }
        } catch (Exception $e) {
            log_message('error', 'Consultation appointment creation error: ' . $e->getMessage());
        }

        return [
            'service' => 'Consultation',
            'status' => 'pending',
            'date' => $date,
            'time' => '11:00 AM',
            'department' => 'OPD'
        ];
    }

    // =============================================
    // GCC PACKAGE CONFIGURATION
    // =============================================

    private function getGCCLabTests($package) {
        $tests = [
            'basic' => ['CBC', 'Blood Sugar', 'Lipid Profile', 'Kidney Function'],
            'comprehensive' => ['CBC', 'Blood Sugar', 'Lipid Profile', 'Kidney Function', 'Liver Function', 'Thyroid Function', 'Vitamin D'],
            'executive' => ['CBC', 'Blood Sugar', 'Lipid Profile', 'Kidney Function', 'Liver Function', 'Thyroid Function', 'Vitamin D', 'Vitamin B12', 'Cardiac Markers'],
            'women' => ['CBC', 'Blood Sugar', 'Lipid Profile', 'Kidney Function', 'Thyroid Function', 'Vitamin D', 'Hormonal Profile', 'Pap Smear'],
            'cardiac' => ['CBC', 'Blood Sugar', 'Lipid Profile', 'Cardiac Markers', 'ECG', 'Echo']
        ];

        return isset($tests[$package]) ? $tests[$package] : $tests['basic'];
    }

    private function getGCCRadiologyTests($package) {
        $tests = [
            'basic' => ['Chest X-Ray'],
            'comprehensive' => ['Chest X-Ray', 'Abdominal Ultrasound'],
            'executive' => ['Chest X-Ray', 'Abdominal Ultrasound', 'Echocardiogram'],
            'women' => ['Chest X-Ray', 'Pelvic Ultrasound', 'Mammogram'],
            'cardiac' => ['Chest X-Ray', 'Echocardiogram', 'Stress Test']
        ];

        return isset($tests[$package]) ? $tests[$package] : $tests['basic'];
    }

    private function getGCCPackagePrice($package, $service) {
        $prices = [
            'basic' => ['lab' => 3500, 'radiology' => 2000],
            'comprehensive' => ['lab' => 6000, 'radiology' => 3500],
            'executive' => ['lab' => 9000, 'radiology' => 5000],
            'women' => ['lab' => 7500, 'radiology' => 4000],
            'cardiac' => ['lab' => 8000, 'radiology' => 4500]
        ];

        return isset($prices[$package][$service]) ? $prices[$package][$service] : 3500;
    }
    
    /**
     * FIXED: Get patient details with better error handling and field detection
     */
public function getPatientDetails() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('id');
        
        if (!$patientId) {
            echo json_encode([
                'error' => 'Patient ID required'
            ]);
            return;
        }
        
        // Enhanced query with workflow information
        $this->db->select('
            patients.*, 
            blood_group.name as blood_group_name,
            COALESCE(patient_workflow, patient_type, "general") as workflow,
            CASE 
                WHEN UPPER(COALESCE(patient_workflow, patient_type, "general")) = "GCC"
                THEN 1 
                ELSE 0 
            END as requires_verification,
            CASE 
                WHEN fingerprint_template IS NOT NULL AND TRIM(fingerprint_template) != "" 
                THEN 1 
                ELSE 0 
            END as has_fingerprint_data
        ');
        $this->db->from('patients');
        $this->db->join('blood_group', 'blood_group.id = patients.blood_group', 'left');
        $this->db->where('patients.id', $patientId);
        $this->db->where('patients.is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            // Handle potential missing fields
            if (!isset($patient['blood_group_name']) || empty($patient['blood_group_name'])) {
                try {
                    $this->db->select('name');
                    $this->db->from('blood_group');
                    $this->db->where('id', $patient['blood_group'] ?? 0);
                    $bg_query = $this->db->get();
                    
                    if ($bg_query->num_rows() > 0) {
                        $patient['blood_group_name'] = $bg_query->row()->name;
                    } else {
                        $patient['blood_group_name'] = 'Unknown';
                    }
                } catch (Exception $e) {
                    $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
                }
            } else {
                $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
            }
            
            // Set default values for missing fields
            $patient['gender'] = $patient['gender'] ?? 'Unknown';
            $patient['known_allergies'] = $patient['known_allergies'] ?? $patient['allergies'] ?? 'None';
            $patient['image'] = !empty($patient['image']) ? $patient['image'] : 'uploads/patient_images/no_image.png';
            
            // Enhanced fingerprint verification status with workflow info
            $hasFingerprint = $patient['has_fingerprint_data'] == 1;
            $requiresVerification = $patient['requires_verification'] == 1;
            
            $patient['has_fingerprint'] = $hasFingerprint;
            $patient['verification_capable'] = $hasFingerprint && strlen($patient['fingerprint_template'] ?? '') > 100;
            $patient['workflow_requires_verification'] = $requiresVerification;
            
            // Enhanced fingerprint status object with workflow context
            $patient['fingerprint_status'] = [
                'enrolled' => $hasFingerprint,
                'template_size' => strlen($patient['fingerprint_template'] ?? ''),
                'image_available' => !empty($patient['fingerprint_image']),
                'enrollment_date' => $patient['created_at'] ?? null,
                'workflow' => $patient['workflow'],
                'verification_required' => $requiresVerification,
                'verification_policy' => $requiresVerification ? 
                    'GCC patients require mandatory biometric verification for triage' : 
                    'General patients can proceed without verification'
            ];
            
            // Add current location and routing history
            $patient['current_location'] = $this->getCurrentPatientLocation($patientId);
            $patient['routing_history'] = $this->getPatientRoutingHistory($patientId);
            
            // Log workflow detection for debugging
            log_message('info', "Triage: Patient {$patient['patient_name']} - Workflow: {$patient['workflow']}, Requires Verification: " . ($requiresVerification ? 'Yes' : 'No'));
            
            // Remove sensitive data from response
            unset($patient['fingerprint_template']);
            if (isset($patient['fingerprint_image'])) {
                unset($patient['fingerprint_image']);
            }
            
            echo json_encode($patient);
            
        } else {
            echo json_encode([
                'error' => 'Patient not found or inactive',
                'patient_id' => $patientId
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Triage: Error getting patient details: ' . $e->getMessage());
        echo json_encode([
            'error' => 'Database error: ' . $e->getMessage(),
            'patient_id' => $patientId ?? 'unknown'
        ]);
    }
}

public function getStoredTemplate() {
    // This is the endpoint the JavaScript is actually calling
    $this->getStoredFingerprintTemplate();
}

/**
 * FIXED: Update getStoredFingerprintTemplate to match JavaScript expectations
 */
public function getStoredFingerprintTemplate() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if fingerprint columns exist
        if (!$this->db->field_exists('fingerprint_template', 'patients')) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Fingerprint functionality not available'
            ]);
            return;
        }
        
        // Get fingerprint data
        $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        
        // Add is_active check if column exists
        if ($this->db->field_exists('is_active', 'patients')) {
            $this->db->where('is_active', 'yes');
        }
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            if (!empty($patient['fingerprint_template'])) {
                $template = trim($patient['fingerprint_template']);
                
                // FIXED: Return field names that JavaScript expects
                echo json_encode([
                    'status' => 'success',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'template' => $template, // JavaScript expects 'template', not 'fingerprint_template'
                    'image' => $patient['fingerprint_image'] ?? '', // JavaScript expects 'image', not 'fingerprint_image'
                    'template_size' => strlen($template),
                    'quality' => 85, // Add quality field that JavaScript expects
                    'has_image' => !empty($patient['fingerprint_image'])
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No fingerprint template found for this patient',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name']
                ]);
            }
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or inactive',
                'patient_id' => $patientId
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

public function saveTriage() {
    // This is the endpoint the JavaScript is actually calling
    $this->save();
}

/**
 * Debug method to test triage endpoints
 */
public function testEndpoints() {
    echo "<h2>Triage Endpoints Test</h2>";
    echo "<p>Base URL: " . base_url() . "</p>";
    
    // Test getAllPatients
    echo "<h3>Test getAllPatients:</h3>";
    echo "<p><a href='" . base_url('admin/triage/getAllPatients') . "' target='_blank'>Test getAllPatients</a></p>";
    
    // Test getStoredTemplate
    echo "<h3>Test getStoredTemplate:</h3>";
    echo "<form method='post' action='" . base_url('admin/triage/getStoredTemplate') . "' target='_blank'>";
    echo "<input type='hidden' name='patient_id' value='133'>";
    echo "<button type='submit'>Test getStoredTemplate with Patient ID 133</button>";
    echo "</form>";
    
    // Check if fingerprint column exists
    if ($this->db->field_exists('fingerprint_template', 'patients')) {
        echo "<p>✅ fingerprint_template column exists in patients table</p>";
        
        // Check if any patients have fingerprints
        $this->db->select('COUNT(*) as count');
        $this->db->from('patients');
        $this->db->where('fingerprint_template IS NOT NULL');
        $this->db->where('fingerprint_template !=', '');
        $query = $this->db->get();
        $result = $query->row();
        
        echo "<p>Patients with fingerprints: " . $result->count . "</p>";
        
        if ($result->count > 0) {
            // Show first few patients with fingerprints
            $this->db->select('id, patient_name, LENGTH(fingerprint_template) as template_length');
            $this->db->from('patients');
            $this->db->where('fingerprint_template IS NOT NULL');
            $this->db->where('fingerprint_template !=', '');
            $this->db->limit(5);
            $query = $this->db->get();
            
            echo "<h4>Patients with fingerprints:</h4>";
            echo "<table border='1'>";
            echo "<tr><th>ID</th><th>Name</th><th>Template Length</th><th>Test</th></tr>";
            foreach ($query->result_array() as $patient) {
                echo "<tr>";
                echo "<td>" . $patient['id'] . "</td>";
                echo "<td>" . $patient['patient_name'] . "</td>";
                echo "<td>" . $patient['template_length'] . " chars</td>";
                echo "<td><a href='" . base_url('admin/triage/getStoredTemplate') . "?patient_id=" . $patient['id'] . "' target='_blank'>Test</a></td>";
                echo "</tr>";
            }
            echo "</table>";
        }
    } else {
        echo "<p>❌ fingerprint_template column does not exist in patients table</p>";
    }
}

    public function verifyFingerprint() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('patient_id');
            $matchScore = $this->input->post('match_score');
            $isMatch = $this->input->post('is_match');
            $threshold = $this->input->post('threshold') ?? 60;
            $serviceUrl = $this->input->post('service_url');
            
            if (!$patientId) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID required'
                ]);
                return;
            }
            
            if (!is_numeric($matchScore)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invalid match score: ' . $matchScore
                ]);
                return;
            }
            
            // Validate patient exists
            $this->db->select('id, patient_name, fingerprint_template');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            
            if ($this->db->field_exists('is_active', 'patients')) {
                $this->db->where('is_active', 'yes');
            }
            
            $patient_query = $this->db->get();
            
            if ($patient_query->num_rows() == 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found or inactive',
                    'patient_id' => $patientId
                ]);
                return;
            }
            
            $patient = $patient_query->row_array();
            $hasStoredTemplate = !empty($patient['fingerprint_template']);
            
            // Log verification attempt
            $verificationLogId = $this->logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold, $serviceUrl, $hasStoredTemplate);
            
            // Prepare response
            $verificationResult = [
                'patient_id' => $patientId,
                'patient_name' => $patient['patient_name'],
                'verification_type' => '1:1',
                'timestamp' => date('Y-m-d H:i:s'),
                'staff_id' => $this->getStaffId(),
                'match_score' => floatval($matchScore),
                'threshold_used' => intval($threshold),
                'max_possible_score' => 199,
                'service_url' => $serviceUrl ?? 'unknown',
                'verification_log_id' => $verificationLogId,
                'has_stored_template' => $hasStoredTemplate,
                'template_size' => $hasStoredTemplate ? strlen($patient['fingerprint_template']) : 0
            ];
            
            // Determine verification result
            $matchPassed = ($isMatch == 'true' || $isMatch === true || $isMatch == 1);
            
            if ($matchPassed && $hasStoredTemplate) {
                $verificationResult['status'] = 'success';
                $verificationResult['message'] = 'Patient identity verified successfully for triage';
                $verificationResult['verification_passed'] = true;
                $verificationResult['verification_method'] = 'fingerprint_match';
            } else if (!$hasStoredTemplate) {
                $verificationResult['status'] = 'warning';
                $verificationResult['message'] = 'No stored fingerprint available for verification';
                $verificationResult['verification_passed'] = false;
                $verificationResult['verification_method'] = 'no_template';
            } else {
                $verificationResult['status'] = 'failed';
                $verificationResult['message'] = 'Fingerprint verification failed - identity not confirmed';
                $verificationResult['verification_passed'] = false;
                $verificationResult['verification_method'] = 'fingerprint_mismatch';
            }
            
            echo json_encode($verificationResult);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Verification error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'unknown'
            ]);
        }
    }

    // =============================================
    // TRIAGE MANAGEMENT METHODS
    // =============================================

public function save() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id', TRUE);
        $department_assigned = $this->input->post('department_assigned', TRUE);
        $priority = $this->input->post('priority', TRUE);
        $vitals = $this->input->post('vitals', TRUE);
        $chief_complaint = $this->input->post('chief_complaint', TRUE);
        $assessment_notes = $this->input->post('assessment_notes', TRUE);
        
        // Validation
        if (!$patient_id) {
            echo json_encode(['status' => 'error', 'message' => 'Patient ID is required']);
            return;
        }
        
        if (!$priority) {
            echo json_encode(['status' => 'error', 'message' => 'Triage priority is required']);
            return;
        }
        
        if (!$chief_complaint) {
            echo json_encode(['status' => 'error', 'message' => 'Chief complaint is required']);
            return;
        }
        
        // Get patient details
        $patient = $this->patient_model->getpatientDetails($patient_id);
        if (!$patient) {
            echo json_encode(['status' => 'error', 'message' => 'Patient not found']);
            return;
        }
        
        // Prepare triage data
        $triage_data = [
            'patient_id' => $patient_id,
            'priority' => $priority,
            'chief_complaint' => $chief_complaint,
            'assessment_notes' => $assessment_notes,
            'vitals' => $vitals,
            'department_assigned' => $department_assigned,
            'verification_status' => $this->input->post('verification_status', TRUE) ?: '0',
            'verification_score' => $this->input->post('verification_score', TRUE) ?: '0',
            'verification_method' => $this->input->post('verification_method', TRUE) ?: 'none',
            'patient_workflow' => $this->input->post('patient_workflow', TRUE) ?: 'general',
            'triage_by' => $this->getStaffId(),
            'triage_date' => date('Y-m-d'),
            'triage_time' => date('H:i:s'),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Insert triage record
        $this->db->insert('triage', $triage_data);
        $triage_id = $this->db->insert_id();
        
        if (!$triage_id) {
            echo json_encode(['status' => 'error', 'message' => 'Failed to save triage record']);
            return;
        }
        
        $response = [
            'status' => 'success',
            'message' => 'Triage saved successfully',
            'triage_id' => $triage_id,
            'patient_id' => $patient_id,
            'patient_name' => $patient['patient_name'],
            'priority' => $priority
        ];
        
        // Handle department routing if specified
        if ($department_assigned && $department_assigned !== '') {
            $routing_result = $this->routePatientToDepartment($triage_id, $patient_id, $department_assigned, $priority);
            $response['routing_result'] = $routing_result;
            
            // Send notification to target department
            if ($routing_result['status'] === 'success') {
                $notification_data = [
                    'patient_id' => $patient_id,
                    'patient_name' => $patient['patient_name'],
                    'triage_id' => $triage_id,
                    'priority' => $priority,
                    'chief_complaint' => $chief_complaint,
                    'vitals' => json_decode($vitals, true),
                    'from_department' => 'triage',
                    'routed_at' => date('Y-m-d H:i:s'),
                    'routing_id' => $routing_result['routing_id'] ?? null
                ];
                
                $this->sendNotificationToDepartment($department_assigned, 'new_patient_from_triage', $notification_data);
                
                $response['notification_sent'] = true;
                $response['target_department'] = $department_assigned;
            }
        }
        
        // Add workflow compliance info
        $response['workflow_compliance'] = [
            'workflow_type' => $triage_data['patient_workflow'],
            'verification_completed' => $triage_data['verification_status'] === '1',
            'verification_method' => $triage_data['verification_method'],
            'compliance_status' => 'compliant'
        ];
        
        echo json_encode($response);
        
    } catch (Exception $e) {
        log_message('error', 'Triage save error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to save triage: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get patient triage history
 */
public function getPatientTriageHistory() {
    header('Content-Type: application/json');
    
    $patientId = $this->input->post('patient_id');
    
    if (!$patientId) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Patient ID required'
        ]);
        return;
    }
    
    try {
        // Check if triage table exists
        if (!$this->db->table_exists('triage')) {
            echo json_encode([
                'status' => 'success',
                'records' => [],
                'message' => 'No triage records available'
            ]);
            return;
        }
        
        $this->db->select('
            t.id,
            t.patient_id,
            t.vitals,
            t.notes,
            t.priority,
            t.status,
            t.verification_status,
            t.created_at,
            t.updated_at,
            COALESCE(s.name, CONCAT(s.firstname, " ", s.lastname), "Unknown Staff") as staff_name
        ');
        $this->db->from('triage t');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->where('t.patient_id', $patientId);
        $this->db->order_by('t.created_at', 'DESC');
        $this->db->limit(10); // Last 10 triage records
        
        $query = $this->db->get();
        
        if ($query) {
            $records = $query->result_array();
            
            // Parse vitals JSON for each record
            foreach ($records as &$record) {
                if (!empty($record['vitals'])) {
                    $record['vitals_parsed'] = json_decode($record['vitals'], true);
                }
            }
            
            echo json_encode([
                'status' => 'success',
                'records' => $records,
                'count' => count($records)
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'records' => []
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient triage history: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'records' => []
        ]);
    }
}

/**
 * Get patient routing history (this method already exists but let me show the enhanced version)
 */
public function getPatientRoutingHistoryJson() {
    header('Content-Type: application/json');
    
    $patientId = $this->input->post('patient_id');
    
    if (!$patientId) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Patient ID required'
        ]);
        return;
    }
    
    try {
        $routingHistory = $this->getPatientRoutingHistory($patientId);
        
        echo json_encode([
            'status' => 'success',
            'routing_history' => $routingHistory,
            'count' => count($routingHistory)
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient routing history: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'routing_history' => []
        ]);
    }
}

/**
 * Complete current triage queue
 */
private function complete_triage_queue($patient_id) {
    try {
        $this->db->where('patient_id', $patient_id);
        $this->db->where('department', 'triage');
        $this->db->where_in('status', ['waiting', 'in_progress']);
        $this->db->update('patient_queue', [
            'status' => 'completed',
            'completed_at' => date('Y-m-d H:i:s'),
            'notes' => 'Triage assessment completed'
        ]);
        
        return true;
    } catch (Exception $e) {
        log_message('error', 'Error completing triage queue: ' . $e->getMessage());
        return false;
    }
}

/**
 * Route patient after triage completion
 */
private function route_after_triage($patient_id, $priority, $workflow_type) {
    try {
        $routing_results = [];
        
        switch ($workflow_type) {
            case 'gcc':
                // GCC patients go to lab and radiology
                $lab_result = $this->Queue_model->addPatientToQueue($patient_id, 'laboratory', $priority, 'gcc');
                $rad_result = $this->Queue_model->addPatientToQueue($patient_id, 'radiology', $priority, 'gcc');
                
                $routing_results['laboratory'] = $lab_result;
                $routing_results['radiology'] = $rad_result;
                
                // Send notifications
                $this->send_queue_notification('laboratory', 'new_patient_from_triage', [
                    'patient_id' => $patient_id,
                    'priority' => $priority,
                    'workflow' => 'gcc'
                ]);
                
                $this->send_queue_notification('radiology', 'new_patient_from_triage', [
                    'patient_id' => $patient_id,
                    'priority' => $priority,
                    'workflow' => 'gcc'
                ]);
                break;
                
            case 'emergency':
                // Emergency patients go directly to emergency department
                $emr_result = $this->Queue_model->addPatientToQueue($patient_id, 'emergency', 'urgent', 'emergency');
                $routing_results['emergency'] = $emr_result;
                
                $this->send_queue_notification('emergency', 'urgent_patient_incoming', [
                    'patient_id' => $patient_id,
                    'priority' => 'urgent',
                    'from_triage' => true
                ]);
                break;
                
            default:
                // General patients go to consultation
                $consultation_priority = ($priority === 'urgent') ? 'high' : $priority;
                $cons_result = $this->Queue_model->addPatientToQueue($patient_id, 'consultation', $consultation_priority, 'general');
                $routing_results['consultation'] = $cons_result;
                
                $this->send_queue_notification('consultation', 'new_patient_from_triage', [
                    'patient_id' => $patient_id,
                    'priority' => $consultation_priority,
                    'workflow' => 'general'
                ]);
        }
        
        return $routing_results;
        
    } catch (Exception $e) {
        log_message('error', 'Error routing after triage: ' . $e->getMessage());
        return ['error' => $e->getMessage()];
    }
}

/**
 * AJAX: Get triage queue status
 */
public function get_queue_status() {
    header('Content-Type: application/json');
    
    try {
        $result = $this->Queue_model->getDepartmentQueue('triage');
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Call next patient for triage
 */
public function call_next_patient() {
    header('Content-Type: application/json');
    
    try {
        $served_by = $this->getStaffId();
        $result = $this->Queue_model->callNextPatient('triage', $served_by);
        
        if ($result['status'] === 'success') {
            // Play notification sound
            $this->send_queue_notification('triage', 'patient_called_for_triage', $result['patient']);
        }
        
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Remove patient from triage queue (double-click action)
 */
public function remove_from_queue() {
    header('Content-Type: application/json');
    
    try {
        $queue_id = $this->input->post('queue_id');
        $reason = $this->input->post('reason', TRUE) ?: 'Triage completed';
        
        $result = $this->Queue_model->skipPatient($queue_id, $reason);
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Quick triage assessment
 */
public function quick_triage() {
    header('Content-Type: application/json');
    
    try {
        $queue_id = $this->input->post('queue_id');
        $priority = $this->input->post('priority', TRUE);
        $next_department = $this->input->post('next_department', TRUE);
        $notes = $this->input->post('notes', TRUE);
        
        // Get queue details
        $this->db->where('id', $queue_id);
        $queue = $this->db->get('patient_queue')->row_array();
        
        if (!$queue) {
            echo json_encode(['status' => 'error', 'message' => 'Queue record not found']);
            return;
        }
        
        // Create quick triage record
        $triage_data = [
            'patient_id' => $queue['patient_id'],
            'vitals' => json_encode(['quick_assessment' => true]),
            'notes' => $notes ?: 'Quick triage assessment',
            'priority' => $priority,
            'status' => 'completed',
            'triage_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('triage', $triage_data);
        
        // Complete current queue and route to next department
        $result = $this->Queue_model->completeService($queue_id, $next_department, $notes);
        
        if ($result['status'] === 'success' && $next_department) {
            $this->send_queue_notification($next_department, 'quick_triage_patient_incoming', [
                'patient_id' => $queue['patient_id'],
                'priority' => $priority,
                'from_quick_triage' => true
            ]);
        }
        
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Check for incoming patient notifications from other departments
 */
public function check_incoming_notifications() {
    header('Content-Type: application/json');
    
    try {
        $notification_file = APPPATH . 'cache/queue_notifications_triage.json';
        
        if (file_exists($notification_file)) {
            $notification = json_decode(file_get_contents($notification_file), true);
            
            // Clear notification after reading
            unlink($notification_file);
            
            // Mark as read in database
            $this->db->where('department', 'triage');
            $this->db->where('status', 'pending');
            $this->db->update('queue_notifications', [
                'status' => 'read',
                'read_at' => date('Y-m-d H:i:s')
            ]);
            
            echo json_encode([
                'status' => 'success',
                'notification' => $notification,
                'has_notification' => true,
                'sound' => true
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'has_notification' => false
            ]);
        }
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error checking notifications: ' . $e->getMessage()
        ]);
    }
}

/**
 * Send notification to department about patient routing
 */
public function send_queue_notification() {
    header('Content-Type: application/json');
    
    try {
        $department = $this->input->post('department', TRUE);
        $action = $this->input->post('action', TRUE);
        $notification_data = $this->input->post('notification_data', TRUE);
        
        if (!$department || !$action) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Department and action are required'
            ]);
            return;
        }
        
        // Parse notification data
        $data = [];
        if ($notification_data) {
            $data = json_decode($notification_data, true) ?: [];
        }
        
        // Create notification payload
        $notification = [
            'department' => $department,
            'action' => $action,
            'data' => $data,
            'timestamp' => time(),
            'from_department' => 'triage',
            'sound' => true,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Store notification in database for audit trail
        $this->db->insert('queue_notifications', [
            'department' => $department,
            'action' => $action,
            'notification_data' => json_encode($notification),
            'status' => 'pending',
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        $notification_id = $this->db->insert_id();
        
        // Store in file system for real-time pickup
        $notification_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
        if (!is_dir(APPPATH . 'cache')) {
            mkdir(APPPATH . 'cache', 0755, true);
        }
        
        file_put_contents($notification_file, json_encode($notification));
        
        // Log the notification
        log_message('info', "Triage notification sent to {$department}: {$action}");
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Notification sent successfully',
            'notification_id' => $notification_id,
            'department' => $department,
            'action' => $action
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Triage notification failed: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to send notification: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get triage statistics for dashboard
 */
public function get_triage_stats() {
    header('Content-Type: application/json');
    
    try {
        $stats = $this->Queue_model->getQueueStatistics('triage');
        
        // Add triage-specific stats
        $today = date('Y-m-d');
        
        // Priority breakdown
        $this->db->select("priority, COUNT(*) as count");
        $this->db->from('patient_queue');
        $this->db->where('department', 'triage');
        $this->db->where('DATE(created_at)', $today);
        $this->db->group_by('priority');
        $priority_stats = $this->db->get()->result_array();
        
        $stats['priority_breakdown'] = $priority_stats;
        
        echo json_encode(['status' => 'success', 'stats' => $stats]);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Bulk triage operations
 */
public function bulk_triage() {
    header('Content-Type: application/json');
    
    try {
        $operation = $this->input->post('operation', TRUE);
        $queue_ids = $this->input->post('queue_ids');
        $priority = $this->input->post('priority', TRUE);
        $next_department = $this->input->post('next_department', TRUE);
        
        if (!$operation || !$queue_ids) {
            echo json_encode(['status' => 'error', 'message' => 'Operation and queue IDs required']);
            return;
        }
        
        $processed = 0;
        
        foreach ($queue_ids as $queue_id) {
            switch ($operation) {
                case 'quick_complete':
                    $result = $this->Queue_model->completeService($queue_id, $next_department, 'Bulk triage completed');
                    if ($result['status'] === 'success') $processed++;
                    break;
                    
                case 'change_priority':
                    $this->db->where('id', $queue_id);
                    $this->db->update('patient_queue', ['priority' => $priority]);
                    $processed++;
                    break;
                    
                case 'skip_all':
                    $result = $this->Queue_model->skipPatient($queue_id, 'Bulk skip operation');
                    if ($result['status'] === 'success') $processed++;
                    break;
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'message' => "Bulk operation completed. {$processed} patients processed."
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Route patient to department with workflow compliance
 */
private function routePatientToDepartment($patientId, $triageId, $department, $priority) {
    try {
        // Create routing table if it doesn't exist
        $this->createPatientRoutingTable();
        
        $routingData = [
            'patient_id' => $patientId,
            'triage_id' => $triageId,
            'from_department' => 'triage',
            'to_department' => $department,
            'priority' => $priority,
            'status' => 'pending',
            'routed_by' => $this->getStaffId(),
            'routed_at' => date('Y-m-d H:i:s'),
            'routing_notes' => "Routed from triage with priority: $priority"
        ];
        
        $this->db->insert('patient_routing', $routingData);
        
        return [
            'status' => 'success',
            'message' => "Patient routed to $department successfully",
            'department' => $department,
            'priority' => $priority,
            'routing_id' => $this->db->insert_id()
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'message' => 'Routing failed: ' . $e->getMessage()
        ];
    }
}

/**
 * Log triage workflow compliance for audit trail
 */
private function logTriageWorkflowCompliance($patientId, $triageId, $details) {
    try {
        $logData = [
            'patient_id' => $patientId,
            'triage_id' => $triageId,
            'action' => 'triage_assessment',
            'details' => json_encode($details),
            'performed_by' => $this->getStaffId(),
            'performed_at' => date('Y-m-d H:i:s'),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent(),
            'module' => 'triage'
        ];
        
        // Insert into workflow compliance log (create table if needed)
        if ($this->db->table_exists('workflow_compliance_log')) {
            $this->db->insert('workflow_compliance_log', $logData);
        }
        
        // Also log to PHP log for debugging
        log_message('info', "Triage Workflow Compliance: Assessment for patient $patientId - " . json_encode($details));
        
    } catch (Exception $e) {
        log_message('error', 'Error logging triage workflow compliance: ' . $e->getMessage());
    }
}

/**
 * Check patient workflow requirement for triage
 */
public function checkPatientWorkflow() {
    header('Content-Type: application/json');
    
    $patientId = $this->input->post('patient_id') ?: $this->input->get('patient_id');
    
    if (!$patientId) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Patient ID required'
        ]);
        return;
    }
    
    try {
        // Get patient workflow and verification requirements
        $this->db->select('
            id,
            patient_name,
            patient_type,
            COALESCE(patient_workflow, patient_type, "general") as workflow,
            CASE 
                WHEN fingerprint_template IS NOT NULL AND TRIM(fingerprint_template) != "" 
                THEN 1 
                ELSE 0 
            END as has_fingerprint,
            CASE 
                WHEN UPPER(COALESCE(patient_workflow, patient_type, "general")) = "GCC"
                THEN 1 
                ELSE 0 
            END as requires_verification,
            LENGTH(fingerprint_template) as template_length
        ');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            echo json_encode([
                'status' => 'success',
                'patient' => $patient,
                'verification_policy' => [
                    'workflow' => $patient['workflow'],
                    'requires_verification' => $patient['requires_verification'] == 1,
                    'has_fingerprint' => $patient['has_fingerprint'] == 1,
                    'can_proceed_without_verification' => $patient['requires_verification'] == 0,
                    'policy_reason' => $patient['requires_verification'] == 1 ? 
                        'GCC patients require mandatory biometric verification for triage' : 
                        'General/Specialized patients can proceed without verification',
                    'module' => 'triage'
                ]
            ]);
            
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or not active'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get triage workflow statistics
 */
public function getTriageWorkflowStats() {
    header('Content-Type: application/json');
    
    try {
        $stats = [];
        
        // Total patients by workflow in triage
        if ($this->db->table_exists('triage')) {
            $this->db->select('
                patient_workflow,
                verification_method,
                COUNT(*) as count
            ');
            $this->db->from('triage');
            $this->db->where('created_at >=', date('Y-m-d', strtotime('-30 days')));
            $this->db->group_by(['patient_workflow', 'verification_method']);
            $query = $this->db->get();
            
            $stats['recent_triage_assessments'] = $query->result_array();
        }
        
        // GCC patients compliance in triage
        $this->db->select('
            COUNT(*) as total_gcc,
            SUM(CASE WHEN fingerprint_template IS NOT NULL AND fingerprint_template != "" THEN 1 ELSE 0 END) as gcc_with_fingerprints
        ');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->where('UPPER(COALESCE(patient_workflow, patient_type, "general"))', 'GCC');
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $gccStats = $query->row_array();
            $stats['gcc_triage_compliance'] = [
                'total_gcc_patients' => $gccStats['total_gcc'],
                'gcc_with_fingerprints' => $gccStats['gcc_with_fingerprints'],
                'gcc_without_fingerprints' => $gccStats['total_gcc'] - $gccStats['gcc_with_fingerprints'],
                'compliance_rate' => $gccStats['total_gcc'] > 0 ? 
                    round(($gccStats['gcc_with_fingerprints'] / $gccStats['total_gcc']) * 100, 1) : 0
            ];
        }
        
        echo json_encode([
            'status' => 'success',
            'stats' => $stats,
            'generated_at' => date('Y-m-d H:i:s'),
            'module' => 'triage'
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error generating triage stats: ' . $e->getMessage()
        ]);
    }
}

/**
 * Debug patient workflow detection for triage
 */
public function debugPatientWorkflow() {
    header('Content-Type: application/json');
    
    $patientId = $this->input->get('patient_id') ?: $this->input->post('patient_id');
    
    if (!$patientId) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Patient ID required'
        ]);
        return;
    }
    
    try {
        // Get all workflow-related fields for debugging
        $this->db->select('
            id,
            patient_name,
            patient_type,
            patient_workflow,
            COALESCE(patient_workflow, patient_type, "general") as resolved_workflow,
            CASE 
                WHEN UPPER(COALESCE(patient_workflow, patient_type, "general")) = "GCC"
                THEN "REQUIRED" 
                ELSE "OPTIONAL" 
            END as verification_requirement,
            fingerprint_template IS NOT NULL as has_template_field,
            fingerprint_template IS NOT NULL AND fingerprint_template != "" as has_template_data,
            LENGTH(fingerprint_template) as template_length
        ');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            echo json_encode([
                'status' => 'success',
                'patient_id' => $patientId,
                'debug_info' => $patient,
                'triage_workflow_logic' => [
                    'patient_type_field' => $patient['patient_type'],
                    'patient_workflow_field' => $patient['patient_workflow'],
                    'resolved_workflow' => $patient['resolved_workflow'],
                    'is_gcc' => strtoupper($patient['resolved_workflow']) === 'GCC',
                    'verification_required' => $patient['verification_requirement'] === 'REQUIRED',
                    'has_fingerprint' => $patient['has_template_data'] == 1,
                    'triage_policy' => $patient['verification_requirement'] === 'REQUIRED' ? 
                        'Must verify before triage assessment' : 
                        'Can proceed directly to triage'
                ],
                'sql_query' => $this->db->last_query(),
                'module' => 'triage'
            ]);
            
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found',
                'patient_id' => $patientId
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Debug failed: ' . $e->getMessage(),
            'patient_id' => $patientId
        ]);
    }
}
    
    public function viewTriage($triage_id = null) {
    header('Content-Type: application/json');
    
    if (!$triage_id) {
        $triage_id = $this->input->post('triage_id');
    }
    
    if (!$triage_id) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Triage ID is required'
        ]);
        return;
    }
    
    try {
        // Get triage record with patient and staff details
        $this->db->select('t.*, p.patient_name, p.mobileno, p.age, p.gender, s.name as staff_name');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.staff_id', 'left');
        $this->db->where('t.id', $triage_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $triage = $query->row_array();
            
            // Parse vitals JSON
            if (!empty($triage['vitals'])) {
                $triage['vitals_parsed'] = json_decode($triage['vitals'], true);
            }
            
            // Get routing history for this triage
            $triage['routing_history'] = $this->getPatientRoutingHistory($triage['patient_id']);
            
            echo json_encode([
                'status' => 'success',
                'data' => $triage
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Triage record not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
}

public function editTriage($triage_id = null) {
    header('Content-Type: application/json');
    
    if (!$triage_id) {
        $triage_id = $this->input->post('triage_id');
    }
    
    if (!$triage_id) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Triage ID is required'
        ]);
        return;
    }
    
    try {
        // Get current triage data
        $this->db->select('t.*, p.patient_name, p.mobileno, p.age, p.gender');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->where('t.id', $triage_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $triage = $query->row_array();
            
            // Parse vitals JSON for editing
            if (!empty($triage['vitals'])) {
                $triage['vitals_parsed'] = json_decode($triage['vitals'], true);
            }
            
            echo json_encode([
                'status' => 'success',
                'data' => $triage,
                'edit_mode' => true
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Triage record not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
}

public function updateTriage() {
    header('Content-Type: application/json');
    
    try {
        $triage_id = $this->input->post('triage_id');
        $patient_id = $this->input->post('patient_id');
        $staff_id = $this->getStaffId();
        
        if (!$triage_id || !$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Triage ID and Patient ID are required'
            ]);
            return;
        }
        
        // Compile vitals data
        $vitals = [
            'blood_pressure' => [
                'systolic' => $this->input->post('bp_systolic'),
                'diastolic' => $this->input->post('bp_diastolic')
            ],
            'heart_rate' => $this->input->post('heart_rate'),
            'temperature' => $this->input->post('temperature'),
            'respiratory_rate' => $this->input->post('respiratory_rate'),
            'oxygen_saturation' => $this->input->post('oxygen_saturation'),
            'weight' => $this->input->post('weight'),
            'height' => $this->input->post('height'),
            'pain_scale' => $this->input->post('pain_scale'),
            'triage_priority' => $this->input->post('priority')
        ];
        
        // Prepare update data
        $updateData = [
            'priority' => $this->input->post('priority'),
            'vitals' => json_encode($vitals),
            'chief_complaint' => $this->input->post('chief_complaint'),
            'assessment_notes' => $this->input->post('assessment_notes'),
            'department_assigned' => $this->input->post('department_assigned'),
            'updated_at' => date('Y-m-d H:i:s'),
            'updated_by' => $staff_id
        ];
        
        // Update triage record
        $this->db->where('id', $triage_id);
        $result = $this->db->update('triage', $updateData);
        
        if ($result) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Triage record updated successfully',
                'triage_id' => $triage_id
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update triage record'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
}

public function viewRouting($routing_id = null) {
    header('Content-Type: application/json');
    
    if (!$routing_id) {
        $routing_id = $this->input->post('routing_id');
    }
    
    if (!$routing_id) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Routing ID is required'
        ]);
        return;
    }
    
    try {
        // Get routing record with patient and staff details
        $this->db->select('pr.*, p.patient_name, p.mobileno, p.age, p.gender, s.name as staff_name, t.priority');
        $this->db->from('patient_routing pr');
        $this->db->join('patients p', 'p.id = pr.patient_id', 'left');
        $this->db->join('staff s', 's.id = pr.routed_by', 'left');
        $this->db->join('triage t', 't.id = pr.triage_id', 'left');
        $this->db->where('pr.id', $routing_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $routing = $query->row_array();
            
            echo json_encode([
                'status' => 'success',
                'data' => $routing
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Routing record not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
}

    // =============================================
    // PATIENT ROUTING METHODS
    // =============================================

    public function routePatient($patient_id, $triage_id, $department, $priority = 'normal', $notes = '') {
        try {
            // Create routing tables if they don't exist
            $this->createPatientRoutingTable();
            
            // Validate inputs
            if (!$patient_id || !$department) {
                return [
                    'status' => 'error',
                    'message' => 'Patient ID and department are required'
                ];
            }
            
            // Create routing record
            $routing_id = $this->createPatientRoutingRecord($patient_id, $triage_id, $department, $priority, $notes);
            
            if (!$routing_id) {
                return [
                    'status' => 'error',
                    'message' => 'Failed to create routing record'
                ];
            }
            
            // Create department-specific record
            $departmentResult = $this->createDepartmentRecord($patient_id, $department, $triage_id, $priority);
            
            return [
                'status' => 'success',
                'message' => "Patient successfully routed to {$department}",
                'routing_id' => $routing_id,
                'department' => $department,
                'priority' => $priority,
                'department_result' => $departmentResult
            ];
            
        } catch (Exception $e) {
            log_message('error', 'Patient routing error: ' . $e->getMessage());
            return [
                'status' => 'error',
                'message' => 'Routing failed: ' . $e->getMessage()
            ];
        }
    }

    public function getPatientRoutingHistory($patient_id) {
        try {
            if (!$this->db->table_exists('patient_routing')) {
                return [];
            }
            
            $this->db->select('pr.*, s.name as staff_name');
            $this->db->from('patient_routing pr');
            $this->db->join('staff s', 's.id = pr.routed_by', 'left');
            $this->db->where('pr.patient_id', $patient_id);
            $this->db->order_by('pr.routed_at', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            return $query->result_array();
            
        } catch (Exception $e) {
            log_message('error', 'Error getting routing history: ' . $e->getMessage());
            return [];
        }
    }

    public function getDepartmentQueue() {
        header('Content-Type: application/json');
        
        try {
            $department = $this->input->get('department');
            
            if (!$department) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Department parameter required'
                ]);
                return;
            }
            
            // Get current queue for department
            $this->db->select('pr.*, p.patient_name, t.priority');
            $this->db->from('patient_routing pr');
            $this->db->join('patients p', 'p.id = pr.patient_id', 'left');
            $this->db->join('triage t', 't.id = pr.triage_id', 'left');
            $this->db->where('pr.to_department', $department);
            $this->db->where('pr.status', 'pending');
            $this->db->order_by('pr.routed_at', 'ASC');
            
            $query = $this->db->get();
            $queue = $query->result_array();
            
            echo json_encode([
                'status' => 'success',
                'department' => $department,
                'queue' => $queue,
                'count' => count($queue)
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    public function getAllDepartmentQueues() {
        header('Content-Type: application/json');
        
        try {
            $departments = $this->getAvailableDepartments();
            $queues = [];
            
            foreach ($departments as $dept_code => $dept_name) {
                // Count pending patients in each department
                $this->db->select('COUNT(*) as count');
                $this->db->from('patient_routing');
                $this->db->where('to_department', $dept_code);
                $this->db->where('status', 'pending');
                $query = $this->db->get();
                $result = $query->row();
                
                $queues[$dept_code] = [
                    'name' => $dept_name,
                    'count' => intval($result->count),
                    'status' => $result->count > 0 ? 'active' : 'idle'
                ];
            }
            
            echo json_encode([
                'status' => 'success',
                'queues' => $queues,
                'timestamp' => date('Y-m-d H:i:s')
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
    
    public function getRoutingList() {
    header('Content-Type: application/json');
    
    try {
        // Create routing table if it doesn't exist
        $this->createPatientRoutingTable();
        
        if (!$this->db->table_exists('patient_routing')) {
            echo json_encode([
                'draw' => intval($this->input->post('draw')) ?: 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Routing table not found'
            ]);
            return;
        }
        
        // Get DataTables parameters
        $draw = intval($this->input->post('draw')) ?: 1;
        $start = intval($this->input->post('start')) ?: 0;
        $length = intval($this->input->post('length')) ?: 10;
        $searchValue = $this->input->post('search')['value'] ?? '';
        
        // Base query for total records
        $this->db->from('patient_routing pr');
        $totalRecords = $this->db->count_all_results('', false);
        
        // Build main query with joins
        $this->db->select('
            pr.id,
            pr.patient_id,
            pr.triage_id,
            pr.from_department,
            pr.to_department,
            pr.priority,
            pr.status,
            pr.routed_at,
            pr.updated_at,
            pr.completed_at,
            pr.routing_notes,
            COALESCE(p.patient_name, CONCAT(p.firstname, " ", p.lastname), p.name, CONCAT("Patient #", pr.patient_id)) as patient_name,
            p.mobileno,
            p.age,
            p.gender,
            COALESCE(s.name, CONCAT(s.firstname, " ", s.lastname), "Unknown Staff") as staff_name,
            t.priority as triage_priority
        ');
        $this->db->from('patient_routing pr');
        $this->db->join('patients p', 'p.id = pr.patient_id', 'left');
        $this->db->join('staff s', 's.id = pr.routed_by', 'left');
        $this->db->join('triage t', 't.id = pr.triage_id', 'left');
        
        // Apply search filter if provided
        if (!empty($searchValue)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $searchValue);
            $this->db->or_like('p.firstname', $searchValue);
            $this->db->or_like('p.lastname', $searchValue);
            $this->db->or_like('p.mobileno', $searchValue);
            $this->db->or_like('pr.to_department', $searchValue);
            $this->db->or_like('pr.status', $searchValue);
            $this->db->group_end();
        }
        
        // Count filtered records
        $filteredRecords = $this->db->count_all_results('', false);
        
        // Apply ordering and limit
        $this->db->order_by('pr.routed_at', 'DESC');
        $this->db->limit($length, $start);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'draw' => $draw,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Database query failed: ' . $this->db->error()['message']
            ]);
            return;
        }
        
        $routings = $query->result_array();
        $data = [];
        
        foreach ($routings as $routing) {
            // Format patient info
            $patientInfo = '<div class="patient-info">';
            $patientInfo .= '<strong>' . htmlspecialchars($routing['patient_name']) . '</strong>';
            if (!empty($routing['age']) || !empty($routing['gender'])) {
                $patientInfo .= '<br><small class="text-muted">';
                if (!empty($routing['age'])) $patientInfo .= $routing['age'] . 'y ';
                if (!empty($routing['gender'])) $patientInfo .= ucfirst($routing['gender']);
                $patientInfo .= '</small>';
            }
            if (!empty($routing['mobileno'])) {
                $patientInfo .= '<br><small class="text-info"><i class="fa fa-phone"></i> ' . htmlspecialchars($routing['mobileno']) . '</small>';
            }
            $patientInfo .= '</div>';
            
            // Format departments
            $fromDept = $this->formatDepartmentDisplay($routing['from_department'] ?: 'Triage');
            $toDept = $this->formatDepartmentDisplay($routing['to_department']);
            
            // Format priority
            $priority = $routing['priority'] ?: $routing['triage_priority'] ?: 'normal';
            $priorityBadge = $this->formatPriorityBadge($priority);
            
            // Format time
            $routedAt = '';
            if ($routing['routed_at']) {
                $routedAt = '<span title="' . htmlspecialchars($routing['routed_at']) . '">';
                $routedAt .= date('d M Y', strtotime($routing['routed_at']));
                $routedAt .= '<br><small class="text-muted">' . date('H:i', strtotime($routing['routed_at'])) . '</small>';
                $routedAt .= '</span>';
            }
            
            // Format status
            $statusBadge = $this->formatStatusBadge($routing['status']);
            
            // Format actions
            $actions = '<div class="btn-group btn-group-xs">';
            $actions .= '<button class="btn btn-info" onclick="viewRouting(' . $routing['id'] . ')" title="View Details">';
            $actions .= '<i class="fa fa-eye"></i>';
            $actions .= '</button>';
            
            if ($routing['status'] !== 'completed') {
                $actions .= '<button class="btn btn-warning" onclick="updateRoutingStatus(' . $routing['id'] . ', \'in_progress\')" title="Mark In Progress">';
                $actions .= '<i class="fa fa-play"></i>';
                $actions .= '</button>';
                $actions .= '<button class="btn btn-success" onclick="updateRoutingStatus(' . $routing['id'] . ', \'completed\')" title="Mark Completed">';
                $actions .= '<i class="fa fa-check"></i>';
                $actions .= '</button>';
            }
            
            $actions .= '</div>';
            
            $data[] = [
                $patientInfo,
                $fromDept,
                $toDept,
                $priorityBadge,
                $routedAt,
                $statusBadge,
                $actions
            ];
        }
        
        echo json_encode([
            'draw' => $draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data,
            'success' => true
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error in getRoutingList: ' . $e->getMessage());
        echo json_encode([
            'draw' => intval($this->input->post('draw')) ?: 1,
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => 'Exception: ' . $e->getMessage(),
            'debug' => [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'last_query' => $this->db->last_query()
            ]
        ]);
    }
}

// Helper method to format department display with icons
private function formatDepartmentDisplay($department) {
    $icons = [
        'consultation' => 'fa-user-md',
        'emergency' => 'fa-ambulance', 
        'pharmacy' => 'fa-pills',
        'laboratory' => 'fa-flask',
        'radiology' => 'fa-x-ray',
        'admission' => 'fa-bed',
        'triage' => 'fa-stethoscope'
    ];
    
    $departmentNames = [
        'consultation' => 'Consultation/OPD',
        'emergency' => 'Emergency Dept',
        'pharmacy' => 'Pharmacy',
        'laboratory' => 'Laboratory', 
        'radiology' => 'Radiology',
        'admission' => 'IPD Admission',
        'triage' => 'Triage'
    ];
    
    $lowerDept = strtolower($department);
    $icon = isset($icons[$lowerDept]) ? $icons[$lowerDept] : 'fa-building';
    $name = isset($departmentNames[$lowerDept]) ? $departmentNames[$lowerDept] : ucfirst($department);
    
    return '<div class="department-display">
                <i class="fa ' . $icon . '" style="color: #007bff; margin-right: 5px;"></i>
                <span>' . htmlspecialchars($name) . '</span>
            </div>';
}

// Helper method to format priority badge
private function formatPriorityBadge($priority) {
    switch (strtolower($priority)) {
        case 'urgent':
        case '1':
            return '<span class="label label-danger">Urgent</span>';
        case 'high':
        case '2':
            return '<span class="label label-warning">High</span>';
        case 'normal':
        case '3':
            return '<span class="label label-info">Normal</span>';
        case 'low':
        case '4':
        case '5':
            return '<span class="label label-success">Low</span>';
        default:
            return '<span class="label label-default">' . ucfirst($priority) . '</span>';
    }
}

// Helper method to format status badge
private function formatStatusBadge($status) {
    switch (strtolower($status)) {
        case 'pending':
            return '<span class="label label-warning">Pending</span>';
        case 'in_progress':
            return '<span class="label label-info">In Progress</span>';
        case 'completed':
            return '<span class="label label-success">Completed</span>';
        case 'cancelled':
            return '<span class="label label-danger">Cancelled</span>';
        default:
            return '<span class="label label-default">' . ucfirst($status) . '</span>';
    }
}

// Helper method to format routed time
private function formatRoutedTime($routedAt) {
    if (!$routedAt) return '<span class="text-muted">-</span>';
    
    $time = strtotime($routedAt);
    $now = time();
    $diff = $now - $time;
    
    $timeAgo = '';
    if ($diff < 3600) {
        $timeAgo = floor($diff / 60) . ' min ago';
    } elseif ($diff < 86400) {
        $timeAgo = floor($diff / 3600) . ' hr ago';
    } else {
        $timeAgo = floor($diff / 86400) . ' days ago';
    }
    
    return '<span title="' . htmlspecialchars($routedAt) . '">' .
           date('d M Y H:i', $time) .
           '<br><small class="text-muted">' . $timeAgo . '</small>' .
           '</span>';
}

// Helper method to format routing actions
private function formatRoutingActions($routing) {
    $actions = '<div class="btn-group btn-group-xs">';
    
    // View button
    $actions .= '<button class="btn btn-info btn-xs" onclick="viewRouting(' . $routing['id'] . ')" title="View Details">';
    $actions .= '<i class="fa fa-eye"></i>';
    $actions .= '</button>';
    
    // Status update buttons (only if not completed)
    if ($routing['status'] !== 'completed') {
        if ($routing['status'] === 'pending') {
            $actions .= '<button class="btn btn-warning btn-xs" onclick="updateRoutingStatus(' . $routing['id'] . ', \'in_progress\')" title="Start Processing">';
            $actions .= '<i class="fa fa-play"></i>';
            $actions .= '</button>';
        }
        
        $actions .= '<button class="btn btn-success btn-xs" onclick="updateRoutingStatus(' . $routing['id'] . ', \'completed\')" title="Mark Completed">';
        $actions .= '<i class="fa fa-check"></i>';
        $actions .= '</button>';
    }
    
    $actions .= '</div>';
    return $actions;
}

    public function updateRoutingStatus() {
        header('Content-Type: application/json');
        
        try {
            $routing_id = $this->input->post('routing_id');
            $status = $this->input->post('status');
            $notes = $this->input->post('notes');
            
            if (!$routing_id || !$status) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Routing ID and status are required'
                ]);
                return;
            }
            
            $updateData = [
                'status' => $status,
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            if ($notes) {
                $updateData['routing_notes'] = $notes;
            }
            
            if ($status === 'completed') {
                $updateData['completed_at'] = date('Y-m-d H:i:s');
                $updateData['completed_by'] = $this->getStaffId();
            }
            
            $this->db->where('id', $routing_id);
            $result = $this->db->update('patient_routing', $updateData);
            
            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Routing status updated successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update routing status'
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    // =============================================
    // TRIAGE LIST METHODS
    // =============================================

/**
 * Enhanced getTriageList method for DataTables with live data
 */
public function getTriageList() {
    if (!$this->rbac->hasPrivilege('patient', 'can_view')) {
        access_denied();
    }
    
    header('Content-Type: application/json');
    
    try {
        // Check if triage table exists
        if (!$this->db->table_exists('triage')) {
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'message' => 'Triage table does not exist'
            ]);
            return;
        }
        
        // Get DataTables parameters
        $draw = intval($this->input->post('draw')) ?: 1;
        $start = intval($this->input->post('start')) ?: 0;
        $length = intval($this->input->post('length')) ?: 25;
        $searchValue = $this->input->post('search')['value'] ?? '';
        
        // Get filter parameters
        $dateFrom = $this->input->post('date_from');
        $dateTo = $this->input->post('date_to');
        $priorityFilter = $this->input->post('priority_filter');
        $statusFilter = $this->input->post('status_filter');
        $departmentFilter = $this->input->post('department_filter');
        
        // Base query for total records
        $this->db->from('triage');
        $totalRecords = $this->db->count_all_results('', false);
        
        // Build main query with joins
        $this->db->select('
            t.id,
            t.patient_id,
            t.vitals,
            t.notes,
            t.priority,
            t.status,
            t.created_at,
            t.verification_status,
            COALESCE(p.patient_name, CONCAT(p.firstname, " ", p.lastname), p.name, CONCAT("Patient #", t.patient_id)) as patient_name,
            p.mobileno,
            p.phone,
            p.contact,
            p.age,
            p.gender,
            COALESCE(s.name, CONCAT(s.firstname, " ", s.lastname), "Unknown Staff") as staff_name,
            COALESCE(r.name, "Staff") as staff_role,
            pr.to_department as department
        ');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'left');
        $this->db->join('roles r', 'sr.role_id = r.id', 'left');
        $this->db->join('patient_routing pr', 'pr.triage_id = t.id', 'left');
        
        // Apply date filters
        if (!empty($dateFrom)) {
            $this->db->where('DATE(t.created_at) >=', $dateFrom);
        }
        if (!empty($dateTo)) {
            $this->db->where('DATE(t.created_at) <=', $dateTo);
        }
        
        // Apply other filters
        if (!empty($priorityFilter)) {
            $this->db->where('t.priority', $priorityFilter);
        }
        if (!empty($statusFilter)) {
            $this->db->where('t.status', $statusFilter);
        }
        if (!empty($departmentFilter)) {
            $this->db->where('pr.to_department', $departmentFilter);
        }
        
        // Apply search filter
        if (!empty($searchValue)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $searchValue);
            $this->db->or_like('p.firstname', $searchValue);
            $this->db->or_like('p.lastname', $searchValue);
            $this->db->or_like('p.mobileno', $searchValue);
            $this->db->or_like('t.patient_id', $searchValue);
            $this->db->or_like('s.name', $searchValue);
            $this->db->group_end();
        }
        
        // Count filtered records
        $filteredRecords = $this->db->count_all_results('', false);
        
        // Apply ordering and limit
        $orderColumn = $this->input->post('order')[0]['column'] ?? 1;
        $orderDir = $this->input->post('order')[0]['dir'] ?? 'desc';
        
        $orderColumns = ['t.id', 't.created_at', 'patient_name', 't.priority', 't.vitals', 'department', 't.status', 'staff_name', 't.id'];
        if (isset($orderColumns[$orderColumn])) {
            $this->db->order_by($orderColumns[$orderColumn], $orderDir);
        } else {
            $this->db->order_by('t.created_at', 'DESC');
        }
        
        $this->db->limit($length, $start);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'draw' => $draw,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Database query failed: ' . $this->db->error()['message']
            ]);
            return;
        }
        
        $data = [];
        foreach ($query->result_array() as $row) {
            // Format patient info
            $patientInfo = [
                'name' => $row['patient_name'],
                'age' => $row['age'],
                'gender' => $row['gender'],
                'phone' => $row['mobileno'] ?: $row['phone'] ?: $row['contact']
            ];
            
            // Format staff info
            $staffInfo = [
                'name' => $row['staff_name'],
                'role' => $row['staff_role']
            ];
            
            $data[] = [
                'id' => $row['id'],
                'patient_id' => $row['patient_id'],
                'created_at' => $row['created_at'],
                'patient_info' => $patientInfo,
                'priority' => $row['priority'],
                'vitals' => $row['vitals'],
                'department' => $row['department'],
                'status' => $row['status'],
                'staff_info' => $staffInfo,
                'notes' => $row['notes'],
                'verification_status' => $row['verification_status']
            ];
        }
        
        echo json_encode([
            'draw' => $draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'draw' => 1,
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => 'Error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get triage statistics for dashboard
 */
public function getTriageStatistics() {
     if (!$this->rbac->hasPrivilege('triage', 'can_view')) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Access denied - insufficient privileges'
        ]);
        return;
    }
    
    header('Content-Type: application/json');
    
    try {
        $dateFrom = $this->input->get('date_from') ?: date('Y-m-d');
        $dateTo = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Emergency cases (urgent priority)
        $this->db->where('priority', 'urgent');
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $emergency = $this->db->count_all_results('triage');
        
        // Urgent cases (high priority)
        $this->db->where('priority', 'high');
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $urgent = $this->db->count_all_results('triage');
        
        // Completed cases
        $this->db->where('status', 'completed');
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $completed = $this->db->count_all_results('triage');
        
        // Total cases
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $total = $this->db->count_all_results('triage');
        
        echo json_encode([
            'status' => 'success',
            'data' => [
                'emergency' => $emergency,
                'urgent' => $urgent,
                'completed' => $completed,
                'total' => $total
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

/**
 * Get detailed triage record for viewing
 */
public function getTriageDetails() {
    if (!$this->rbac->hasPrivilege('triage', 'can_view')) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Access denied - insufficient privileges'
        ]);
        return;
    }
    
    header('Content-Type: application/json');
    
    $id = $this->input->get('id');
    
    if (!$id) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Triage ID is required'
        ]);
        return;
    }
    
    try {
        $this->db->select('
            t.*,
            COALESCE(p.patient_name, CONCAT(p.firstname, " ", p.lastname), p.name, CONCAT("Patient #", t.patient_id)) as patient_name,
            p.mobileno,
            p.phone,
            p.contact,
            p.age,
            p.gender,
            p.dob,
            COALESCE(s.name, CONCAT(s.firstname, " ", s.lastname), "Unknown Staff") as staff_name,
            COALESCE(r.name, "Staff") as staff_role,
            pr.to_department as department
        ');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'left');
        $this->db->join('roles r', 'sr.role_id = r.id', 'left');
        $this->db->join('patient_routing pr', 'pr.triage_id = t.id', 'left');
        $this->db->where('t.id', $id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $record = $query->row_array();
            
            // Parse vitals JSON
            if (!empty($record['vitals'])) {
                $record['vitals'] = json_decode($record['vitals'], true);
            }
            
            echo json_encode([
                'status' => 'success',
                'data' => $record
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Triage record not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Export triage records to Excel
 */
public function exportTriageRecords() {
    try {
        // Get filter parameters
        $dateFrom = $this->input->get('date_from');
        $dateTo = $this->input->get('date_to');
        $priorityFilter = $this->input->get('priority_filter');
        $statusFilter = $this->input->get('status_filter');
        $departmentFilter = $this->input->get('department_filter');
        
        // Build query with same filters as main list
        $this->db->select('
            t.id,
            t.patient_id,
            t.created_at,
            t.priority,
            t.status,
            t.notes,
            t.verification_status,
            COALESCE(p.patient_name, CONCAT(p.firstname, " ", p.lastname), p.name, CONCAT("Patient #", t.patient_id)) as patient_name,
            p.mobileno,
            p.age,
            p.gender,
            COALESCE(s.name, CONCAT(s.firstname, " ", s.lastname), "Unknown Staff") as staff_name,
            pr.to_department as department
        ');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->join('patient_routing pr', 'pr.triage_id = t.id', 'left');
        
        // Apply filters
        if (!empty($dateFrom)) {
            $this->db->where('DATE(t.created_at) >=', $dateFrom);
        }
        if (!empty($dateTo)) {
            $this->db->where('DATE(t.created_at) <=', $dateTo);
        }
        if (!empty($priorityFilter)) {
            $this->db->where('t.priority', $priorityFilter);
        }
        if (!empty($statusFilter)) {
            $this->db->where('t.status', $statusFilter);
        }
        if (!empty($departmentFilter)) {
            $this->db->where('pr.to_department', $departmentFilter);
        }
        
        $this->db->order_by('t.created_at', 'DESC');
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            // Set headers for CSV download
            $filename = 'triage_records_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            
            $output = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($output, [
                'ID', 'Patient ID', 'Patient Name', 'Phone', 'Age', 'Gender',
                'Priority', 'Status', 'Department', 'Staff', 'Date', 'Notes'
            ]);
            
            // CSV data
            foreach ($query->result_array() as $row) {
                fputcsv($output, [
                    $row['id'],
                    $row['patient_id'],
                    $row['patient_name'],
                    $row['mobileno'],
                    $row['age'],
                    $row['gender'],
                    strtoupper($row['priority']),
                    strtoupper(str_replace('_', ' ', $row['status'])),
                    $row['department'],
                    $row['staff_name'],
                    $row['created_at'],
                    strip_tags($row['notes'])
                ]);
            }
            
            fclose($output);
            exit;
        } else {
            echo 'No records found for export';
        }
        
    } catch (Exception $e) {
        echo 'Export error: ' . $e->getMessage();
    }
}

/**
 * Print individual triage record
 */
public function printTriageRecord($id = null) {
    if (!$id) {
        show_404();
        return;
    }
    
    try {
        // Get triage record with all details
        $this->db->select('
            t.*,
            COALESCE(p.patient_name, CONCAT(p.firstname, " ", p.lastname), p.name, CONCAT("Patient #", t.patient_id)) as patient_name,
            p.mobileno,
            p.age,
            p.gender,
            COALESCE(s.name, CONCAT(s.firstname, " ", s.lastname), "Unknown Staff") as staff_name,
            pr.to_department as department
        ');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->join('patient_routing pr', 'pr.triage_id = t.id', 'left');
        $this->db->where('t.id', $id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $data['record'] = $query->row_array();
            
            // Parse vitals
            if (!empty($data['record']['vitals'])) {
                $data['record']['vitals'] = json_decode($data['record']['vitals'], true);
            }
            
            // Load print view
            $this->load->view('admin/triage/print_record', $data);
        } else {
            show_404();
        }
        
    } catch (Exception $e) {
        show_error('Error loading record: ' . $e->getMessage());
    }
}

/**
 * Triage records listing page
 */
public function triageRecords() {
    // Add RBAC privilege check
    if (!$this->rbac->hasPrivilege('triage', 'can_view')) {
        access_denied();
    }
    
    // Set session menu data
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'triage');
    
    $data['page_title'] = 'Triage Records';
    
    try {
        $this->load->view('layout/header', $data);
        $this->load->view('admin/triage/records_list', $data);
        $this->load->view('layout/footer', $data);
    } catch (Exception $e) {
        log_message('error', 'View loading error: ' . $e->getMessage());
        show_error('Failed to load triage records view: ' . $e->getMessage());
    }
}

    // =============================================
    // HELPER METHODS
    // =============================================

    private function createTriageTable() {
        if (!$this->db->table_exists('triage')) {
            $sql = "CREATE TABLE IF NOT EXISTS `triage` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` varchar(50) NOT NULL,
                `vitals` text NOT NULL,
                `notes` text DEFAULT NULL,
                `priority` enum('urgent','high','normal','low') DEFAULT 'normal',
                `status` enum('pending','in_progress','completed','cancelled') DEFAULT 'pending',
                `verification_status` enum('verified','unverified','skipped') DEFAULT 'unverified',
                `triage_by` int(11) DEFAULT 1,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `triage_by` (`triage_by`),
                KEY `created_at` (`created_at`),
                KEY `status` (`status`),
                KEY `priority` (`priority`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
        } else {
            $this->addTriageColumns();
        }
    }

    private function createPatientRoutingTable() {
        if (!$this->db->table_exists('patient_routing')) {
            $sql = "CREATE TABLE IF NOT EXISTS `patient_routing` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` varchar(50) NOT NULL,
                `triage_id` int(11) DEFAULT NULL,
                `from_department` varchar(100) DEFAULT 'triage',
                `to_department` varchar(100) NOT NULL,
                `priority` enum('urgent','high','normal','low') DEFAULT 'normal',
                `status` enum('pending','in_progress','completed','cancelled') DEFAULT 'pending',
                `routing_notes` text DEFAULT NULL,
                `routed_by` int(11) NOT NULL,
                `routed_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `completed_at` timestamp NULL DEFAULT NULL,
                `completed_by` int(11) DEFAULT NULL,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `triage_id` (`triage_id`),
                KEY `to_department` (`to_department`),
                KEY `status` (`status`),
                KEY `routed_at` (`routed_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
        }
    }

    private function addTriageColumns() {
        try {
            $existingFields = $this->db->list_fields('triage');
            
            if (!in_array('priority', $existingFields)) {
                $sql = "ALTER TABLE `triage` 
                        ADD COLUMN `priority` ENUM('urgent','high','normal','low') DEFAULT 'normal' AFTER `notes`";
                $this->db->query($sql);
            }
            
            if (!in_array('status', $existingFields)) {
                $sql = "ALTER TABLE `triage` 
                        ADD COLUMN `status` ENUM('pending','in_progress','completed','cancelled') DEFAULT 'pending' AFTER `priority`";
                $this->db->query($sql);
            }
            
            if (!in_array('verification_status', $existingFields)) {
                $sql = "ALTER TABLE `triage` 
                        ADD COLUMN `verification_status` ENUM('verified','unverified','skipped') DEFAULT 'unverified' AFTER `status`";
                $this->db->query($sql);
            }
            
            if (!in_array('triage_by', $existingFields)) {
                $sql = "ALTER TABLE `triage` 
                        ADD COLUMN `triage_by` INT(11) DEFAULT 1 AFTER `verification_status`";
                $this->db->query($sql);
            }
            
            if (!in_array('updated_at', $existingFields)) {
                $sql = "ALTER TABLE `triage` 
                        ADD COLUMN `updated_at` TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP AFTER `created_at`";
                $this->db->query($sql);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error adding triage columns: ' . $e->getMessage());
        }
    }

    private function createPatientRoutingRecord($patient_id, $triage_id, $department, $priority, $notes) {
        try {
            $routingData = [
                'patient_id' => $patient_id,
                'triage_id' => $triage_id,
                'from_department' => 'triage',
                'to_department' => $department,
                'priority' => $priority ?: 'normal',
                'routing_notes' => $notes,
                'routed_by' => $this->getStaffId(),
                'routed_at' => date('Y-m-d H:i:s'),
                'status' => 'pending'
            ];
            
            $this->db->insert('patient_routing', $routingData);
            return $this->db->insert_id();
            
        } catch (Exception $e) {
            log_message('error', 'Error creating patient routing: ' . $e->getMessage());
            return false;
        }
    }

    private function createDepartmentRecord($patient_id, $department, $triage_id, $priority) {
        try {
            switch ($department) {
                case 'consultation':
                case 'opd':
                    return $this->createConsultationRecord($patient_id, $triage_id, $priority);
                    
                case 'pharmacy':
                    return $this->createPharmacyRecord($patient_id, $triage_id, $priority);
                    
                case 'laboratory':
                    return $this->createLaboratoryRecord($patient_id, $triage_id, $priority);
                    
                case 'radiology':
                    return $this->createRadiologyRecord($patient_id, $triage_id, $priority);
                    
                case 'emergency':
                    return $this->createEmergencyRecord($patient_id, $triage_id, $priority);
                    
                case 'admission':
                case 'ipd':
                    return $this->createAdmissionRecord($patient_id, $triage_id, $priority);
                    
                default:
                    return ['status' => 'routed', 'message' => 'Patient routed to department'];
            }
        } catch (Exception $e) {
            log_message('error', 'Error creating department record: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    private function createConsultationRecord($patient_id, $triage_id, $priority) {
        // Create OPD/consultation appointment
        if ($this->db->table_exists('opd_details')) {
            $opdData = [
                'patient_id' => $patient_id,
                'case_type' => 'Triage Referral',
                'casualty' => $priority === 'urgent' ? 'yes' : 'no',
                'old_patient' => 'yes',
                'symptoms' => 'Referred from Triage - Priority: ' . $priority,
                'appointment_date' => date('Y-m-d H:i:s'),
                'cons_doctor' => $this->getAvailableDoctor(),
                'generated_by' => $this->getStaffId()
            ];
            
            $this->db->insert('opd_details', $opdData);
            return ['status' => 'success', 'opd_id' => $this->db->insert_id()];
        }
        
        return ['status' => 'routed', 'message' => 'Routed to consultation'];
    }

/**
 * Fixed createLaboratoryRecord method - uses proper pathology_report structure
 */
private function createLaboratoryRecord($patient_id, $triage_id, $priority) {
    try {
        // Create lab order using pathology_report table
        if ($this->db->table_exists('pathology_report')) {
            $labData = [
                'patient_id' => $patient_id,
                'test_name' => 'Triage Lab Order',
                'short_name' => 'TRIAGE-LAB',
                'test_parameter_name' => json_encode(['Basic Blood Work', 'Urinalysis']),
                'patient_charge' => 0,
                'doctor_name' => $this->getAvailableDoctor(),
                'collection_date' => date('Y-m-d'),
                'reporting_date' => date('Y-m-d', strtotime('+2 days')),
                'generated_by' => $this->getStaffId(),
                'description' => 'Lab tests ordered from triage - Priority: ' . $priority,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('pathology_report', $labData);
            return ['status' => 'success', 'lab_id' => $this->db->insert_id()];
        }
        
        return ['status' => 'routed', 'message' => 'Routed to laboratory'];
        
    } catch (Exception $e) {
        log_message('error', 'Error creating laboratory record: ' . $e->getMessage());
        return ['status' => 'error', 'message' => $e->getMessage()];
    }
}

    private function createPharmacyRecord($patient_id, $triage_id, $priority) {
        // Create pharmacy record if table exists
        if ($this->db->table_exists('pharmacy')) {
            $pharmacyData = [
                'patient_id' => $patient_id,
                'prescription' => 'Medications as per triage assessment',
                'priority' => $priority,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('pharmacy', $pharmacyData);
            return ['status' => 'success', 'pharmacy_id' => $this->db->insert_id()];
        }
        
        return ['status' => 'routed', 'message' => 'Routed to pharmacy'];
    }

    private function createRadiologyRecord($patient_id, $triage_id, $priority) {
        // Create radiology order if table exists
        if ($this->db->table_exists('radiology')) {
            $radiologyData = [
                'patient_id' => $patient_id,
                'test_name' => 'Imaging as per triage assessment',
                'priority' => $priority,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('radiology', $radiologyData);
            return ['status' => 'success', 'radiology_id' => $this->db->insert_id()];
        }
        
        return ['status' => 'routed', 'message' => 'Routed to radiology'];
    }

    private function createEmergencyRecord($patient_id, $triage_id, $priority) {
        // Create emergency record
        if ($this->db->table_exists('emergency')) {
            $emergencyData = [
                'patient_id' => $patient_id,
                'triage_id' => $triage_id,
                'priority' => $priority,
                'status' => 'pending',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('emergency', $emergencyData);
            return ['status' => 'success', 'emergency_id' => $this->db->insert_id()];
        }
        
        return ['status' => 'routed', 'message' => 'Routed to emergency'];
    }

    private function createAdmissionRecord($patient_id, $triage_id, $priority) {
        // Create IPD admission record
        if ($this->db->table_exists('ipd_details')) {
            $ipdData = [
                'patient_id' => $patient_id,
                'case_type' => 'Triage Admission',
                'casualty' => $priority === 'urgent' ? 'yes' : 'no',
                'symptoms' => 'Admitted from triage - Priority: ' . $priority,
                'date' => date('Y-m-d H:i:s'),
                'consultant_doctor' => $this->getAvailableDoctor(),
                'generated_by' => $this->getStaffId()
            ];
            
            $this->db->insert('ipd_details', $ipdData);
            return ['status' => 'success', 'ipd_id' => $this->db->insert_id()];
        }
        
        return ['status' => 'routed', 'message' => 'Routed to IPD'];
    }

    private function getCurrentPatientLocation($patient_id) {
        // Check latest routing record
        if ($this->db->table_exists('patient_routing')) {
            $this->db->select('to_department, status, routed_at');
            $this->db->from('patient_routing');
            $this->db->where('patient_id', $patient_id);
            $this->db->where('status !=', 'completed');
            $this->db->order_by('routed_at', 'DESC');
            $this->db->limit(1);
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $routing = $query->row_array();
                return [
                    'department' => $this->getDepartmentName($routing['to_department']),
                    'status' => $routing['status'],
                    'since' => $routing['routed_at']
                ];
            }
        }
        
        return [
            'department' => 'Not tracked',
            'status' => 'unknown',
            'since' => null
        ];
    }
    
    /**
     * FIXED: This method was missing - defines triage priorities for dropdown
     */
    private function getTriagePriorities() {
        return [
            '1' => [
                'name' => 'Immediate (Red)', 
                'color' => '#d9534f', 
                'description' => 'Life-threatening - Immediate attention required'
            ],
            '2' => [
                'name' => 'Urgent (Orange)', 
                'color' => '#f0ad4e', 
                'description' => 'High priority - Within 10 minutes'
            ],
            '3' => [
                'name' => 'Less Urgent (Yellow)', 
                'color' => '#f39c12', 
                'description' => 'Moderate priority - Within 30 minutes'
            ],
            '4' => [
                'name' => 'Non-Urgent (Green)', 
                'color' => '#5cb85c', 
                'description' => 'Low priority - Within 60 minutes'
            ],
            '5' => [
                'name' => 'Fast Track (Blue)', 
                'color' => '#5bc0de', 
                'description' => 'Minor cases - Quick treatment'
            ]
        ];
    }

    private function getAvailableDepartments() {
        return [
            'consultation' => 'Consultation/OPD',
            'emergency' => 'Emergency Department',
            'pharmacy' => 'Pharmacy',
            'laboratory' => 'Laboratory',
            'radiology' => 'Radiology/Imaging',
            'admission' => 'IPD Admission'
        ];
    }

    private function getDepartmentName($code) {
        $departments = $this->getAvailableDepartments();
        return $departments[$code] ?? ucfirst($code);
    }

    private function getPriorityBadge($priority) {
        switch ($priority) {
            case 'urgent':
                return '<span class="label label-danger">Urgent</span>';
            case 'high':
                return '<span class="label label-warning">High</span>';
            case 'normal':
                return '<span class="label label-info">Normal</span>';
            case 'low':
                return '<span class="label label-success">Low</span>';
            default:
                return '<span class="label label-default">Unknown</span>';
        }
    }

    private function getStatusBadge($status) {
        switch ($status) {
            case 'pending':
                return '<span class="label label-warning">Pending</span>';
            case 'in_progress':
                return '<span class="label label-info">In Progress</span>';
            case 'completed':
                return '<span class="label label-success">Completed</span>';
            case 'cancelled':
                return '<span class="label label-danger">Cancelled</span>';
            default:
                return '<span class="label label-default">Unknown</span>';
        }
    }

    private function logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold = 60, $serviceUrl = null, $hasStoredTemplate = true) {
        try {
            $this->createVerificationLogTable();
            
            $logData = [
                'patient_id' => $patientId,
                'verification_type' => '1:1',
                'match_score' => floatval($matchScore),
                'is_match' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 1 : 0,
                'threshold_used' => intval($threshold),
                'service_url' => $serviceUrl,
                'verification_date' => date('Y-m-d H:i:s'),
                'staff_id' => $this->getStaffId(),
                'ip_address' => $this->input->ip_address(),
                'user_agent' => $this->input->user_agent(),
                'has_stored_template' => $hasStoredTemplate ? 1 : 0,
                'verification_status' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 'success' : 'failed',
                'module' => 'triage'
            ];
            
            $this->db->insert('fingerprint_verification_log', $logData);
            return $this->db->insert_id();
            
        } catch (Exception $e) {
            log_message('error', 'Failed to log triage verification: ' . $e->getMessage());
            return null;
        }
    }

    private function createVerificationLogTable() {
        if (!$this->db->table_exists('fingerprint_verification_log')) {
            $sql = "CREATE TABLE `fingerprint_verification_log` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` varchar(50) NOT NULL,
                `verification_type` enum('1:1','1:N') DEFAULT '1:1',
                `match_score` decimal(5,2) DEFAULT NULL,
                `is_match` tinyint(1) DEFAULT 0,
                `threshold_used` int(3) DEFAULT 60,
                `service_url` varchar(255) DEFAULT NULL,
                `verification_date` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `staff_id` int(11) DEFAULT NULL,
                `ip_address` varchar(45) DEFAULT NULL,
                `user_agent` text DEFAULT NULL,
                `has_stored_template` tinyint(1) DEFAULT 1,
                `verification_status` enum('success','failed','error') DEFAULT 'failed',
                `module` varchar(50) DEFAULT 'unknown',
                `notes` text DEFAULT NULL,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `verification_date` (`verification_date`),
                KEY `staff_id` (`staff_id`),
                KEY `verification_status` (`verification_status`),
                KEY `module` (`module`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
        }
    }

private function getTriageStaff() {
    try {
        // Get nurses (role_id = 9) and doctors (role_id = 3)
        $this->db->select('s.id, s.name, s.surname, s.employee_id, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'inner');
        $this->db->join('roles r', 'sr.role_id = r.id', 'inner');
        $this->db->where('s.is_active', 'yes');
        $this->db->where_in('sr.role_id', [3, 9]); // 3 = Doctor, 9 = Nurse
        $this->db->order_by('r.name, s.name');
        
        $query = $this->db->get();
        
        if ($query && $query->num_rows() > 0) {
            return $query->result_array();
        }
        
        // Alternative: Get all active staff if no specific roles found
        $this->db->select('s.id, s.name, s.surname, s.employee_id');
        $this->db->from('staff s');
        $this->db->where('s.is_active', 'yes');
        $this->db->order_by('s.name');
        
        $query = $this->db->get();
        
        if ($query && $query->num_rows() > 0) {
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting triage staff: ' . $e->getMessage());
        return [];
    }
}

    private function getStaffId() {
        try {
            if (method_exists($this->customlib, 'getStaffID')) {
                return $this->customlib->getStaffID();
            } else {
                $staffId = $this->session->userdata('admin_id');
                return $staffId ?: 1;
            }
        } catch (Exception $e) {
            log_message('error', 'Error getting staff ID: ' . $e->getMessage());
            return 1;
        }
    }

private function getAvailableDoctor() {
    try {
        // First, try to get a doctor using the staff_roles relationship
        $this->db->select('s.id');
        $this->db->from('staff s');
        $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'inner');
        $this->db->join('roles r', 'sr.role_id = r.id', 'inner');
        $this->db->where('s.is_active', 'yes');
        $this->db->where('r.name', 'Doctor'); // Role name is 'Doctor'
        $this->db->limit(1);
        
        $query = $this->db->get();
        
        if ($query && $query->num_rows() > 0) {
            return $query->row()->id;
        }
        
        // Alternative: Try with role_id = 3 (Doctor role ID from the SQL dump)
        $this->db->select('s.id');
        $this->db->from('staff s');
        $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'inner');
        $this->db->where('s.is_active', 'yes');
        $this->db->where('sr.role_id', 3); // Role ID 3 is Doctor
        $this->db->limit(1);
        
        $query = $this->db->get();
        
        if ($query && $query->num_rows() > 0) {
            return $query->row()->id;
        }
        
        // Last resort: Get any active staff member
        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('is_active', 'yes');
        $this->db->limit(1);
        
        $query = $this->db->get();
        
        if ($query && $query->num_rows() > 0) {
            return $query->row()->id;
        }
        
        return 1; // Default doctor ID
        
    } catch (Exception $e) {
        log_message('error', 'Error getting available doctor: ' . $e->getMessage());
        return 1; // Default doctor ID
    }
}

    // =============================================
    // DEBUGGING AND TESTING METHODS
    // =============================================

    public function debug() {
        echo "<h1>Enhanced Triage Controller Debug</h1>";
        echo "<p>Controller is working!</p>";
        echo "<p>Base URL: " . base_url() . "</p>";
        echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
        
        try {
            // Check patients table
            if ($this->db->table_exists('patients')) {
                echo "<h3>Patients table exists</h3>";
                
                // Get table structure
                $fields = $this->db->list_fields('patients');
                echo "<p><strong>Available columns:</strong> " . implode(', ', $fields) . "</p>";
                
                // Count patients
                $this->db->select('COUNT(*) as patient_count');
                $this->db->from('patients');
                $query = $this->db->get();
                $result = $query->row();
                echo "<p>Patients in database: " . $result->patient_count . "</p>";
                
                // Test basic query
                $this->db->select('*');
                $this->db->from('patients');
                $this->db->limit(5);
                $query = $this->db->get();
                
                if ($query->num_rows() > 0) {
                    echo "<h4>Sample patients:</h4>";
                    echo "<table border='1'>";
                    echo "<tr><th>ID</th><th>Name</th><th>Phone</th></tr>";
                    foreach ($query->result_array() as $patient) {
                        $name = $patient['patient_name'] ?? 
                               ($patient['firstname'] ?? '') . ' ' . ($patient['lastname'] ?? '') ??
                               $patient['name'] ?? 
                               'Patient #' . $patient['id'];
                        $phone = $patient['mobileno'] ?? $patient['phone'] ?? $patient['contact'] ?? 'N/A';
                        echo "<tr>";
                        echo "<td>" . $patient['id'] . "</td>";
                        echo "<td>" . $name . "</td>";
                        echo "<td>" . $phone . "</td>";
                        echo "</tr>";
                    }
                    echo "</table>";
                } else {
                    echo "<p>No patients found in database!</p>";
                }
            } else {
                echo "<p style='color: red;'>Patients table does not exist!</p>";
            }
            
            // Check other tables
            $tables = ['triage', 'patient_routing', 'laboratory', 'phlebotomy'];
            foreach ($tables as $table) {
                if ($this->db->table_exists($table)) {
                    $this->db->select('COUNT(*) as count');
                    $this->db->from($table);
                    $query = $this->db->get();
                    $result = $query->row();
                    echo "<p>{$table} table: {$result->count} records</p>";
                } else {
                    echo "<p style='color: orange;'>{$table} table does not exist</p>";
                }
            }
            
            // Test the getAllPatients method
            echo "<h3>Testing getAllPatients method:</h3>";
            echo "<p><a href='" . base_url('admin/triage/getAllPatients') . "' target='_blank'>Click here to test getAllPatients JSON response</a></p>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
            echo "<p><strong>Last Query:</strong> " . $this->db->last_query() . "</p>";
        }
    }

    public function addPatient() {
        header('Content-Type: application/json');
        
        try {
            $patientName = $this->input->post('patientName');
            $patientAge = $this->input->post('patientAge');
            $patientGender = $this->input->post('patientGender');
            $patientPhone = $this->input->post('patientPhone');
            
            if (!$patientName || !$patientAge || !$patientGender || !$patientPhone) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'All fields are required'
                ]);
                return;
            }
            
            // Prepare patient data based on table structure
            $fields = $this->db->list_fields('patients');
            $patientData = [];
            
            // Map fields based on what exists in the table
            if (in_array('patient_name', $fields)) {
                $patientData['patient_name'] = $patientName;
            } elseif (in_array('firstname', $fields)) {
                $nameParts = explode(' ', $patientName, 2);
                $patientData['firstname'] = $nameParts[0];
                if (isset($nameParts[1]) && in_array('lastname', $fields)) {
                    $patientData['lastname'] = $nameParts[1];
                }
            } elseif (in_array('name', $fields)) {
                $patientData['name'] = $patientName;
            }
            
            if (in_array('age', $fields)) {
                $patientData['age'] = $patientAge;
            }
            
            if (in_array('gender', $fields)) {
                $patientData['gender'] = $patientGender;
            }
            
            if (in_array('mobileno', $fields)) {
                $patientData['mobileno'] = $patientPhone;
            } elseif (in_array('phone', $fields)) {
                $patientData['phone'] = $patientPhone;
            } elseif (in_array('contact', $fields)) {
                $patientData['contact'] = $patientPhone;
            }
            
            if (in_array('is_active', $fields)) {
                $patientData['is_active'] = 'yes';
            }
            
            if (in_array('created_at', $fields)) {
                $patientData['created_at'] = date('Y-m-d H:i:s');
            }
            
            // Insert patient
            $result = $this->db->insert('patients', $patientData);
            $patientId = $this->db->insert_id();
            
            if ($result && $patientId) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Patient added successfully',
                    'patient_id' => $patientId
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to add patient'
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
 * Generate case reference number
 */
private function generateCaseReference() {
    $prefix = 'TR' . date('Y');
    $this->db->select('MAX(CAST(SUBSTRING(case_reference_id, 7) AS UNSIGNED)) as max_id');
    $this->db->from('case_references');
    $this->db->where('case_reference_id LIKE', $prefix . '%');
    $query = $this->db->get();
    
    if ($query && $query->num_rows() > 0) {
        $result = $query->row();
        $next_id = ($result->max_id ?? 0) + 1;
    } else {
        $next_id = 1;
    }
    
    return $prefix . str_pad($next_id, 4, '0', STR_PAD_LEFT);
}

/**
 * Debug method to check database structure
 */
public function checkDatabase() {
    echo "<h2>Database Structure Check</h2>";
    
    // Check staff table structure
    if ($this->db->table_exists('staff')) {
        echo "<h3>Staff Table Structure:</h3>";
        $fields = $this->db->list_fields('staff');
        echo "<p>Columns: " . implode(', ', $fields) . "</p>";
    }
    
    // Check roles relationship
    if ($this->db->table_exists('staff_roles')) {
        echo "<h3>Staff Roles Relationship:</h3>";
        $this->db->select('s.id, s.name, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'inner');
        $this->db->join('roles r', 'sr.role_id = r.id', 'inner');
        $this->db->where('s.is_active', 'yes');
        $this->db->limit(5);
        $query = $this->db->get();
        
        if ($query && $query->num_rows() > 0) {
            echo "<table border='1'>";
            echo "<tr><th>Staff ID</th><th>Staff Name</th><th>Role</th></tr>";
            foreach ($query->result_array() as $row) {
                echo "<tr><td>{$row['id']}</td><td>{$row['name']}</td><td>{$row['role_name']}</td></tr>";
            }
            echo "</table>";
        } else {
            echo "<p>No staff with roles found</p>";
        }
    }
    
    // Test the fixed methods
    echo "<h3>Test Fixed Methods:</h3>";
    echo "<p>Available Doctor ID: " . $this->getAvailableDoctor() . "</p>";
    
    $staff = $this->getTriageStaff();
    echo "<p>Triage Staff Count: " . count($staff) . "</p>";
    
    if (!empty($staff)) {
        echo "<h4>Triage Staff:</h4>";
        echo "<table border='1'>";
        echo "<tr><th>ID</th><th>Name</th><th>Role</th></tr>";
        foreach ($staff as $member) {
            echo "<tr>";
            echo "<td>{$member['id']}</td>";
            echo "<td>{$member['name']} {$member['surname']}</td>";
            echo "<td>" . ($member['role_name'] ?? 'N/A') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
}



}

?>