<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Specialized_services extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('specialized_services_model');
        $this->load->model('staff_model');
        $this->load->model('setting_model');
        $this->load->library('form_validation');
        $this->load->library('datatables');
        $this->load->helper('customfield_helper');
        $this->load->helper('custom');
        
        // Get system currency
        $this->setSystemCurrency();
    }

    private function setSystemCurrency() {
        try {
            $settings = $this->setting_model->get();
            $this->system_currency = isset($settings[0]['currency_symbol']) ? $settings[0]['currency_symbol'] : 'Ksh.';
        } catch (Exception $e) {
            $this->system_currency = 'Ksh.';
        }
    }

    // =============================================
    // DASHBOARD
    // =============================================

    public function index() {
        redirect('admin/specialized_services/dashboard');
    }

    public function dashboard() {
        if (!$this->rbac->hasPrivilege('specialized_services', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'services');
        $this->session->set_userdata('sub_menu', 'specialized_services/dashboard');

        $data['title'] = 'Specialized Services Dashboard';
        $data['currency_symbol'] = $this->system_currency;
        
        try {
            $data['stats'] = $this->specialized_services_model->getSetupStats();
        } catch (Exception $e) {
            $data['stats'] = array(
                'total_categories' => 0,
                'total_services' => 0,
                'dental_services' => 0,
                'optical_services' => 0,
                'total_bookings' => 0,
                'services_by_category' => array()
            );
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/specialized_services/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // CATEGORIES MANAGEMENT
    // =============================================

    public function categories() {
        if (!$this->rbac->hasPrivilege('specialized_services', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'services');
        $this->session->set_userdata('sub_menu', 'specialized_services/categories');

        $data['title'] = 'Specialized Service Categories';
        $data['categories'] = $this->specialized_services_model->getServiceCategories();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/specialized_services/categories', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // SERVICES MANAGEMENT
    // =============================================

    public function services() {
        if (!$this->rbac->hasPrivilege('specialized_services', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'services');
        $this->session->set_userdata('sub_menu', 'specialized_services/services');

        $data['title'] = 'Specialized Services Management';
        $data['categories'] = $this->specialized_services_model->getServiceCategories();
        $data['service_types'] = $this->getServiceTypes();
        $data['category_types'] = $this->getCategoryTypes();
        $data['currency_symbol'] = $this->system_currency;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/specialized_services/services', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get services data for DataTables
     */
    public function get_services_datatable() {
        header('Content-Type: application/json; charset=utf-8');
        
        try {
            if (!$this->db->table_exists('specialized_services')) {
                echo json_encode([
                    'draw' => intval($this->input->post('draw') ?: 1),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Specialized services table does not exist'
                ]);
                return;
            }

            $draw = intval($this->input->post('draw') ?: 1);
            $start = intval($this->input->post('start') ?: 0);
            $length = intval($this->input->post('length') ?: 10);
            
            $search_value = '';
            $search_array = $this->input->post('search');
            if ($search_array && isset($search_array['value'])) {
                $search_value = $search_array['value'];
            }

            $order_array = $this->input->post('order');
            $order_column_index = 0;
            $order_dir = 'asc';
            
            if ($order_array && isset($order_array[0])) {
                $order_column_index = intval($order_array[0]['column'] ?? 0);
                $order_dir = $order_array[0]['dir'] ?? 'asc';
            }

            $columns = array(
                0 => 'specialized_services.service_name',
                1 => 'specialized_services.service_code',
                2 => 'specialized_service_categories.category_name',
                3 => 'specialized_services.service_type',
                4 => 'specialized_services.cash_price',
                5 => 'specialized_services.duration_minutes',
                6 => 'specialized_services.requires_specialist',
                7 => 'specialized_services.is_active'
            );

            $order_column = $columns[$order_column_index] ?? 'specialized_services.service_name';

            $this->db->select('
                specialized_services.id,
                specialized_services.service_name,
                specialized_services.service_code,
                specialized_services.service_type,
                specialized_services.cash_price,
                specialized_services.insurance_price,
                specialized_services.duration_minutes,
                specialized_services.requires_specialist,
                specialized_services.is_active,
                specialized_service_categories.category_name,
                specialized_service_categories.category_type
            ');
            
            $this->db->from('specialized_services');
            $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id', 'left');

            if (!empty($search_value)) {
                $this->db->group_start();
                $this->db->like('specialized_services.service_name', $search_value);
                $this->db->or_like('specialized_services.service_code', $search_value);
                $this->db->or_like('specialized_service_categories.category_name', $search_value);
                $this->db->or_like('specialized_services.service_type', $search_value);
                $this->db->group_end();
            }

            $count_sql = $this->db->get_compiled_select();
            $total_records = $this->db->query("SELECT COUNT(*) as count FROM (" . $count_sql . ") as temp")->row()->count;

            $this->db->reset_query();
            $this->db->select('
                specialized_services.id,
                specialized_services.service_name,
                specialized_services.service_code,
                specialized_services.service_type,
                specialized_services.cash_price,
                specialized_services.insurance_price,
                specialized_services.duration_minutes,
                specialized_services.requires_specialist,
                specialized_services.is_active,
                specialized_service_categories.category_name,
                specialized_service_categories.category_type
            ');
            
            $this->db->from('specialized_services');
            $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id', 'left');

            if (!empty($search_value)) {
                $this->db->group_start();
                $this->db->like('specialized_services.service_name', $search_value);
                $this->db->or_like('specialized_services.service_code', $search_value);
                $this->db->or_like('specialized_service_categories.category_name', $search_value);
                $this->db->or_like('specialized_services.service_type', $search_value);
                $this->db->group_end();
            }

            $this->db->order_by($order_column, $order_dir);
            if ($length != -1) {
                $this->db->limit($length, $start);
            }

            $query = $this->db->get();
            
            if (!$query) {
                throw new Exception('Database query failed');
            }

            $data = array();
            foreach ($query->result() as $row) {
                $sub_array = array();
                
                // Service name with category icon
                $category_icon = $this->getCategoryIcon($row->category_type);
                $sub_array[] = '<i class="fa ' . $category_icon . '" style="color: ' . $this->getCategoryColor($row->category_type) . ';"></i> ' . htmlspecialchars($row->service_name);
                
                // Service code
                $sub_array[] = htmlspecialchars($row->service_code ?: 'N/A');
                
                // Category
                $sub_array[] = '<span class="badge" style="background-color: ' . $this->getCategoryColor($row->category_type) . ';">' . htmlspecialchars($row->category_name) . '</span>';
                
                // Service type
                $sub_array[] = '<span class="label label-info">' . ucfirst(htmlspecialchars($row->service_type)) . '</span>';
                
                // Price
                $price_display = $this->system_currency . ' ' . number_format($row->cash_price, 2);
                if ($row->insurance_price && $row->insurance_price != $row->cash_price) {
                    $price_display .= '<br><small class="text-muted">Ins: ' . $this->system_currency . ' ' . number_format($row->insurance_price, 2) . '</small>';
                }
                $sub_array[] = $price_display;
                
                // Duration
                $sub_array[] = $row->duration_minutes ? $row->duration_minutes . ' min' : 'N/A';
                
                // Requires specialist
                if ($row->requires_specialist == 1) {
                    $specialist_badge = '<span class="badge badge-warning" style="background-color: #f39c12; color: white; font-size: 10px;">
                                        <i class="fa fa-user-md"></i> Specialist
                                      </span>';
                } else {
                    $specialist_badge = '<span class="badge badge-secondary" style="background-color: #95a5a6; color: white; font-size: 10px;">
                                        <i class="fa fa-user"></i> General
                                      </span>';
                }
                $sub_array[] = $specialist_badge;
                
                // Status
                if ($row->is_active == 1) {
                    $status = '<span class="badge" style="background-color: #28a745; color: white;">Active</span>';
                } else {
                    $status = '<span class="badge" style="background-color: #dc3545; color: white;">Inactive</span>';
                }
                $sub_array[] = $status;
                
                // Actions
                $actions = '';
                
                $actions .= '<button type="button" class="btn btn-xs btn-info" onclick="viewService(' . $row->id . ')" title="View" style="margin-right: 2px;">
                                <i class="fa fa-eye"></i>
                             </button>';
                
                $actions .= '<button type="button" class="btn btn-xs btn-primary" onclick="editService(' . $row->id . ')" title="Edit" style="margin-right: 2px;">
                                <i class="fa fa-edit"></i>
                             </button>';
                
                // Specialist toggle button
                $specialist_action = $row->requires_specialist == 1 ? 'Remove Specialist Requirement' : 'Require Specialist';
                $specialist_icon = $row->requires_specialist == 1 ? 'fa-user' : 'fa-user-md';
                $specialist_class = $row->requires_specialist == 1 ? 'btn-warning' : 'btn-info';
                
                $actions .= '<button type="button" class="btn btn-xs ' . $specialist_class . '" 
                                onclick="toggleSpecialist(' . $row->id . ', ' . $row->requires_specialist . ')" 
                                title="' . $specialist_action . '" style="margin-right: 2px;">
                                <i class="fa ' . $specialist_icon . '"></i>
                             </button>';
                
                $actions .= '<button type="button" class="btn btn-xs btn-danger" onclick="deleteService(' . $row->id . ')" title="Delete">
                                <i class="fa fa-trash"></i>
                             </button>';
                
                $sub_array[] = $actions;
                $data[] = $sub_array;
            }

            $output = array(
                "draw" => $draw,
                "recordsTotal" => $total_records,
                "recordsFiltered" => $total_records,
                "data" => $data
            );

            echo json_encode($output);

        } catch (Exception $e) {
            log_message('error', 'DataTables error in get_services_datatable: ' . $e->getMessage());
            
            echo json_encode([
                'draw' => intval($this->input->post('draw') ?: 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Add new specialized service
     */
    public function add_service() {
        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('service_name', 'Service Name', 'required|trim');
            $this->form_validation->set_rules('category_id', 'Category', 'required');
            $this->form_validation->set_rules('cash_price', 'Cash Price', 'required|numeric|greater_than[0]');
            $this->form_validation->set_rules('service_type', 'Service Type', 'required');
            
            if ($this->form_validation->run() === FALSE) {
                echo json_encode([
                    'status' => 'error',
                    'error' => $this->form_validation->error_array(),
                    'message' => 'Please fix the validation errors'
                ]);
                return;
            }

            $data = array(
                'service_name' => $this->input->post('service_name'),
                'service_code' => $this->input->post('service_code'),
                'category_id' => $this->input->post('category_id'),
                'service_type' => $this->input->post('service_type'),
                'description' => $this->input->post('description'),
                'cash_price' => $this->input->post('cash_price'),
                'insurance_price' => $this->input->post('insurance_price') ?: null,
                'duration_minutes' => $this->input->post('duration_minutes') ?: 30,
                'requires_specialist' => $this->input->post('requires_specialist') ? 1 : 0,
                'requires_equipment' => $this->input->post('requires_equipment') ? 1 : 0,
                'equipment_list' => $this->input->post('equipment_list'),
                'pre_procedure_requirements' => $this->input->post('pre_procedure_requirements'),
                'post_procedure_instructions' => $this->input->post('post_procedure_instructions'),
                'complications_risk_level' => $this->input->post('complications_risk_level') ?: 'low',
                'age_restrictions' => $this->input->post('age_restrictions'),
                'contraindications' => $this->input->post('contraindications'),
                'billing_code' => $this->input->post('billing_code'),
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            );

            // Check for duplicate service name
            $this->db->where('service_name', $data['service_name']);
            $this->db->where('category_id', $data['category_id']);
            $existing = $this->db->get('specialized_services')->row();
            
            if ($existing) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'A service with this name already exists in this category'
                ]);
                return;
            }

            if ($this->db->insert('specialized_services', $data)) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Specialized service added successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to add specialized service'
                ]);
            }
        }
    }

    /**
     * Toggle specialist requirement
     */
    public function toggle_specialist_requirement() {
        if ($this->input->method() === 'post') {
            $service_id = $this->input->post('service_id');
            $requires_specialist = $this->input->post('requires_specialist');
            
            $data = array(
                'requires_specialist' => $requires_specialist ? 1 : 0,
                'updated_at' => date('Y-m-d H:i:s')
            );

            $this->db->where('id', $service_id);
            
            if ($this->db->update('specialized_services', $data)) {
                $action = $requires_specialist ? 'enabled' : 'disabled';
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Specialist requirement ' . $action . ' successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update specialist requirement'
                ]);
            }
        }
    }

    /**
     * Edit service
     */
    public function edit_service($id) {
        if ($this->input->method() === 'get') {
            $data['service'] = $this->db->get_where('specialized_services', array('id' => $id))->row();
            $data['categories'] = $this->db->get_where('specialized_service_categories', array('is_active' => 1))->result_array();
            $data['service_types'] = $this->getServiceTypes();
            $data['currency_symbol'] = $this->system_currency;
            
            $this->load->view('admin/specialized_services/edit_service_modal', $data);
            
        } elseif ($this->input->method() === 'post') {
            $this->form_validation->set_rules('service_name', 'Service Name', 'required|trim');
            $this->form_validation->set_rules('category_id', 'Category', 'required');
            $this->form_validation->set_rules('cash_price', 'Cash Price', 'required|numeric|greater_than[0]');
            $this->form_validation->set_rules('service_type', 'Service Type', 'required');
            
            if ($this->form_validation->run() === FALSE) {
                echo json_encode([
                    'status' => 'error',
                    'error' => $this->form_validation->error_array(),
                    'message' => 'Please fix the validation errors'
                ]);
                return;
            }

            $data = array(
                'service_name' => $this->input->post('service_name'),
                'service_code' => $this->input->post('service_code'),
                'category_id' => $this->input->post('category_id'),
                'service_type' => $this->input->post('service_type'),
                'description' => $this->input->post('description'),
                'cash_price' => $this->input->post('cash_price'),
                'insurance_price' => $this->input->post('insurance_price') ?: null,
                'duration_minutes' => $this->input->post('duration_minutes') ?: 30,
                'requires_specialist' => $this->input->post('requires_specialist') ? 1 : 0,
                'requires_equipment' => $this->input->post('requires_equipment') ? 1 : 0,
                'equipment_list' => $this->input->post('equipment_list'),
                'pre_procedure_requirements' => $this->input->post('pre_procedure_requirements'),
                'post_procedure_instructions' => $this->input->post('post_procedure_instructions'),
                'complications_risk_level' => $this->input->post('complications_risk_level'),
                'age_restrictions' => $this->input->post('age_restrictions'),
                'contraindications' => $this->input->post('contraindications'),
                'billing_code' => $this->input->post('billing_code'),
                'updated_at' => date('Y-m-d H:i:s')
            );

            // Check for duplicate service name (excluding current service)
            $this->db->where('service_name', $data['service_name']);
            $this->db->where('category_id', $data['category_id']);
            $this->db->where('id !=', $id);
            $existing = $this->db->get('specialized_services')->row();
            
            if ($existing) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'A service with this name already exists in this category'
                ]);
                return;
            }

            $this->db->where('id', $id);
            
            if ($this->db->update('specialized_services', $data)) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Specialized service updated successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update specialized service'
                ]);
            }
        }
    }

    /**
     * View service details
     */
    public function view_service($id) {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id', 'left');
        $this->db->where('specialized_services.id', $id);
        
        $data['service'] = $this->db->get()->row();
        $data['currency_symbol'] = $this->system_currency;
        
        $this->load->view('admin/specialized_services/view_service_modal', $data);
    }

    /**
     * Delete service
     */
    public function delete_service($id) {
        if ($this->input->method() === 'post') {
            // Check if service is being used in any bookings/orders
            // You might want to add this check based on your booking system
            
            $this->db->where('id', $id);
            
            if ($this->db->delete('specialized_services')) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Specialized service deleted successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to delete specialized service'
                ]);
            }
        }
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Get service types
     */
    private function getServiceTypes() {
        return array(
            'consultation' => 'Consultation',
            'procedure' => 'Procedure',
            'treatment' => 'Treatment',
            'diagnostic' => 'Diagnostic',
            'preventive' => 'Preventive',
            'cosmetic' => 'Cosmetic'
        );
    }

    /**
     * Get category types
     */
    private function getCategoryTypes() {
        return array(
            'dental' => 'Dental',
            'optical' => 'Optical',
            'physiotherapy' => 'Physiotherapy',
            'orthopedic' => 'Orthopedic',
            'cardiology' => 'Cardiology',
            'neurology' => 'Neurology',
            'other' => 'Other'
        );
    }

    /**
     * Get category icon based on type
     */
    private function getCategoryIcon($category_type) {
        $icons = array(
            'dental' => 'fa-tooth',
            'optical' => 'fa-eye',
            'physiotherapy' => 'fa-heartbeat',
            'orthopedic' => 'fa-wheelchair',
            'cardiology' => 'fa-heart',
            'neurology' => 'fa-brain',
            'other' => 'fa-stethoscope'
        );
        
        return isset($icons[$category_type]) ? $icons[$category_type] : 'fa-stethoscope';
    }

    /**
     * Get category color based on type
     */
    private function getCategoryColor($category_type) {
        $colors = array(
            'dental' => '#2E86AB',
            'optical' => '#1B998B',
            'physiotherapy' => '#FF6B35',
            'orthopedic' => '#A23B72',
            'cardiology' => '#F18F01',
            'neurology' => '#C73E1D',
            'other' => '#6C757D'
        );
        
        return isset($colors[$category_type]) ? $colors[$category_type] : '#6C757D';
    }

    // =============================================
    // SYSTEM SETUP AND DEBUGGING
    // =============================================

    /**
     * Setup specialized services system
     */
    public function setup_system() {
        echo "<h2>Setting up Specialized Services System</h2>";
        echo "<style>body{font-family:Arial;margin:20px;} table{border-collapse:collapse;width:100%;} th,td{border:1px solid #ddd;padding:8px;} th{background-color:#f2f2f2;}</style>";
        
        try {
            // Create tables
            $this->createSpecializedServicesTables();
            echo "<p style='color:green;'>✓ Database tables created successfully</p>";
            
            // Insert default data
            $this->insertDefaultData();
            echo "<p style='color:green;'>✓ Default data inserted successfully</p>";
            
            echo "<h3 style='color:green;'>✅ Specialized Services System Ready!</h3>";
            echo "<p><a href='" . base_url('admin/specialized_services/services') . "' class='btn btn-primary'>Go to Services Management</a></p>";
            
        } catch (Exception $e) {
            echo "<p style='color:red;'>✗ Error: " . $e->getMessage() . "</p>";
        }
    }

    /**
     * Create specialized services tables
     */
    private function createSpecializedServicesTables() {
        // Read and execute the SQL schema
        $sql_file = APPPATH . 'sql/specialized_services_schema.sql';
        
        // If SQL file doesn't exist, create tables directly
        if (!file_exists($sql_file)) {
            $this->createTablesDirectly();
        } else {
            $sql = file_get_contents($sql_file);
            $this->db->query($sql);
        }
    }

    /**
     * Create tables directly with SQL
     */
    private function createTablesDirectly() {
        // Create categories table
        $sql = "CREATE TABLE IF NOT EXISTS `specialized_service_categories` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `category_name` varchar(100) NOT NULL,
            `category_type` enum('dental','optical','physiotherapy','orthopedic','cardiology','neurology','other') NOT NULL,
            `description` text,
            `icon_class` varchar(50) DEFAULT NULL,
            `color_code` varchar(10) DEFAULT NULL,
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `category_name_type` (`category_name`, `category_type`),
            KEY `idx_category_type` (`category_type`),
            KEY `idx_is_active` (`is_active`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $this->db->query($sql);

        // Create services table
        $sql = "CREATE TABLE IF NOT EXISTS `specialized_services` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `service_name` varchar(255) NOT NULL,
            `service_code` varchar(50) DEFAULT NULL,
            `category_id` int(11) NOT NULL,
            `service_type` enum('consultation','procedure','treatment','diagnostic','preventive','cosmetic') DEFAULT 'procedure',
            `description` text,
            `cash_price` decimal(10,2) NOT NULL DEFAULT 0.00,
            `insurance_price` decimal(10,2) DEFAULT NULL,
            `duration_minutes` int DEFAULT 30,
            `requires_specialist` tinyint(1) DEFAULT 1,
            `requires_equipment` tinyint(1) DEFAULT 0,
            `equipment_list` text,
            `pre_procedure_requirements` text,
            `post_procedure_instructions` text,
            `complications_risk_level` enum('low','medium','high') DEFAULT 'low',
            `age_restrictions` varchar(100) DEFAULT NULL,
            `contraindications` text,
            `billing_code` varchar(50) DEFAULT NULL,
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `service_name_category` (`service_name`, `category_id`),
            KEY `idx_category_id` (`category_id`),
            KEY `idx_service_type` (`service_type`),
            KEY `idx_is_active` (`is_active`),
            KEY `idx_requires_specialist` (`requires_specialist`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $this->db->query($sql);
    }

    /**
     * Insert default data
     */
    private function insertDefaultData() {
        // Insert categories if they don't exist
        $categories = [
            ['category_name' => 'General Dentistry', 'category_type' => 'dental', 'description' => 'General dental procedures and consultations', 'icon_class' => 'fa-tooth', 'color_code' => '#2E86AB'],
            ['category_name' => 'Oral Surgery', 'category_type' => 'dental', 'description' => 'Surgical dental procedures', 'icon_class' => 'fa-user-md', 'color_code' => '#A23B72'],
            ['category_name' => 'Orthodontics', 'category_type' => 'dental', 'description' => 'Braces and teeth alignment', 'icon_class' => 'fa-compress', 'color_code' => '#F18F01'],
            ['category_name' => 'Prosthodontics', 'category_type' => 'dental', 'description' => 'Dentures, crowns, and bridges', 'icon_class' => 'fa-puzzle-piece', 'color_code' => '#C73E1D'],
            ['category_name' => 'Eye Examination', 'category_type' => 'optical', 'description' => 'Comprehensive eye examinations', 'icon_class' => 'fa-eye', 'color_code' => '#1B998B'],
            ['category_name' => 'Vision Correction', 'category_type' => 'optical', 'description' => 'Glasses and contact lens services', 'icon_class' => 'fa-glasses', 'color_code' => '#2D5016'],
            ['category_name' => 'Eye Surgery', 'category_type' => 'optical', 'description' => 'Surgical eye procedures', 'icon_class' => 'fa-eye-slash', 'color_code' => '#FF6B35'],
            ['category_name' => 'Contact Lens Services', 'category_type' => 'optical', 'description' => 'Contact lens fitting and care', 'icon_class' => 'fa-circle-o', 'color_code' => '#004E89']
        ];

        foreach ($categories as $category) {
            $this->db->where('category_name', $category['category_name']);
            $this->db->where('category_type', $category['category_type']);
            $existing = $this->db->get('specialized_service_categories')->row();
            
            if (!$existing) {
                $this->db->insert('specialized_service_categories', $category);
            }
        }
    }

    /**
     * Test endpoint
     */
    public function test_endpoint() {
        echo json_encode([
            'status' => 'success',
            'message' => 'Specialized services controller is working',
            'timestamp' => date('Y-m-d H:i:s'),
            'base_url' => base_url(),
            'request_method' => $this->input->method(),
            'tables_exist' => [
                'specialized_services' => $this->db->table_exists('specialized_services'),
                'specialized_service_categories' => $this->db->table_exists('specialized_service_categories')
            ]
        ]);
    }

    /**
     * Debug database structure
     */
    public function debug_database() {
        echo "<h2>Specialized Services Database Debug</h2>";
        echo "<style>body{font-family:Arial;margin:20px;} table{border-collapse:collapse;width:100%;} th,td{border:1px solid #ddd;padding:8px;} th{background-color:#f2f2f2;}</style>";
        
        try {
            $tables = ['specialized_services', 'specialized_service_categories'];
            
            foreach ($tables as $table) {
                echo "<h3>$table Table</h3>";
                if ($this->db->table_exists($table)) {
                    echo "<p style='color:green;'>✓ $table table exists</p>";
                    
                    $fields = $this->db->field_data($table);
                    echo "<h4>Columns:</h4><ul>";
                    foreach ($fields as $field) {
                        echo "<li>" . $field->name . " (" . $field->type . ")</li>";
                    }
                    echo "</ul>";
                    
                    $count = $this->db->count_all($table);
                    echo "<p>Total records: $count</p>";
                } else {
                    echo "<p style='color:red;'>✗ $table table does not exist</p>";
                }
            }
            
            echo "<p><a href='" . base_url('admin/specialized_services/setup_system') . "'>Setup System</a></p>";
            
        } catch (Exception $e) {
            echo "<p style='color:red;'>Error: " . $e->getMessage() . "</p>";
        }
    }
}

?>