<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Reports Dashboard Controller
 * 
 * Handles all reporting functionality with role-based access control
 * Location: application/controllers/admin/Reports_dashboard.php
 */
class Reports_dashboard extends Admin_Controller {
    
    public function __construct() {
        parent::__construct();
        
        // Load required models
        $this->load->model('reports_model');
        $this->load->model('patient_model');
        $this->load->model('transaction_model');
        $this->load->model('staff_model');
        
        // Load libraries
        $this->load->library('customlib');
        $this->load->library('datatables');
        
        // Check base reports permission
        if (!$this->rbac->hasPrivilege('reports', 'can_view')) {
            access_denied();
        }
    }
    
    /**
     * Main dashboard view
     */
    public function index() {
        $this->session->set_userdata('top_menu', 'Reports');
        $this->session->set_userdata('sub_menu', 'reports/dashboard');
        
        // Get user permissions
        $data['permissions'] = $this->getUserReportPermissions();
        
        // Get available report types based on permissions
        $data['available_reports'] = $this->getAvailableReports($data['permissions']);
        
        // Get date range for default view
        $data['date_from'] = date('Y-m-01');
        $data['date_to'] = date('Y-m-d');
        
        // Get quick stats if user has permission
        if (in_array('executive_dashboard', $data['permissions'])) {
            $data['quick_stats'] = $this->getQuickStats();
        }
        
        // Load the view
        $this->load->view('layout/header', $data);
        $this->load->view('admin/reports/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }
    
    /**
     * Get revenue analysis data
     */
    public function revenue_analysis() {
        // Check permission
        if (!$this->rbac->hasPrivilege('revenue_reports', 'can_view')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $date_from = $this->input->post('date_from') ?: date('Y-m-01');
        $date_to = $this->input->post('date_to') ?: date('Y-m-d');
        $group_by = $this->input->post('group_by') ?: 'day';
        
        try {
            $data = array();
            
            // Get total revenue
            $data['total_revenue'] = $this->reports_model->getTotalRevenue($date_from, $date_to);
            
            // Get revenue by payment method
            $data['payment_methods'] = $this->reports_model->getRevenueByPaymentMethod($date_from, $date_to);
            
            // Get daily/weekly/monthly trend
            $data['revenue_trend'] = $this->reports_model->getRevenueTrend($date_from, $date_to, $group_by);
            
            // Get comparison with previous period
            $data['comparison'] = $this->reports_model->getRevenueComparison($date_from, $date_to);
            
            // Get department-wise revenue
            $data['department_revenue'] = $this->reports_model->getDepartmentRevenue($date_from, $date_to);
            
            // Log report access
            $this->logReportAccess('revenue_analysis', array(
                'date_from' => $date_from,
                'date_to' => $date_to,
                'group_by' => $group_by
            ));
            
            echo json_encode(['status' => 1, 'data' => $data]);
            
        } catch (Exception $e) {
            log_message('error', 'Revenue analysis error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error generating report']);
        }
    }
    
    /**
     * Get patient visit statistics
     */
    public function patient_visits() {
        // Check permission
        if (!$this->rbac->hasPrivilege('patient_visit_reports', 'can_view')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $date_from = $this->input->post('date_from') ?: date('Y-m-01');
        $date_to = $this->input->post('date_to') ?: date('Y-m-d');
        $visit_type = $this->input->post('visit_type') ?: 'all';
        
        try {
            $data = array();
            
            // Get total visits
            $data['total_visits'] = $this->reports_model->getTotalVisits($date_from, $date_to);
            
            // Get walk-in vs registered
            $data['visit_breakdown'] = $this->reports_model->getVisitBreakdown($date_from, $date_to);
            
            // Get new vs returning patients
            $data['patient_type'] = $this->reports_model->getPatientTypeBreakdown($date_from, $date_to);
            
            // Get hourly distribution
            $data['hourly_distribution'] = $this->reports_model->getHourlyVisitDistribution($date_from, $date_to);
            
            // Get department-wise visits
            $data['department_visits'] = $this->reports_model->getDepartmentVisits($date_from, $date_to);
            
            // Get average wait time
            $data['wait_times'] = $this->reports_model->getAverageWaitTimes($date_from, $date_to);
            
            echo json_encode(['status' => 1, 'data' => $data]);
            
        } catch (Exception $e) {
            log_message('error', 'Patient visits error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error generating report']);
        }
    }
    
    /**
     * Get insurance reports
     */
    public function insurance_reports() {
        // Check permission
        if (!$this->rbac->hasPrivilege('insurance_reports', 'can_view')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $date_from = $this->input->post('date_from') ?: date('Y-m-01');
        $date_to = $this->input->post('date_to') ?: date('Y-m-d');
        $insurance_id = $this->input->post('insurance_id');
        
        try {
            $data = array();
            
            // Get insurance claims summary
            $data['claims_summary'] = $this->reports_model->getInsuranceClaimsSummary($date_from, $date_to, $insurance_id);
            
            // Get pending claims
            $data['pending_claims'] = $this->reports_model->getPendingInsuranceClaims($insurance_id);
            
            // Get approved vs rejected claims
            $data['claim_status'] = $this->reports_model->getClaimStatusBreakdown($date_from, $date_to, $insurance_id);
            
            // Get insurance company wise breakdown
            $data['company_breakdown'] = $this->reports_model->getInsuranceCompanyBreakdown($date_from, $date_to);
            
            echo json_encode(['status' => 1, 'data' => $data]);
            
        } catch (Exception $e) {
            log_message('error', 'Insurance reports error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error generating report']);
        }
    }
    
    /**
     * Get cash payment reports
     */
    public function cash_payments() {
        // Check permission
        if (!$this->rbac->hasPrivilege('cash_payment_reports', 'can_view')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $date_from = $this->input->post('date_from') ?: date('Y-m-01');
        $date_to = $this->input->post('date_to') ?: date('Y-m-d');
        $cashier_id = $this->input->post('cashier_id');
        
        try {
            $data = array();
            
            // Get total cash collected
            $data['total_cash'] = $this->reports_model->getTotalCashCollected($date_from, $date_to, $cashier_id);
            
            // Get cashier-wise collection
            $data['cashier_collection'] = $this->reports_model->getCashierWiseCollection($date_from, $date_to);
            
            // Get denomination breakdown if available
            $data['denomination_breakdown'] = $this->reports_model->getDenominationBreakdown($date_from, $date_to);
            
            // Get cash flow by hour
            $data['hourly_cash_flow'] = $this->reports_model->getHourlyCashFlow($date_from, $date_to);
            
            echo json_encode(['status' => 1, 'data' => $data]);
            
        } catch (Exception $e) {
            log_message('error', 'Cash payment reports error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error generating report']);
        }
    }
    
    /**
     * Get department revenue reports
     */
    public function department_revenue() {
        // Check permission
        if (!$this->rbac->hasPrivilege('department_revenue_reports', 'can_view')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $date_from = $this->input->post('date_from') ?: date('Y-m-01');
        $date_to = $this->input->post('date_to') ?: date('Y-m-d');
        $department = $this->input->post('department') ?: 'all';
        
        try {
            $data = array();
            
            // Get department-wise detailed revenue
            $departments = ['OPD', 'IPD', 'Pharmacy', 'Pathology', 'Radiology', 'Blood Bank', 'Ambulance'];
            
            foreach ($departments as $dept) {
                if ($department == 'all' || $department == $dept) {
                    $data[$dept] = $this->reports_model->getDepartmentDetailedRevenue($dept, $date_from, $date_to);
                }
            }
            
            // Get comparative analysis
            $data['comparison'] = $this->reports_model->getDepartmentComparison($date_from, $date_to);
            
            // Get top services per department
            $data['top_services'] = $this->reports_model->getTopServicesByDepartment($department, $date_from, $date_to);
            
            echo json_encode(['status' => 1, 'data' => $data]);
            
        } catch (Exception $e) {
            log_message('error', 'Department revenue error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error generating report']);
        }
    }
    
    /**
     * Get outstanding payment reports
     */
    public function outstanding_payments() {
        // Check permission
        if (!$this->rbac->hasPrivilege('outstanding_payment_reports', 'can_view')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $aging_days = $this->input->post('aging_days') ?: 30;
        $department = $this->input->post('department') ?: 'all';
        
        try {
            $data = array();
            
            // Get total outstanding amount
            $data['total_outstanding'] = $this->reports_model->getTotalOutstanding($department);
            
            // Get aging analysis
            $data['aging_analysis'] = $this->reports_model->getAgingAnalysis($aging_days, $department);
            
            // Get patient-wise outstanding
            $data['patient_outstanding'] = $this->reports_model->getPatientOutstanding($department);
            
            // Get insurance outstanding
            $data['insurance_outstanding'] = $this->reports_model->getInsuranceOutstanding();
            
            echo json_encode(['status' => 1, 'data' => $data]);
            
        } catch (Exception $e) {
            log_message('error', 'Outstanding payments error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error generating report']);
        }
    }
    
    /**
     * Export report to various formats
     */
    public function export_report($report_type = '', $format = 'pdf') {
        // Check export permission
        if (!$this->rbac->hasPrivilege('export_reports', 'can_view')) {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">Access denied</div>');
            redirect('admin/dashboard');
        }
        
        // Validate report type permission
        if (!$this->rbac->hasPrivilege($report_type . '_reports', 'can_view')) {
            $this->session->set_flashdata('msg', '<div class="alert alert-danger">Access denied for this report type</div>');
            redirect('admin/reports_dashboard');
        }
        
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Log export action
        $this->logReportAccess($report_type . '_export', array(
            'format' => $format,
            'date_from' => $date_from,
            'date_to' => $date_to
        ));
        
        switch ($format) {
            case 'pdf':
                $this->exportPDF($report_type, $date_from, $date_to);
                break;
                
            case 'excel':
                $this->exportExcel($report_type, $date_from, $date_to);
                break;
                
            case 'csv':
                $this->exportCSV($report_type, $date_from, $date_to);
                break;
                
            default:
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Invalid export format</div>');
                redirect('admin/reports_dashboard');
        }
    }
    
    /**
     * Get custom report builder view
     */
    public function custom_report() {
        // Check permission
        if (!$this->rbac->hasPrivilege('custom_report_builder', 'can_view')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'Reports');
        $this->session->set_userdata('sub_menu', 'reports/custom');
        
        // Get available tables and fields for report builder
        $data['available_tables'] = $this->reports_model->getReportableTables();
        $data['saved_reports'] = $this->reports_model->getSavedCustomReports();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/reports/custom_report_builder', $data);
        $this->load->view('layout/footer', $data);
    }
    
    /**
     * Save custom report configuration
     */
    public function save_custom_report() {
        // Check permission
        if (!$this->rbac->hasPrivilege('custom_report_builder', 'can_add')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $report_data = array(
            'report_name' => $this->input->post('report_name'),
            'report_code' => $this->generateReportCode($this->input->post('report_name')),
            'report_category' => $this->input->post('report_category'),
            'report_description' => $this->input->post('report_description'),
            'report_query' => $this->input->post('report_query'),
            'report_parameters' => json_encode($this->input->post('parameters')),
            'report_format' => $this->input->post('report_format'),
            'chart_type' => $this->input->post('chart_type'),
            'created_by' => $this->session->userdata('admin')['id'],
            'is_active' => 1
        );
        
        try {
            $report_id = $this->reports_model->saveCustomReport($report_data);
            
            if ($report_id) {
                echo json_encode(['status' => 1, 'message' => 'Report saved successfully', 'report_id' => $report_id]);
            } else {
                echo json_encode(['status' => 0, 'message' => 'Failed to save report']);
            }
        } catch (Exception $e) {
            log_message('error', 'Save custom report error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error saving report']);
        }
    }
    
    /**
     * Private helper methods
     */
    
    private function getUserReportPermissions() {
        $role_id = $this->session->userdata('admin')['roles'][0];
        $permissions = array();
        
        $report_permissions = array(
            'revenue_reports',
            'patient_visit_reports',
            'insurance_reports',
            'cash_payment_reports',
            'department_revenue_reports',
            'daily_collection_reports',
            'outstanding_payment_reports',
            'walkin_patient_reports',
            'registered_patient_reports',
            'patient_demographics_reports',
            'patient_flow_reports',
            'opd_service_reports',
            'ipd_service_reports',
            'pharmacy_sales_reports',
            'lab_test_reports',
            'radiology_reports',
            'executive_dashboard',
            'export_reports',
            'custom_report_builder'
        );
        
        foreach ($report_permissions as $permission) {
            if ($this->rbac->hasPrivilege($permission, 'can_view')) {
                $permissions[] = $permission;
            }
        }
        
        return $permissions;
    }
    
    private function getAvailableReports($permissions) {
        $all_reports = array(
            'Financial' => array(
                'revenue_reports' => 'Revenue Analysis',
                'cash_payment_reports' => 'Cash Payments',
                'insurance_reports' => 'Insurance Claims',
                'department_revenue_reports' => 'Department Revenue',
                'outstanding_payment_reports' => 'Outstanding Payments'
            ),
            'Patient Analytics' => array(
                'patient_visit_reports' => 'Visit Statistics',
                'walkin_patient_reports' => 'Walk-in Patients',
                'registered_patient_reports' => 'Registered Patients',
                'patient_demographics_reports' => 'Demographics',
                'patient_flow_reports' => 'Patient Flow'
            ),
            'Service Reports' => array(
                'opd_service_reports' => 'OPD Services',
                'ipd_service_reports' => 'IPD Services',
                'pharmacy_sales_reports' => 'Pharmacy Sales',
                'lab_test_reports' => 'Lab Tests',
                'radiology_reports' => 'Radiology'
            )
        );
        
        $available = array();
        
        foreach ($all_reports as $category => $reports) {
            foreach ($reports as $key => $name) {
                if (in_array($key, $permissions)) {
                    if (!isset($available[$category])) {
                        $available[$category] = array();
                    }
                    $available[$category][$key] = $name;
                }
            }
        }
        
        return $available;
    }
    
    private function getQuickStats() {
        $stats = array();
        
        // Today's statistics
        $stats['today'] = array(
            'revenue' => $this->reports_model->getTodayRevenue(),
            'patients' => $this->reports_model->getTodayPatients(),
            'appointments' => $this->reports_model->getTodayAppointments(),
            'admissions' => $this->reports_model->getTodayAdmissions()
        );
        
        // Month statistics
        $stats['month'] = array(
            'revenue' => $this->reports_model->getMonthRevenue(),
            'patients' => $this->reports_model->getMonthPatients(),
            'growth' => $this->reports_model->getMonthGrowth()
        );
        
        return $stats;
    }
    
    private function logReportAccess($report_type, $parameters = array()) {
        $log_data = array(
            'report_code' => $report_type,
            'user_id' => $this->session->userdata('admin')['id'],
            'parameters_used' => json_encode($parameters),
            'ip_address' => $this->input->ip_address(),
            'access_date' => date('Y-m-d H:i:s')
        );
        
        $this->reports_model->logReportAccess($log_data);
    }
    
    private function generateReportCode($report_name) {
        $code = strtolower(str_replace(' ', '_', $report_name));
        $code = preg_replace('/[^a-z0-9_]/', '', $code);
        return 'custom_' . $code . '_' . time();
    }
    
    /**
     * Export methods
     */
    
    private function exportPDF($report_type, $date_from, $date_to) {
        $this->load->library('pdf');
        
        // Get report data
        $report_data = $this->getReportData($report_type, $date_from, $date_to);
        
        // Get hospital info
        $hospital_info = $this->setting_model->getHospitalDetail();
        
        $html = $this->load->view('admin/reports/pdf/' . $report_type, array(
            'hospital' => $hospital_info,
            'report_data' => $report_data,
            'date_from' => $date_from,
            'date_to' => $date_to
        ), true);
        
        $this->pdf->loadHtml($html);
        $this->pdf->setPaper('A4', 'portrait');
        $this->pdf->render();
        
        $filename = $report_type . '_' . date('Y-m-d') . '.pdf';
        $this->pdf->stream($filename, array("Attachment" => 1));
    }
    
    private function exportExcel($report_type, $date_from, $date_to) {
        $this->load->library('excel');
        
        // Get report data
        $report_data = $this->getReportData($report_type, $date_from, $date_to);
        
        $objPHPExcel = new PHPExcel();
        $objPHPExcel->getProperties()
            ->setCreator("HMS Reports")
            ->setTitle(ucwords(str_replace('_', ' ', $report_type)))
            ->setSubject("Report from " . $date_from . " to " . $date_to);
        
        // Add data to excel
        $this->fillExcelData($objPHPExcel, $report_data, $report_type);
        
        // Save and download
        $filename = $report_type . '_' . date('Y-m-d') . '.xlsx';
        
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');
        
        $objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
        $objWriter->save('php://output');
    }
    
    private function exportCSV($report_type, $date_from, $date_to) {
        // Get report data
        $report_data = $this->getReportData($report_type, $date_from, $date_to);
        
        $filename = $report_type . '_' . date('Y-m-d') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // Add headers
        fputcsv($output, array('Report: ' . ucwords(str_replace('_', ' ', $report_type))));
        fputcsv($output, array('Date Range: ' . $date_from . ' to ' . $date_to));
        fputcsv($output, array()); // Empty line
        
        // Add data based on report type
        $this->outputCSVData($output, $report_data, $report_type);
        
        fclose($output);
    }
    
    private function getReportData($report_type, $date_from, $date_to) {
        switch ($report_type) {
            case 'revenue':
                return $this->reports_model->getFullRevenueReport($date_from, $date_to);
            case 'patient_visit':
                return $this->reports_model->getFullPatientReport($date_from, $date_to);
            case 'department_revenue':
                return $this->reports_model->getFullDepartmentReport($date_from, $date_to);
            default:
                return array();
        }
    }
    
    private function fillExcelData($objPHPExcel, $data, $report_type) {
        // Implementation specific to each report type
        // This would be customized based on your specific report structures
    }
    
    private function outputCSVData($output, $data, $report_type) {
        // Implementation specific to each report type
        // This would be customized based on your specific report structures
    }
}