<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Radiology extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->model('radiology_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
            
            // Add queue model
            $this->load->model('Queue_model');
        } catch (Exception $e) {
            log_message('error', 'Radiology controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'radiology');

        // Prepare data for dashboard
        $data['title'] = 'Radiology Dashboard';
        
        // Get queue data for radiology department
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('radiology');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('radiology');
        
        try {
            // Get pending orders
            $data['pending_orders'] = $this->getPendingOrders();
            
            // Get today's appointments
            $data['today_appointments'] = $this->getTodayAppointments();
            
            // Get equipment list
            $data['equipment_list'] = $this->getEquipmentList();
            
            // Get radiology staff
            $data['radiology_staff'] = $this->getRadiologyStaff();
            
            // Load dashboard view
            $this->load->view('layout/header', $data);
            $this->load->view('admin/radiology/dashboard', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Dashboard loading error: ' . $e->getMessage());
            show_error('Failed to load radiology dashboard: ' . $e->getMessage());
        }
    }

    /**
     * Patient verification interface (separate from dashboard)
     */
    public function patient_verification() {
        // Set session data
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'radiology');

        // Prepare data for verification view
        $data['title'] = 'Radiology & Imaging with Biometric Verification';
        
        // Get radiology staff (technicians, radiologists)
        try {
            $data['radiology_staff'] = $this->getRadiologyStaff();
        } catch (Exception $e) {
            log_message('error', 'Error loading radiology staff: ' . $e->getMessage());
            $data['radiology_staff'] = [];
        }
        
        // Get imaging test types
        $data['imaging_types'] = $this->getImagingTypes();
        
        // Get gender list
        try {
            if (method_exists($this->customlib, 'getGender_Patient')) {
                $data['genderList'] = $this->customlib->getGender_Patient();
            } else {
                $data['genderList'] = array(
                    'male' => 'Male',
                    'female' => 'Female',
                    'other' => 'Other'
                );
            }
        } catch (Exception $e) {
            $data['genderList'] = array('male' => 'Male', 'female' => 'Female');
        }
        
        // Load verification views (your original index view)
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/radiology/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'Verification view loading error: ' . $e->getMessage());
            show_error('Failed to load radiology verification view: ' . $e->getMessage());
        }
    }
    
    /**
 * Radiology Queue Management View
 * URL: /admin/radiology/queue
 */
public function queue() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'radiology/queue');
    
    $data['title'] = 'Radiology Queue Management';
    $data['department'] = 'radiology';
    $data['queue_data'] = $this->Queue_model->getDepartmentQueue('radiology');
    $data['queue_config'] = $this->Queue_model->getQueueConfig('radiology');
    
    // Get radiologists/technicians
    $data['radiology_staff'] = $this->getRadiologyStaff();
    $data['imaging_types'] = $this->getImagingTypes();
    
    $this->load->view('layout/header', $data);
    $this->load->view('admin/radiology/queue_management', $data);
    $this->load->view('layout/footer');
}


    /**
     * Get pending radiology orders
     */
    private function getPendingOrders() {
        try {
            $this->db->select('
                ro.id,
                ro.patient_id,
                ro.total_amount,
                ro.priority,
                ro.status,
                ro.order_date,
                p.patient_name,
                p.patient_unique_id,
                p.mobileno,
                p.fingerprint_template,
                CONCAT(s.name, " ", s.surname) as doctor_name
            ');
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'ro.patient_id = p.id', 'left');
            $this->db->join('staff s', 'ro.referring_doctor_id = s.id', 'left');
            $this->db->where('ro.status', 'pending');
            $this->db->or_where('ro.status', 'in_progress');
            $this->db->order_by('ro.created_at', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            return $query->result_array();
            
        } catch (Exception $e) {
            log_message('error', 'Error getting pending orders: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get today's appointments
     */
    private function getTodayAppointments() {
        try {
            $today = date('Y-m-d');
            
            $this->db->select('
                ro.id as order_id,
                ro.scheduled_time,
                ro.status,
                p.patient_name,
                p.patient_unique_id,
                "" as equipment_name,
                "" as technician_name
            ');
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'ro.patient_id = p.id', 'left');
            $this->db->where('DATE(ro.scheduled_date)', $today);
            $this->db->where_in('ro.status', ['pending', 'in_progress']);
            $this->db->order_by('ro.scheduled_time', 'ASC');
            
            $query = $this->db->get();
            $appointments = $query->result_array();
            
            // Add default equipment and technician info if not available
            foreach ($appointments as &$appointment) {
                if (empty($appointment['equipment_name'])) {
                    $appointment['equipment_name'] = 'X-Ray Machine 1';
                }
                if (empty($appointment['technician_name'])) {
                    $appointment['technician_name'] = 'Available';
                }
                if (empty($appointment['scheduled_time'])) {
                    $appointment['scheduled_time'] = '09:00:00';
                }
            }
            
            return $appointments;
            
        } catch (Exception $e) {
            log_message('error', 'Error getting today appointments: ' . $e->getMessage());
            return [];
        }
    }
    
        /**
 * AJAX: Get queue notifications for department
 * This method checks for actual waiting patients and returns them as notifications
 */
public function get_queue_notifications() {
    header('Content-Type: application/json');
    
    try {
        $department = $this->input->get('department', TRUE);
        $last_check = $this->input->get('last_check', TRUE);
        
        // Default to checking all departments if none specified
        if (empty($department) || $department === 'all') {
            $department = null;
        }
        
        // If no last_check provided, use current time minus 30 seconds
        if (empty($last_check)) {
            $last_check = time() - 30;
        }
        
        // Convert timestamp to datetime for database query
        $last_check_datetime = date('Y-m-d H:i:s', $last_check);
        
        // Query for new patients added to queue since last check
        $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->where('pq.created_at >', $last_check_datetime);
        
        // Filter by department if specified
        if ($department) {
            $this->db->where('pq.department', $department);
        }
        
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'notifications' => [],
                'timestamp' => time()
            ]);
            return;
        }
        
        $new_patients = $query->result_array();
        $notifications = [];
        
        // Convert database results to notification format
        foreach ($new_patients as $patient) {
            $notifications[] = [
                'patient_id' => $patient['patient_id'],
                'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                'queue_number' => $patient['queue_number'],
                'department' => $patient['department'],
                'priority' => $patient['priority'],
                'workflow_type' => $patient['queue_type'] ?: 'general',
                'time' => date('H:i:s', strtotime($patient['created_at'])),
                'created_at' => $patient['created_at'],
                'phone' => $patient['mobileno'] ?: '',
                'age' => $patient['age'] ?: '',
                'gender' => $patient['gender'] ?: '',
                'patient_unique_id' => $patient['patient_unique_id'] ?: ''
            ];
        }
        
        // Also check for priority patients (urgent/high priority)
        if (empty($new_patients)) {
            // Check for any urgent patients that might have been missed
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.status', 'waiting');
            $this->db->where_in('pq.priority', ['urgent', 'high']);
            
            if ($department) {
                $this->db->where('pq.department', $department);
            }
            
            $this->db->order_by('pq.created_at', 'DESC');
            $this->db->limit(5);
            
            $urgent_query = $this->db->get();
            $urgent_patients = $urgent_query->result_array();
            
            foreach ($urgent_patients as $patient) {
                $notifications[] = [
                    'patient_id' => $patient['patient_id'],
                    'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                    'queue_number' => $patient['queue_number'],
                    'department' => $patient['department'],
                    'priority' => $patient['priority'],
                    'workflow_type' => $patient['queue_type'] ?: 'urgent',
                    'time' => date('H:i:s', strtotime($patient['created_at'])),
                    'created_at' => $patient['created_at'],
                    'phone' => $patient['mobileno'] ?: '',
                    'age' => $patient['age'] ?: '',
                    'gender' => $patient['gender'] ?: '',
                    'patient_unique_id' => $patient['patient_unique_id'] ?: '',
                    'is_urgent' => true
                ];
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'notifications' => $notifications,
            'count' => count($notifications),
            'timestamp' => time(),
            'last_check' => $last_check,
            'department' => $department ?: 'all',
            'debug' => [
                'query_time' => $last_check_datetime,
                'current_time' => date('Y-m-d H:i:s'),
                'department_filter' => $department
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Exception: ' . $e->getMessage(),
            'notifications' => [],
            'timestamp' => time()
        ]);
    }
}

public function debug_queue_status() {
    header('Content-Type: application/json');
    
    try {
        // Get all waiting patients
        $this->db->select('pq.*, p.patient_name, p.mobileno');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(20);
        
        $query = $this->db->get();
        $waiting_patients = $query->result_array();
        
        // Get queue counts by department
        $this->db->select('department, COUNT(*) as count');
        $this->db->from('patient_queue');
        $this->db->where('status', 'waiting');
        $this->db->group_by('department');
        $dept_counts = $this->db->get()->result_array();
        
        echo json_encode([
            'status' => 'success',
            'waiting_patients' => $waiting_patients,
            'department_counts' => $dept_counts,
            'total_waiting' => count($waiting_patients),
            'timestamp' => time(),
            'current_time' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}
    /**
     * View specific order details
     */
    public function view_order($order_id = null) {
        if (!$order_id) {
            show_404();
        }

        try {
            // Get order details
            $this->db->select('
                ro.*,
                p.patient_name,
                p.patient_unique_id,
                p.mobileno,
                CONCAT(s.name, " ", s.surname) as doctor_name
            ');
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'ro.patient_id = p.id', 'left');
            $this->db->join('staff s', 'ro.referring_doctor_id = s.id', 'left');
            $this->db->where('ro.id', $order_id);
            
            $query = $this->db->get();
            
            if ($query->num_rows() === 0) {
                show_404();
            }
            
            $data['order'] = $query->row_array();
            $data['title'] = 'Order Details - #' . $order_id;
            
            // Get order tests
            if ($this->db->table_exists('radiology_order_tests')) {
                $this->db->select('rot.*, rt.test_name, rt.price');
                $this->db->from('radiology_order_tests rot');
                $this->db->join('radiology_tests rt', 'rot.test_id = rt.id', 'left');
                $this->db->where('rot.radiology_order_id', $order_id);
                $tests_query = $this->db->get();
                $data['order_tests'] = $tests_query->result_array();
            } else {
                $data['order_tests'] = [];
            }
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/radiology/view_order', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Error viewing order: ' . $e->getMessage());
            show_error('Failed to load order details');
        }
    }

    /**
     * Start examination
     */
    public function start_exam($order_id = null) {
        header('Content-Type: application/json');
        
        if (!$order_id) {
            echo json_encode(['success' => false, 'message' => 'Order ID required']);
            return;
        }

        try {
            // Update order status
            $this->db->where('id', $order_id);
            $this->db->update('radiology_orders', [
                'status' => 'in_progress',
                'started_at' => date('Y-m-d H:i:s'),
                'technician_id' => $this->getStaffIdSafe(),
                'updated_at' => date('Y-m-d H:i:s')
            ]);
            
            if ($this->db->affected_rows() > 0) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Examination started successfully',
                    'order_id' => $order_id
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'Failed to start examination'
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get equipment list
     */
    private function getEquipmentList() {
        try {
            // Try to get from equipment table if it exists
            if ($this->db->table_exists('radiology_equipment')) {
                $this->db->select('*');
                $this->db->from('radiology_equipment');
                $this->db->where('is_active', 1);
                $query = $this->db->get();
                
                if ($query->num_rows() > 0) {
                    return $query->result_array();
                }
            }
            
            // Return default equipment list if no table exists
            return [
                [
                    'id' => 1,
                    'equipment_name' => 'X-Ray Machine',
                    'manufacturer' => 'Philips',
                    'model' => 'DigitalDiagnost C90',
                    'location' => 'Room 101',
                    'status' => 'active'
                ],
                [
                    'id' => 2,
                    'equipment_name' => 'CT Scanner',
                    'manufacturer' => 'GE Healthcare',
                    'model' => 'Revolution CT',
                    'location' => 'Room 102',
                    'status' => 'active'
                ],
                [
                    'id' => 3,
                    'equipment_name' => 'Ultrasound',
                    'manufacturer' => 'Siemens',
                    'model' => 'ACUSON X300',
                    'location' => 'Room 103',
                    'status' => 'active'
                ],
                [
                    'id' => 4,
                    'equipment_name' => 'MRI Scanner',
                    'manufacturer' => 'Siemens',
                    'model' => 'MAGNETOM Vida',
                    'location' => 'Room 104',
                    'status' => 'maintenance'
                ]
            ];
            
        } catch (Exception $e) {
            log_message('error', 'Error getting equipment list: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
 * Orders Management Page
 */
public function orders() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'radiology');

    $data['title'] = 'Radiology Orders Management';
    
    try {
        // Get all orders with filters
        $status_filter = $this->input->get('status');
        $date_filter = $this->input->get('date');
        
        $data['orders'] = $this->getAllRadiologyOrders($status_filter, $date_filter);
        $data['status_filter'] = $status_filter;
        $data['date_filter'] = $date_filter;
        
        // Get order statistics
        $data['order_stats'] = $this->getOrderStatistics();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/radiology/orders', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Orders page error: ' . $e->getMessage());
        show_error('Failed to load orders page: ' . $e->getMessage());
    }
}

/**
 * Schedule Management Page
 */
public function schedule() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'radiology');

    $data['title'] = 'Radiology Scheduling';
    
    try {
        // Get appointments for the week
        $week_start = $this->input->get('week') ?: date('Y-m-d', strtotime('monday this week'));
        $data['week_start'] = $week_start;
        $data['appointments'] = $this->getWeeklyAppointments($week_start);
        
        // Get available time slots
        $data['time_slots'] = $this->getAvailableTimeSlots();
        
        // Get equipment for scheduling
        $data['equipment_list'] = $this->getEquipmentList();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/radiology/schedule', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Schedule page error: ' . $e->getMessage());
        show_error('Failed to load schedule page: ' . $e->getMessage());
    }
}

/**
 * Reports Page
 */
public function reports() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'radiology');

    $data['title'] = 'Radiology Reports';
    
    try {
        // Get report data
        $start_date = $this->input->get('start_date') ?: date('Y-m-01');
        $end_date = $this->input->get('end_date') ?: date('Y-m-d');
        
        $data['start_date'] = $start_date;
        $data['end_date'] = $end_date;
        
        // Get various reports
        $data['revenue_report'] = $this->getRevenueReport($start_date, $end_date);
        $data['productivity_report'] = $this->getProductivityReport($start_date, $end_date);
        $data['equipment_usage'] = $this->getEquipmentUsageReport($start_date, $end_date);
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/radiology/reports', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Reports page error: ' . $e->getMessage());
        show_error('Failed to load reports page: ' . $e->getMessage());
    }
}

/**
 * FIXED: Verify patient for specific order (not general verification)
 */
public function verify_order($order_id = null) {
    if (!$order_id) {
        $order_id = $this->input->get('order_id');
    }
    
    if (!$order_id) {
        show_404();
    }

    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'radiology');

    try {
        // Get order details
        $this->db->select('
            ro.*,
            p.patient_name,
            p.patient_unique_id,
            p.mobileno,
            p.fingerprint_template,
            p.patient_type,
            CONCAT(s.name, " ", s.surname) as doctor_name
        ');
        $this->db->from('radiology_orders ro');
        $this->db->join('patients p', 'ro.patient_id = p.id', 'left');
        $this->db->join('staff s', 'ro.referring_doctor_id = s.id', 'left');
        $this->db->where('ro.id', $order_id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() === 0) {
            show_404();
        }
        
        $order = $query->row_array();
        
        // Get order tests
        $order_tests = [];
        if ($this->db->table_exists('radiology_order_tests')) {
            $this->db->select('rot.*, rt.test_name, rt.price');
            $this->db->from('radiology_order_tests rot');
            $this->db->join('radiology_tests rt', 'rot.test_id = rt.id', 'left');
            $this->db->where('rot.radiology_order_id', $order_id);
            $tests_query = $this->db->get();
            $order_tests = $tests_query->result_array();
        }
        
        $data = [
            'title' => 'Verify Patient for Order #' . $order_id,
            'order' => $order,
            'order_tests' => $order_tests,
            'verification_mode' => 'order_specific', // Flag to indicate this is order-specific verification
            'radiology_staff' => $this->getRadiologyStaff(),
            'imaging_types' => $this->getImagingTypes()
        ];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/radiology/verify_order', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Error in verify_order: ' . $e->getMessage());
        show_error('Failed to load order verification page');
    }
}

/**
 * Process order verification
 */
public function process_order_verification() {
    header('Content-Type: application/json');
    
    try {
        $order_id = $this->input->post('order_id');
        $verification_result = $this->input->post('verification_result'); // success/failed
        $verification_method = $this->input->post('verification_method'); // biometric/manual/override
        $match_score = $this->input->post('match_score');
        
        if (!$order_id) {
            echo json_encode(['success' => false, 'message' => 'Order ID required']);
            return;
        }
        
        // Update order verification status
        $update_data = [
            'verification_status' => $verification_result === 'success' ? 'completed' : 'failed',
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Add verification fields if columns exist
        if ($this->db->field_exists('verification_method', 'radiology_orders')) {
            $update_data['verification_method'] = $verification_method;
        }
        if ($this->db->field_exists('verification_score', 'radiology_orders')) {
            $update_data['verification_score'] = $match_score;
        }
        
        $this->db->where('id', $order_id);
        $this->db->update('radiology_orders', $update_data);
        
        if ($this->db->affected_rows() > 0) {
            // Log verification attempt
            $this->logVerificationAttempt(
                $this->input->post('patient_id'),
                $match_score ?: 0,
                $verification_result === 'success',
                $verification_method ?: 'manual',
                "Order #$order_id verification"
            );
            
            echo json_encode([
                'success' => true,
                'message' => 'Order verification completed successfully',
                'order_id' => $order_id,
                'verification_result' => $verification_result
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to update order verification status'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error processing verification: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get all radiology orders with optional filters
 */
private function getAllRadiologyOrders($status_filter = null, $date_filter = null) {
    try {
        $this->db->select('
            ro.*,
            p.patient_name,
            p.patient_unique_id,
            p.mobileno,
            p.fingerprint_template,
            CONCAT(s.name, " ", s.surname) as doctor_name,
            COUNT(rot.id) as test_count
        ');
        $this->db->from('radiology_orders ro');
        $this->db->join('patients p', 'ro.patient_id = p.id', 'left');
        $this->db->join('staff s', 'ro.referring_doctor_id = s.id', 'left');
        
        // Join with order tests if table exists
        if ($this->db->table_exists('radiology_order_tests')) {
            $this->db->join('radiology_order_tests rot', 'ro.id = rot.radiology_order_id', 'left');
        }
        
        // Apply filters
        if ($status_filter && $status_filter !== 'all') {
            $this->db->where('ro.status', $status_filter);
        }
        
        if ($date_filter) {
            $this->db->where('DATE(ro.order_date)', $date_filter);
        }
        
        $this->db->group_by('ro.id');
        $this->db->order_by('ro.created_at', 'DESC');
        $this->db->limit(100); // Limit for performance
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting all orders: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get order statistics
 */
private function getOrderStatistics() {
    try {
        $stats = [
            'total_orders' => 0,
            'pending_orders' => 0,
            'completed_orders' => 0,
            'today_orders' => 0,
            'total_revenue' => 0
        ];
        
        // Total orders
        $this->db->select('COUNT(*) as total');
        $this->db->from('radiology_orders');
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $stats['total_orders'] = $query->row()->total;
        }
        
        // Pending orders
        $this->db->select('COUNT(*) as total');
        $this->db->from('radiology_orders');
        $this->db->where_in('status', ['pending', 'in_progress']);
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $stats['pending_orders'] = $query->row()->total;
        }
        
        // Completed orders
        $this->db->select('COUNT(*) as total');
        $this->db->from('radiology_orders');
        $this->db->where('status', 'completed');
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $stats['completed_orders'] = $query->row()->total;
        }
        
        // Today's orders
        $this->db->select('COUNT(*) as total');
        $this->db->from('radiology_orders');
        $this->db->where('DATE(order_date)', date('Y-m-d'));
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $stats['today_orders'] = $query->row()->total;
        }
        
        // Total revenue
        $this->db->select('SUM(total_amount) as revenue');
        $this->db->from('radiology_orders');
        $this->db->where('status !=', 'cancelled');
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $stats['total_revenue'] = $query->row()->revenue ?: 0;
        }
        
        return $stats;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting order statistics: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get weekly appointments
 */
private function getWeeklyAppointments($week_start) {
    try {
        $week_end = date('Y-m-d', strtotime($week_start . ' +6 days'));
        
        $this->db->select('
            ro.*,
            p.patient_name,
            p.patient_unique_id,
            p.mobileno,
            CONCAT(s.name, " ", s.surname) as doctor_name
        ');
        $this->db->from('radiology_orders ro');
        $this->db->join('patients p', 'ro.patient_id = p.id', 'left');
        $this->db->join('staff s', 'ro.referring_doctor_id = s.id', 'left');
        $this->db->where('ro.scheduled_date >=', $week_start);
        $this->db->where('ro.scheduled_date <=', $week_end);
        $this->db->order_by('ro.scheduled_date, ro.scheduled_time');
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting weekly appointments: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get available time slots
 */
private function getAvailableTimeSlots() {
    return [
        '08:00' => '08:00 AM',
        '08:30' => '08:30 AM',
        '09:00' => '09:00 AM',
        '09:30' => '09:30 AM',
        '10:00' => '10:00 AM',
        '10:30' => '10:30 AM',
        '11:00' => '11:00 AM',
        '11:30' => '11:30 AM',
        '12:00' => '12:00 PM',
        '14:00' => '02:00 PM',
        '14:30' => '02:30 PM',
        '15:00' => '03:00 PM',
        '15:30' => '03:30 PM',
        '16:00' => '04:00 PM',
        '16:30' => '04:30 PM',
        '17:00' => '05:00 PM'
    ];
}

/**
 * Get revenue report
 */
private function getRevenueReport($start_date, $end_date) {
    try {
        $this->db->select('
            DATE(order_date) as date,
            COUNT(*) as order_count,
            SUM(total_amount) as total_revenue,
            AVG(total_amount) as avg_order_value
        ');
        $this->db->from('radiology_orders');
        $this->db->where('order_date >=', $start_date);
        $this->db->where('order_date <=', $end_date);
        $this->db->where('status !=', 'cancelled');
        $this->db->group_by('DATE(order_date)');
        $this->db->order_by('date', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting revenue report: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get productivity report
 */
private function getProductivityReport($start_date, $end_date) {
    try {
        $this->db->select('
            CONCAT(s.name, " ", s.surname) as staff_name,
            COUNT(ro.id) as orders_handled,
            SUM(ro.total_amount) as revenue_generated
        ');
        $this->db->from('radiology_orders ro');
        $this->db->join('staff s', 'ro.technician_id = s.id', 'left');
        $this->db->where('ro.order_date >=', $start_date);
        $this->db->where('ro.order_date <=', $end_date);
        $this->db->where('ro.status', 'completed');
        $this->db->group_by('ro.technician_id');
        $this->db->order_by('orders_handled', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting productivity report: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get equipment usage report
 */
private function getEquipmentUsageReport($start_date, $end_date) {
    try {
        // This would need an equipment usage tracking table
        // For now, return sample data
        return [
            [
                'equipment_name' => 'X-Ray Machine',
                'usage_count' => 45,
                'total_revenue' => 112500,
                'utilization_rate' => 75
            ],
            [
                'equipment_name' => 'CT Scanner',
                'usage_count' => 20,
                'total_revenue' => 170000,
                'utilization_rate' => 60
            ],
            [
                'equipment_name' => 'Ultrasound',
                'usage_count' => 30,
                'total_revenue' => 105000,
                'utilization_rate' => 50
            ]
        ];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting equipment usage report: ' . $e->getMessage());
        return [];
    }
}

    /**
     * Get all patients for radiology selection
     */
    public function getAllPatients() {
        try {
            $patients = $this->radiology_model->getAllPatientsForRadiology();
            
            $formatted_patients = array();
            foreach ($patients as $patient) {
                $fingerprint_status = '';
                if (!empty($patient['fingerprint_template'])) {
                    $fingerprint_status = '<span class="badge badge-success"><i class="fa fa-fingerprint"></i> Verified</span>';
                } else {
                    $fingerprint_status = '<span class="badge badge-warning"><i class="fa fa-exclamation-triangle"></i> No Biometric</span>';
                }
                
                $age_display = $patient['age'];
                if (!empty($patient['month'])) {
                    $age_display .= ' years ' . $patient['month'] . ' months';
                } else {
                    $age_display .= ' years';
                }
                
                $formatted_patients[] = array(
                    'id' => $patient['id'],
                    'patient_name' => $patient['patient_name'],
                    'guardian_name' => $patient['guardian_name'],
                    'age' => $age_display,
                    'gender' => ucfirst($patient['gender']),
                    'mobileno' => $patient['mobileno'],
                    //'patient_unique_id' => $patient['patient_unique_id'],
                    'fingerprint_status' => $fingerprint_status,
                    'last_visit' => $patient['last_visit'] ?? 'Never',
                    'action' => '<button class="btn btn-primary btn-sm" onclick="selectPatientForRadiology(' . $patient['id'] . ')">
                                    <i class="fa fa-user-plus"></i> Select
                                </button>'
                );
            }
            
            echo json_encode(array('data' => $formatted_patients));
        } catch (Exception $e) {
            log_message('error', 'Error getting patients for radiology: ' . $e->getMessage());
            echo json_encode(array('data' => array(), 'error' => $e->getMessage()));
        }
    }

    /**
     * Get patient details for radiology workflow
     */
public function getPatientDetails() {
    /*if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }*/
    
    $patient_id = $this->input->post('id');
    
    if (empty($patient_id)) {
        echo json_encode(['success' => false, 'message' => 'Patient ID required']);
        return;
    }
    
    try {
        // Get patient details including fingerprint status
        $this->db->select('
            id, patient_name, patient_unique_id, guardian_name, gender, 
            age, month, mobileno, dob, fingerprint_template, 
            fingerprint_verified, created_at
        ');
        $this->db->where('id', $patient_id);
        $query = $this->db->get('patients');
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            // Add fingerprint status
            $patient['has_fingerprint'] = !empty($patient['fingerprint_template']);
            $patient['fingerprint_verified'] = $patient['fingerprint_verified'] == 1;
            
            // Remove actual template from response for security
            unset($patient['fingerprint_template']);
            
            echo json_encode([
                'success' => true,
                'patient' => $patient
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Patient not found']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    }
}

public function getPatientData($id = null)
{
    header('Content-Type: application/json');

    if (empty($id)) {
        echo json_encode([
            'success' => false,
            'message' => 'Patient ID is required.'
        ]);
        return;
    }

    // Validate that ID is numeric
    if (!is_numeric($id)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid Patient ID format.'
        ]);
        return;
    }

    // Fetch patient data
    $this->db->select('
        id, patient_name, patient_unique_id, guardian_name, gender, 
        age, month, day, mobileno, dob, email, 
        blood_group, address, image, fingerprint_template, fingerprint_image
    ');
    $this->db->where('id', $id);
    $query = $this->db->get('patients');

    if ($query->num_rows() > 0) {
        $patient = $query->row_array();

        // Optionally remove sensitive fields
        unset($patient['fingerprint_template']);

        echo json_encode([
            'success' => true
        ] + $patient); // Flat JSON
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Patient not found'
        ]);
    }
}

// In application/controllers/admin/Radiology.php
public function get_patients() {
    if (!$this->rbac->hasPrivilege('opd_patient', 'can_view')) {
        // Return JSON error instead of access_denied() which returns HTML
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Access denied']);
        return;
    }

    $search = $this->input->get('search', TRUE);
    
    try {
        $patients = $this->radiology_model->searchPatients($search);
        
        // Ensure we return JSON
        header('Content-Type: application/json');
        echo json_encode($patients);
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

// Add this test method to your Radiology controller
public function test_json() {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'message' => 'JSON response working',
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

public function getStoredFingerprintTemplate() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Get fingerprint data
        $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            if (!empty($patient['fingerprint_template'])) {
                $template = trim($patient['fingerprint_template']);
                
                echo json_encode([
                    'status' => 'success',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'fingerprint_template' => $template,
                    'fingerprint_image' => $patient['fingerprint_image'] ?? '',
                    'template_size' => strlen($template)
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No fingerprint template stored for this patient'
                ]);
            }
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}
    
    /**
 * Get patient fingerprint template for verification
 */
public function getPatientTemplate() {
    if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }
    
    $patient_id = $this->input->post('patient_id');
    
    if (empty($patient_id)) {
        echo json_encode(['success' => false, 'message' => 'Patient ID required']);
        return;
    }
    
    try {
        // Get patient fingerprint template from database
        $this->db->select('fingerprint_template, fingerprint_verified');
        $this->db->where('id', $patient_id);
        $query = $this->db->get('patients');
        
        if ($query->num_rows() > 0) {
            $patient = $query->row();
            
            if (!empty($patient->fingerprint_template)) {
                echo json_encode([
                    'success' => true,
                    'template' => $patient->fingerprint_template,
                    'verified' => $patient->fingerprint_verified == 1
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'No fingerprint template found for this patient'
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Patient not found']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    }
}

/**
 * Verify patient identity using biometric matching
 */
public function verifyPatientIdentity() {
    if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }
    
    $patient_id = $this->input->post('patient_id');
    $scanned_template = $this->input->post('scanned_template');
    
    if (empty($patient_id) || empty($scanned_template)) {
        echo json_encode(['success' => false, 'message' => 'Patient ID and scanned template required']);
        return;
    }
    
    try {
        // Get stored template
        $this->db->select('fingerprint_template, patient_name, patient_unique_id');
        $this->db->where('id', $patient_id);
        $query = $this->db->get('patients');
        
        if ($query->num_rows() > 0) {
            $patient = $query->row();
            
            if (!empty($patient->fingerprint_template)) {
                // In a real implementation, you would call SecuGen matching service here
                // For now, we'll simulate a match score based on template comparison
                $match_score = $this->simulateBiometricMatch($scanned_template, $patient->fingerprint_template);
                
                $verification_result = [
                    'success' => true,
                    'verified' => $match_score >= 50, // 50% threshold
                    'match_score' => $match_score / 100, // Convert to decimal
                    'patient_name' => $patient->patient_name,
                    'patient_id' => $patient->patient_unique_id
                ];
                
                // Log verification attempt
                $this->logVerificationAttempt($patient_id, $match_score, $verification_result['verified']);
                
                echo json_encode($verification_result);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'No stored fingerprint template for verification'
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Patient not found']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Verification error: ' . $e->getMessage()]);
    }
}

public function create_order() {
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    $this->output->enable_profiler(FALSE);
    
    try {
        // Authentication check
        if (!$this->rbac->hasPrivilege('opd_patient', 'can_add') && !$this->rbac->hasPrivilege('opd_patient', 'can_edit')) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied - insufficient privileges'
            ]);
            return;
        }
        
        // Get and validate POST data
        $patient_id = $this->input->post('patient_id');
        $referring_doctor_id = $this->input->post('referring_doctor_id');
        $imaging_tests = $this->input->post('imaging_tests');
        $total_amount = floatval($this->input->post('total_amount') ?: 0);
        
        // Basic validation
        if (empty($patient_id) || empty($referring_doctor_id) || empty($imaging_tests) || !is_array($imaging_tests)) {
            echo json_encode([
                'success' => false,
                'message' => 'Missing required fields: patient, doctor, tests, and amount'
            ]);
            return;
        }
        
        if ($total_amount <= 0) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid amount: must be greater than zero'
            ]);
            return;
        }
        
        // Start database transaction for atomicity
        $this->db->trans_begin();
        
        try {
            // 1. Create the radiology order
            $order_result = $this->createRadiologyOrderRecord($patient_id, $referring_doctor_id, $imaging_tests, $total_amount);
            
            if (!$order_result['success']) {
                throw new Exception($order_result['message']);
            }
            
            $order_id = $order_result['order_id'];
            
            // 2. Create accounting journal entries (double entry)
            $journal_result = $this->createRadiologyJournalEntries($order_id, $patient_id, $total_amount, $imaging_tests);
            
            if (!$journal_result['success']) {
                throw new Exception('Accounting error: ' . $journal_result['message']);
            }
            
            // 3. Update order with journal reference
            $this->updateOrderWithJournalReference($order_id, $journal_result['journal_id']);
            
            // Commit transaction
            $this->db->trans_commit();
            
            log_message('info', "Radiology order created with accounting: Order ID=$order_id, Journal ID={$journal_result['journal_id']}, Amount=$total_amount");
            
            echo json_encode([
                'success' => true,
                'status' => 'success',
                'message' => 'Radiology order created successfully with accounting entries',
                'order_id' => $order_id,
                'journal_id' => $journal_result['journal_id'],
                'total_amount' => $total_amount,
                'test_count' => count($imaging_tests)
            ]);
            
        } catch (Exception $e) {
            // Rollback transaction on any error
            $this->db->trans_rollback();
            throw $e;
        }
        
    } catch (Exception $e) {
        log_message('error', 'Exception in create_order with accounting: ' . $e->getMessage());
        
        echo json_encode([
            'success' => false,
            'message' => 'Error creating order: ' . $e->getMessage(),
            'type' => 'exception'
        ]);
    }
}

/**
 * Create radiology order record
 */
private function createRadiologyOrderRecord($patient_id, $referring_doctor_id, $imaging_tests, $total_amount) {
    try {
        $staff_id = $this->getStaffIdSafe();
        
        // Prepare order data
        $order_data = [
            'patient_id' => intval($patient_id),
            'referring_doctor_id' => intval($referring_doctor_id),
            'clinical_notes' => $this->input->post('clinical_notes') ?: '',
            'priority' => $this->input->post('priority') ?: 'routine',
            'status' => 'pending',
            'order_date' => date('Y-m-d H:i:s'),
            'total_amount' => $total_amount,
            'created_by' => $staff_id,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Add optional fields if columns exist
        if ($this->db->field_exists('verification_status', 'radiology_orders')) {
            $order_data['verification_status'] = 'completed';
        }
        
        if ($this->input->post('scheduled_date') && $this->db->field_exists('scheduled_date', 'radiology_orders')) {
            $order_data['scheduled_date'] = $this->input->post('scheduled_date');
        }
        
        if ($this->input->post('scheduled_time') && $this->db->field_exists('scheduled_time', 'radiology_orders')) {
            $order_data['scheduled_time'] = $this->input->post('scheduled_time');
        }
        
        // Insert order
        $this->db->insert('radiology_orders', $order_data);
        $order_id = $this->db->insert_id();
        
        if (!$order_id) {
            return ['success' => false, 'message' => 'Failed to create order record'];
        }
        
        // Insert test records
        $this->insertRadiologyTests($order_id, $imaging_tests);
        
        return ['success' => true, 'order_id' => $order_id];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Order creation error: ' . $e->getMessage()];
    }
}

/**
 * Insert radiology test records
 */
private function insertRadiologyTests($order_id, $imaging_tests) {
    if ($this->db->table_exists('radiology_order_tests')) {
        foreach ($imaging_tests as $test_id) {
            $test_data = [
                'radiology_order_id' => $order_id,
                'test_id' => intval($test_id),
                'status' => 'pending',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            if ($this->db->field_exists('updated_at', 'radiology_order_tests')) {
                $test_data['updated_at'] = date('Y-m-d H:i:s');
            }
            
            $this->db->insert('radiology_order_tests', $test_data);
        }
    }
}

/**
 * Create journal entries for radiology order (Double Entry Bookkeeping)
 */
private function createRadiologyJournalEntries($order_id, $patient_id, $total_amount, $imaging_tests) {
    try {
        // Get patient info for journal description
        $patient_info = $this->getPatientInfo($patient_id);
        $patient_name = $patient_info ? $patient_info['patient_name'] : 'Unknown Patient';
        
        // Generate journal reference number
        $reference_no = $this->generateJournalReference('RAD', $order_id);
        
        // Get chart of accounts for radiology
        $accounts = $this->getRadiologyAccounts();
        
        if (!$accounts['accounts_receivable'] || !$accounts['radiology_revenue']) {
            return ['success' => false, 'message' => 'Required accounting accounts not found'];
        }
        
        // Calculate tax amount (16% VAT)
        $tax_rate = 0.16;
        $base_amount = $total_amount / (1 + $tax_rate);
        $tax_amount = $total_amount - $base_amount;
        
        // Create journal header
        $journal_data = [
            'reference_no' => $reference_no,
            'date' => date('Y-m-d'),
            'description' => "Radiology Services - $patient_name (Order #$order_id)",
            'total_amount' => $total_amount,
            'source_module' => 'radiology',
            'source_id' => $order_id,
            'patient_id' => $patient_id,
            'status' => 'posted',
            'created_by' => $this->getStaffIdSafe(),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('journal_headers', $journal_data);
        $journal_id = $this->db->insert_id();
        
        if (!$journal_id) {
            return ['success' => false, 'message' => 'Failed to create journal header'];
        }
        
        // Create journal lines (Double Entry)
        $journal_lines = [];
        
        // 1. DEBIT: Accounts Receivable (Asset increases)
        $journal_lines[] = [
            'journal_id' => $journal_id,
            'account_id' => $accounts['accounts_receivable'],
            'debit_amount' => $total_amount,
            'credit_amount' => 0.00,
            'description' => "Radiology services provided to $patient_name",
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // 2. CREDIT: Radiology Revenue (Revenue increases)
        $journal_lines[] = [
            'journal_id' => $journal_id,
            'account_id' => $accounts['radiology_revenue'],
            'debit_amount' => 0.00,
            'credit_amount' => $base_amount,
            'description' => "Revenue from radiology services",
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // 3. CREDIT: Tax Payable (Liability increases) - if tax amount > 0
        if ($tax_amount > 0 && $accounts['tax_payable']) {
            $journal_lines[] = [
                'journal_id' => $journal_id,
                'account_id' => $accounts['tax_payable'],
                'debit_amount' => 0.00,
                'credit_amount' => $tax_amount,
                'description' => "VAT on radiology services (16%)",
                'created_at' => date('Y-m-d H:i:s')
            ];
        }
        
        // Insert all journal lines
        foreach ($journal_lines as $line) {
            $this->db->insert('journal_lines', $line);
        }
        
        // Verify double entry balance
        $debit_total = array_sum(array_column($journal_lines, 'debit_amount'));
        $credit_total = array_sum(array_column($journal_lines, 'credit_amount'));
        
        if (abs($debit_total - $credit_total) > 0.01) { // Allow for rounding differences
            return ['success' => false, 'message' => "Journal entries not balanced: Debit=$debit_total, Credit=$credit_total"];
        }
        
        // Log accounting activity
        $this->logAccountingActivity($journal_id, 'radiology_order_created', $order_id, $total_amount);
        
        return [
            'success' => true,
            'journal_id' => $journal_id,
            'reference_no' => $reference_no,
            'base_amount' => $base_amount,
            'tax_amount' => $tax_amount,
            'total_amount' => $total_amount
        ];
        
    } catch (Exception $e) {
        log_message('error', 'Journal creation error: ' . $e->getMessage());
        return ['success' => false, 'message' => 'Journal creation failed: ' . $e->getMessage()];
    }
}

/**
 * Get radiology chart of accounts
 */
private function getRadiologyAccounts() {
    try {
        $accounts = [
            'accounts_receivable' => null,
            'radiology_revenue' => null,
            'tax_payable' => null
        ];
        
        // Get Accounts Receivable (Asset)
        $this->db->select('id');
        $this->db->from('chart_of_accounts');
        $this->db->where('account_type', 'asset');
        $this->db->where('is_active', 1);
        $this->db->where("(account_name LIKE '%receivable%' OR account_code LIKE '%1200%' OR account_name LIKE '%debtors%')");
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $accounts['accounts_receivable'] = $query->row()->id;
        }
        
        // Get Radiology Revenue (Revenue)
        $this->db->select('id');
        $this->db->from('chart_of_accounts');
        $this->db->where('account_type', 'revenue');
        $this->db->where('is_active', 1);
        $this->db->where("(account_name LIKE '%radiology%' OR account_name LIKE '%imaging%' OR account_code LIKE '%4002%')");
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $accounts['radiology_revenue'] = $query->row()->id;
        }
        
        // Get Tax Payable (Liability)
        $this->db->select('id');
        $this->db->from('chart_of_accounts');
        $this->db->where('account_type', 'liability');
        $this->db->where('is_active', 1);
        $this->db->where("(account_name LIKE '%tax%' OR account_name LIKE '%vat%' OR account_code LIKE '%2001%')");
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $accounts['tax_payable'] = $query->row()->id;
        }
        
        // Create missing accounts if they don't exist
        $accounts = $this->createMissingRadiologyAccounts($accounts);
        
        return $accounts;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting radiology accounts: ' . $e->getMessage());
        return ['accounts_receivable' => null, 'radiology_revenue' => null, 'tax_payable' => null];
    }
}

/**
 * Create missing radiology accounts
 */
private function createMissingRadiologyAccounts($accounts) {
    try {
        // Create Accounts Receivable if missing
        if (!$accounts['accounts_receivable']) {
            $ar_data = [
                'account_code' => '1200',
                'account_name' => 'Accounts Receivable - Patients',
                'account_type' => 'asset',
                'description' => 'Money owed by patients for medical services',
                'is_active' => 1,
                'opening_balance' => 0.00,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            if ($this->db->field_exists('normal_balance', 'chart_of_accounts')) {
                $ar_data['normal_balance'] = 'debit';
            }
            
            $this->db->insert('chart_of_accounts', $ar_data);
            $accounts['accounts_receivable'] = $this->db->insert_id();
        }
        
        // Create Radiology Revenue if missing
        if (!$accounts['radiology_revenue']) {
            $rev_data = [
                'account_code' => '4002',
                'account_name' => 'Radiology Revenue',
                'account_type' => 'revenue',
                'description' => 'Revenue from radiology and imaging services',
                'is_active' => 1,
                'opening_balance' => 0.00,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            if ($this->db->field_exists('normal_balance', 'chart_of_accounts')) {
                $rev_data['normal_balance'] = 'credit';
            }
            
            $this->db->insert('chart_of_accounts', $rev_data);
            $accounts['radiology_revenue'] = $this->db->insert_id();
        }
        
        // Create Tax Payable if missing
        if (!$accounts['tax_payable']) {
            $tax_data = [
                'account_code' => '2001',
                'account_name' => 'VAT Payable',
                'account_type' => 'liability',
                'description' => 'Value Added Tax payable to tax authority',
                'is_active' => 1,
                'opening_balance' => 0.00,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            if ($this->db->field_exists('normal_balance', 'chart_of_accounts')) {
                $tax_data['normal_balance'] = 'credit';
            }
            
            $this->db->insert('chart_of_accounts', $tax_data);
            $accounts['tax_payable'] = $this->db->insert_id();
        }
        
        return $accounts;
        
    } catch (Exception $e) {
        log_message('error', 'Error creating missing accounts: ' . $e->getMessage());
        return $accounts;
    }
}

/**
 * Generate journal reference number
 */
private function generateJournalReference($prefix, $order_id) {
    $date_part = date('Ymd');
    $order_part = str_pad($order_id, 6, '0', STR_PAD_LEFT);
    return "{$prefix}-{$date_part}-{$order_part}";
}

/**
 * Update order with journal reference
 */
private function updateOrderWithJournalReference($order_id, $journal_id) {
    try {
        $update_data = ['updated_at' => date('Y-m-d H:i:s')];
        
        // Add journal_id if column exists
        if ($this->db->field_exists('journal_id', 'radiology_orders')) {
            $update_data['journal_id'] = $journal_id;
        }
        
        $this->db->where('id', $order_id);
        $this->db->update('radiology_orders', $update_data);
        
    } catch (Exception $e) {
        log_message('error', 'Error updating order with journal reference: ' . $e->getMessage());
    }
}

/**
 * Log accounting activity for audit trail
 */
private function logAccountingActivity($journal_id, $activity_type, $source_id, $amount) {
    try {
        if ($this->db->table_exists('journal_audit_trail')) {
            $audit_data = [
                'journal_id' => $journal_id,
                'action' => 'created',
                'new_data' => json_encode([
                    'activity_type' => $activity_type,
                    'source_id' => $source_id,
                    'amount' => $amount
                ]),
                'user_id' => $this->getStaffIdSafe(),
                'ip_address' => $this->input->ip_address(),
                'user_agent' => substr($this->input->user_agent(), 0, 500),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('journal_audit_trail', $audit_data);
        }
    } catch (Exception $e) {
        log_message('error', 'Error logging accounting activity: ' . $e->getMessage());
    }
}

/**
 * Create payment journal entry when payment is received
 */
public function recordRadiologyPayment() {
    header('Content-Type: application/json');
    
    try {
        $order_id = $this->input->post('order_id');
        $payment_amount = floatval($this->input->post('payment_amount'));
        $payment_method = $this->input->post('payment_method'); // cash, card, mobile, insurance
        
        if (!$order_id || $payment_amount <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid order ID or payment amount']);
            return;
        }
        
        // Get order details
        $this->db->select('*');
        $this->db->from('radiology_orders');
        $this->db->where('id', $order_id);
        $order_query = $this->db->get();
        
        if ($order_query->num_rows() === 0) {
            echo json_encode(['success' => false, 'message' => 'Order not found']);
            return;
        }
        
        $order = $order_query->row_array();
        
        // Start transaction
        $this->db->trans_begin();
        
        try {
            // Create payment journal entry
            $payment_journal_result = $this->createPaymentJournalEntry($order, $payment_amount, $payment_method);
            
            if (!$payment_journal_result['success']) {
                throw new Exception($payment_journal_result['message']);
            }
            
            // Update order payment status
            $this->db->where('id', $order_id);
            $this->db->update('radiology_orders', [
                'status' => 'paid',
                'payment_status' => 'completed',
                'paid_amount' => $payment_amount,
                'payment_date' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ]);
            
            $this->db->trans_commit();
            
            echo json_encode([
                'success' => true,
                'message' => 'Payment recorded successfully',
                'journal_id' => $payment_journal_result['journal_id'],
                'payment_amount' => $payment_amount
            ]);
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            throw $e;
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Payment recording error: ' . $e->getMessage()]);
    }
}

/**
 * Create payment journal entry (Cash/Card received, A/R reduced)
 */
private function createPaymentJournalEntry($order, $payment_amount, $payment_method) {
    try {
        $patient_info = $this->getPatientInfo($order['patient_id']);
        $patient_name = $patient_info ? $patient_info['patient_name'] : 'Unknown Patient';
        
        $reference_no = $this->generateJournalReference('PAY', $order['id']);
        
        // Get accounts
        $accounts = $this->getPaymentAccounts($payment_method);
        $radiology_accounts = $this->getRadiologyAccounts();
        
        // Create journal header
        $journal_data = [
            'reference_no' => $reference_no,
            'date' => date('Y-m-d'),
            'description' => "Payment received for radiology services - $patient_name (Order #{$order['id']})",
            'total_amount' => $payment_amount,
            'source_module' => 'radiology_payment',
            'source_id' => $order['id'],
            'patient_id' => $order['patient_id'],
            'status' => 'posted',
            'created_by' => $this->getStaffIdSafe(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('journal_headers', $journal_data);
        $journal_id = $this->db->insert_id();
        
        // Create journal lines
        $journal_lines = [
            // DEBIT: Cash/Bank (Asset increases)
            [
                'journal_id' => $journal_id,
                'account_id' => $accounts['cash_account'],
                'debit_amount' => $payment_amount,
                'credit_amount' => 0.00,
                'description' => "Payment received via $payment_method",
                'created_at' => date('Y-m-d H:i:s')
            ],
            // CREDIT: Accounts Receivable (Asset decreases)
            [
                'journal_id' => $journal_id,
                'account_id' => $radiology_accounts['accounts_receivable'],
                'debit_amount' => 0.00,
                'credit_amount' => $payment_amount,
                'description' => "Payment received from $patient_name",
                'created_at' => date('Y-m-d H:i:s')
            ]
        ];
        
        // Insert journal lines
        foreach ($journal_lines as $line) {
            $this->db->insert('journal_lines', $line);
        }
        
        return ['success' => true, 'journal_id' => $journal_id];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Payment journal error: ' . $e->getMessage()];
    }
}

/**
 * Get payment accounts based on payment method
 */
private function getPaymentAccounts($payment_method) {
    $accounts = ['cash_account' => null];
    
    try {
        switch (strtolower($payment_method)) {
            case 'cash':
                $search_terms = ['cash', 'petty cash', '1001'];
                break;
            case 'card':
            case 'credit_card':
            case 'debit_card':
                $search_terms = ['bank', 'checking', 'current', '1100'];
                break;
            case 'mobile':
            case 'mpesa':
                $search_terms = ['mobile money', 'mpesa', 'mobile', '1150'];
                break;
            default:
                $search_terms = ['cash', '1001'];
        }
        
        foreach ($search_terms as $term) {
            $this->db->select('id');
            $this->db->from('chart_of_accounts');
            $this->db->where('account_type', 'asset');
            $this->db->where('is_active', 1);
            $this->db->where("(account_name LIKE '%$term%' OR account_code LIKE '%$term%')");
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $accounts['cash_account'] = $query->row()->id;
                break;
            }
        }
        
        // Create default cash account if none found
        if (!$accounts['cash_account']) {
            $cash_data = [
                'account_code' => '1001',
                'account_name' => 'Cash in Hand',
                'account_type' => 'asset',
                'description' => 'Cash payments received',
                'is_active' => 1,
                'opening_balance' => 0.00,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('chart_of_accounts', $cash_data);
            $accounts['cash_account'] = $this->db->insert_id();
        }
        
        return $accounts;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting payment accounts: ' . $e->getMessage());
        return ['cash_account' => null];
    }
}

/**
 * Debug method to check accounting setup
 */
public function debug_accounting_setup() {
    header('Content-Type: application/json');
    
    try {
        $accounts = $this->getRadiologyAccounts();
        $payment_accounts = $this->getPaymentAccounts('cash');
        
        $setup_info = [
            'radiology_accounts' => $accounts,
            'payment_accounts' => $payment_accounts,
            'journal_tables' => [
                'journal_headers' => $this->db->table_exists('journal_headers'),
                'journal_lines' => $this->db->table_exists('journal_lines'),
                'chart_of_accounts' => $this->db->table_exists('chart_of_accounts')
            ],
            'sample_chart_of_accounts' => []
        ];
        
        // Get sample chart of accounts
        $this->db->select('id, account_code, account_name, account_type');
        $this->db->from('chart_of_accounts');
        $this->db->where('is_active', 1);
        $this->db->limit(10);
        $query = $this->db->get();
        $setup_info['sample_chart_of_accounts'] = $query->result_array();
        
        echo json_encode([
            'success' => true,
            'accounting_setup' => $setup_info,
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error checking accounting setup: ' . $e->getMessage()
        ]);
    }
}

/**
 * Generate accounting reports for radiology
 */
public function getRadiologyAccountingReport() {
    header('Content-Type: application/json');
    
    try {
        $start_date = $this->input->get('start_date') ?: date('Y-m-01');
        $end_date = $this->input->get('end_date') ?: date('Y-m-d');
        
        // Get radiology revenue summary
        $revenue_query = "
            SELECT 
                DATE(jh.date) as transaction_date,
                COUNT(jh.id) as order_count,
                SUM(jh.total_amount) as total_revenue,
                SUM(jl_tax.credit_amount) as total_tax,
                SUM(jl_revenue.credit_amount) as base_revenue
            FROM journal_headers jh
            LEFT JOIN journal_lines jl_revenue ON jh.id = jl_revenue.journal_id 
                AND jl_revenue.account_id IN (
                    SELECT id FROM chart_of_accounts 
                    WHERE account_type = 'revenue' 
                    AND (account_name LIKE '%radiology%' OR account_name LIKE '%imaging%')
                )
            LEFT JOIN journal_lines jl_tax ON jh.id = jl_tax.journal_id 
                AND jl_tax.account_id IN (
                    SELECT id FROM chart_of_accounts 
                    WHERE account_type = 'liability' 
                    AND (account_name LIKE '%tax%' OR account_name LIKE '%vat%')
                )
            WHERE jh.source_module = 'radiology'
            AND jh.date BETWEEN ? AND ?
            AND jh.status = 'posted'
            GROUP BY DATE(jh.date)
            ORDER BY transaction_date DESC
        ";
        
        $revenue_result = $this->db->query($revenue_query, [$start_date, $end_date]);
        
        // Get outstanding receivables
        $receivables_query = "
            SELECT 
                p.patient_name,
                p.patient_unique_id,
                ro.id as order_id,
                ro.order_date,
                ro.total_amount,
                COALESCE(ro.paid_amount, 0) as paid_amount,
                (ro.total_amount - COALESCE(ro.paid_amount, 0)) as outstanding_amount
            FROM radiology_orders ro
            JOIN patients p ON ro.patient_id = p.id
            WHERE ro.status != 'cancelled'
            AND (ro.payment_status IS NULL OR ro.payment_status != 'completed')
            AND (ro.total_amount - COALESCE(ro.paid_amount, 0)) > 0
            ORDER BY ro.order_date ASC
        ";
        
        $receivables_result = $this->db->query($receivables_query);
        
        // Get payment methods summary
        $payments_query = "
            SELECT 
                CASE 
                    WHEN jh.description LIKE '%cash%' THEN 'Cash'
                    WHEN jh.description LIKE '%card%' THEN 'Card'
                    WHEN jh.description LIKE '%mobile%' OR jh.description LIKE '%mpesa%' THEN 'Mobile Money'
                    ELSE 'Other'
                END as payment_method,
                COUNT(*) as transaction_count,
                SUM(jh.total_amount) as total_amount
            FROM journal_headers jh
            WHERE jh.source_module = 'radiology_payment'
            AND jh.date BETWEEN ? AND ?
            AND jh.status = 'posted'
            GROUP BY payment_method
            ORDER BY total_amount DESC
        ";
        
        $payments_result = $this->db->query($payments_query, [$start_date, $end_date]);
        
        $report_data = [
            'period' => [
                'start_date' => $start_date,
                'end_date' => $end_date
            ],
            'revenue_summary' => $revenue_result->result_array(),
            'outstanding_receivables' => $receivables_result->result_array(),
            'payment_methods' => $payments_result->result_array(),
            'totals' => [
                'total_revenue' => 0,
                'total_tax' => 0,
                'total_receivables' => 0,
                'total_payments' => 0
            ]
        ];
        
        // Calculate totals
        foreach ($report_data['revenue_summary'] as $row) {
            $report_data['totals']['total_revenue'] += $row['total_revenue'];
            $report_data['totals']['total_tax'] += $row['total_tax'];
        }
        
        foreach ($report_data['outstanding_receivables'] as $row) {
            $report_data['totals']['total_receivables'] += $row['outstanding_amount'];
        }
        
        foreach ($report_data['payment_methods'] as $row) {
            $report_data['totals']['total_payments'] += $row['total_amount'];
        }
        
        echo json_encode([
            'success' => true,
            'report' => $report_data,
            'generated_at' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error generating accounting report: ' . $e->getMessage()
        ]);
    }
}

/**
 * Create refund journal entry for cancelled orders
 */
public function processRadiologyRefund() {
    header('Content-Type: application/json');
    
    try {
        $order_id = $this->input->post('order_id');
        $refund_amount = floatval($this->input->post('refund_amount'));
        $refund_reason = $this->input->post('refund_reason');
        
        if (!$order_id || $refund_amount <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid order ID or refund amount']);
            return;
        }
        
        // Get order details
        $this->db->select('*');
        $this->db->from('radiology_orders');
        $this->db->where('id', $order_id);
        $order_query = $this->db->get();
        
        if ($order_query->num_rows() === 0) {
            echo json_encode(['success' => false, 'message' => 'Order not found']);
            return;
        }
        
        $order = $order_query->row_array();
        
        // Start transaction
        $this->db->trans_begin();
        
        try {
            // Create refund journal entry
            $refund_result = $this->createRefundJournalEntry($order, $refund_amount, $refund_reason);
            
            if (!$refund_result['success']) {
                throw new Exception($refund_result['message']);
            }
            
            // Update order status
            $this->db->where('id', $order_id);
            $this->db->update('radiology_orders', [
                'status' => 'refunded',
                'refund_amount' => $refund_amount,
                'refund_date' => date('Y-m-d H:i:s'),
                'refund_reason' => $refund_reason,
                'updated_at' => date('Y-m-d H:i:s')
            ]);
            
            $this->db->trans_commit();
            
            echo json_encode([
                'success' => true,
                'message' => 'Refund processed successfully',
                'journal_id' => $refund_result['journal_id'],
                'refund_amount' => $refund_amount
            ]);
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            throw $e;
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Refund processing error: ' . $e->getMessage()]);
    }
}

/**
 * Create refund journal entry
 */
private function createRefundJournalEntry($order, $refund_amount, $refund_reason) {
    try {
        $patient_info = $this->getPatientInfo($order['patient_id']);
        $patient_name = $patient_info ? $patient_info['patient_name'] : 'Unknown Patient';
        
        $reference_no = $this->generateJournalReference('REF', $order['id']);
        
        // Get accounts
        $accounts = $this->getRadiologyAccounts();
        $cash_accounts = $this->getPaymentAccounts('cash');
        
        // Calculate tax portion
        $tax_rate = 0.16;
        $base_refund = $refund_amount / (1 + $tax_rate);
        $tax_refund = $refund_amount - $base_refund;
        
        // Create journal header
        $journal_data = [
            'reference_no' => $reference_no,
            'date' => date('Y-m-d'),
            'description' => "Refund for radiology services - $patient_name (Order #{$order['id']}) - Reason: $refund_reason",
            'total_amount' => $refund_amount,
            'source_module' => 'radiology_refund',
            'source_id' => $order['id'],
            'patient_id' => $order['patient_id'],
            'status' => 'posted',
            'created_by' => $this->getStaffIdSafe(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('journal_headers', $journal_data);
        $journal_id = $this->db->insert_id();
        
        // Create journal lines (reverse of original entry)
        $journal_lines = [
            // DEBIT: Radiology Revenue (Revenue decreases)
            [
                'journal_id' => $journal_id,
                'account_id' => $accounts['radiology_revenue'],
                'debit_amount' => $base_refund,
                'credit_amount' => 0.00,
                'description' => "Refund - reduction in radiology revenue",
                'created_at' => date('Y-m-d H:i:s')
            ],
            // DEBIT: Tax Payable (Liability decreases)
            [
                'journal_id' => $journal_id,
                'account_id' => $accounts['tax_payable'],
                'debit_amount' => $tax_refund,
                'credit_amount' => 0.00,
                'description' => "VAT refund on cancelled radiology services",
                'created_at' => date('Y-m-d H:i:s')
            ],
            // CREDIT: Cash (Asset decreases)
            [
                'journal_id' => $journal_id,
                'account_id' => $cash_accounts['cash_account'],
                'debit_amount' => 0.00,
                'credit_amount' => $refund_amount,
                'description' => "Cash refund to $patient_name",
                'created_at' => date('Y-m-d H:i:s')
            ]
        ];
        
        // Insert journal lines
        foreach ($journal_lines as $line) {
            $this->db->insert('journal_lines', $line);
        }
        
        return ['success' => true, 'journal_id' => $journal_id];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Refund journal error: ' . $e->getMessage()];
    }
}

/**
 * Method to setup initial chart of accounts for radiology
 */
public function setupRadiologyAccounts() {
    header('Content-Type: application/json');
    
    try {
        $accounts_to_create = [
            // Assets
            [
                'account_code' => '1001',
                'account_name' => 'Cash in Hand',
                'account_type' => 'asset',
                'description' => 'Cash received from patients',
                'normal_balance' => 'debit'
            ],
            [
                'account_code' => '1100',
                'account_name' => 'Bank - Current Account',
                'account_type' => 'asset',
                'description' => 'Main bank account for operations',
                'normal_balance' => 'debit'
            ],
            [
                'account_code' => '1150',
                'account_name' => 'Mobile Money - M-Pesa',
                'account_type' => 'asset',
                'description' => 'Mobile money payments received',
                'normal_balance' => 'debit'
            ],
            [
                'account_code' => '1200',
                'account_name' => 'Accounts Receivable - Patients',
                'account_type' => 'asset',
                'description' => 'Money owed by patients for medical services',
                'normal_balance' => 'debit'
            ],
            
            // Liabilities
            [
                'account_code' => '2001',
                'account_name' => 'VAT Payable',
                'account_type' => 'liability',
                'description' => 'Value Added Tax payable to KRA',
                'normal_balance' => 'credit'
            ],
            [
                'account_code' => '2200',
                'account_name' => 'Accrued Expenses',
                'account_type' => 'liability',
                'description' => 'Expenses incurred but not yet paid',
                'normal_balance' => 'credit'
            ],
            
            // Revenue
            [
                'account_code' => '4001',
                'account_name' => 'Medical Service Revenue',
                'account_type' => 'revenue',
                'description' => 'General medical service revenue',
                'normal_balance' => 'credit'
            ],
            [
                'account_code' => '4002',
                'account_name' => 'Radiology Revenue',
                'account_type' => 'revenue',
                'description' => 'Revenue from radiology and imaging services',
                'normal_balance' => 'credit'
            ],
            [
                'account_code' => '4003',
                'account_name' => 'Laboratory Revenue',
                'account_type' => 'revenue',
                'description' => 'Revenue from laboratory services',
                'normal_balance' => 'credit'
            ],
            
            // Expenses
            [
                'account_code' => '5001',
                'account_name' => 'Staff Salaries',
                'account_type' => 'expense',
                'description' => 'Salaries paid to medical and administrative staff',
                'normal_balance' => 'debit'
            ],
            [
                'account_code' => '5002',
                'account_name' => 'Medical Supplies',
                'account_type' => 'expense',
                'description' => 'Cost of medical supplies and consumables',
                'normal_balance' => 'debit'
            ],
            [
                'account_code' => '5003',
                'account_name' => 'Equipment Maintenance',
                'account_type' => 'expense',
                'description' => 'Maintenance of medical equipment',
                'normal_balance' => 'debit'
            ]
        ];
        
        $created_count = 0;
        $skipped_count = 0;
        
        foreach ($accounts_to_create as $account) {
            // Check if account already exists
            $this->db->select('id');
            $this->db->from('chart_of_accounts');
            $this->db->where('account_code', $account['account_code']);
            $existing = $this->db->get();
            
            if ($existing->num_rows() === 0) {
                $account_data = [
                    'account_code' => $account['account_code'],
                    'account_name' => $account['account_name'],
                    'account_type' => $account['account_type'],
                    'description' => $account['description'],
                    'is_active' => 1,
                    'opening_balance' => 0.00,
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                // Add normal_balance if column exists
                if ($this->db->field_exists('normal_balance', 'chart_of_accounts')) {
                    $account_data['normal_balance'] = $account['normal_balance'];
                }
                
                $this->db->insert('chart_of_accounts', $account_data);
                $created_count++;
            } else {
                $skipped_count++;
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => "Chart of accounts setup completed",
            'created_accounts' => $created_count,
            'skipped_accounts' => $skipped_count,
            'total_processed' => count($accounts_to_create)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error setting up accounts: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get staff ID safely
 */
private function getStaffIdSafe() {
    try {
        if (method_exists($this->customlib, 'getStaffID')) {
            $staff_id = $this->customlib->getStaffID();
            if ($staff_id) return $staff_id;
        }
        
        $admin_id = $this->session->userdata('admin_id');
        if ($admin_id) return $admin_id;
        
        $session_keys = ['id', 'user_id', 'staff_id'];
        foreach ($session_keys as $key) {
            $value = $this->session->userdata($key);
            if ($value) return $value;
        }
        
        return 1; // Default fallback
        
    } catch (Exception $e) {
        log_message('error', 'Error getting staff ID: ' . $e->getMessage());
        return 1;
    }
}

/**
 * Get patient information
 */
private function getPatientInfo($patient_id) {
    try {
        $this->db->select('patient_name, patient_unique_id');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $query = $this->db->get();
        
        return $query->num_rows() > 0 ? $query->row_array() : null;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient info: ' . $e->getMessage());
        return null;
    }
}

/**
 * Simplified order processing method
 */
private function processOrderCreation() {
    try {
        // Get POST data
        $patient_id = $this->input->post('patient_id');
        $referring_doctor_id = $this->input->post('referring_doctor_id');
        $imaging_tests = $this->input->post('imaging_tests');
        
        // Debug what we received
        log_message('debug', 'Order creation - Patient ID: ' . $patient_id);
        log_message('debug', 'Order creation - Doctor ID: ' . $referring_doctor_id);
        log_message('debug', 'Order creation - Tests: ' . json_encode($imaging_tests));
        
        // Basic validation
        if (empty($patient_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        if (empty($referring_doctor_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Referring doctor is required'
            ]);
            return;
        }
        
        if (empty($imaging_tests) || !is_array($imaging_tests)) {
            echo json_encode([
                'success' => false,
                'message' => 'At least one imaging test is required'
            ]);
            return;
        }
        
        // Get staff ID using the same method as other controllers
        $staff_id = $this->getStaffId();
        
        // Prepare order data
        $order_data = [
            'patient_id' => intval($patient_id),
            'referring_doctor_id' => intval($referring_doctor_id),
            'clinical_notes' => $this->input->post('clinical_notes') ?: '',
            'priority' => $this->input->post('priority') ?: 'routine',
            'status' => 'pending',
            'order_date' => date('Y-m-d H:i:s'),
            'verification_status' => 'completed',
            'total_amount' => floatval($this->input->post('total_amount') ?: 0),
            'created_by' => $staff_id,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Optional fields
        if ($this->input->post('scheduled_date')) {
            $order_data['scheduled_date'] = $this->input->post('scheduled_date');
        }
        if ($this->input->post('scheduled_time')) {
            $order_data['scheduled_time'] = $this->input->post('scheduled_time');
        }
        if ($this->input->post('verification_method')) {
            $order_data['verification_method'] = $this->input->post('verification_method');
        }
        if ($this->input->post('verification_score')) {
            $order_data['verification_score'] = floatval($this->input->post('verification_score'));
        }
        
        log_message('debug', 'Inserting order data: ' . json_encode($order_data));
        
        // Insert the order
        $this->db->insert('radiology_orders', $order_data);
        $order_id = $this->db->insert_id();
        
        if (!$order_id) {
            $db_error = $this->db->error();
            log_message('error', 'Failed to insert radiology order: ' . json_encode($db_error));
            
            echo json_encode([
                'success' => false,
                'message' => 'Failed to create order record',
                'db_error' => $db_error['message']
            ]);
            return;
        }
        
        // Insert tests
        $test_count = 0;
        foreach ($imaging_tests as $test_id) {
            $test_data = [
                'radiology_order_id' => $order_id,
                'test_id' => intval($test_id),
                'status' => 'pending',
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('radiology_order_tests', $test_data);
            if ($this->db->affected_rows() > 0) {
                $test_count++;
            }
        }
        
        log_message('info', "Radiology order created successfully: ID=$order_id, Tests=$test_count, Patient=$patient_id, Staff=$staff_id");
        
        // Complete radiology queue
            if ($queue_id) {
                $result = $this->Queue_model->completeService($queue_id, $next_department, $findings);
                
                if ($next_department) {
                    $this->send_queue_notification($next_department, 'radiology_results_ready', [
                        'patient_id' => $patient_id,
                        'report_id' => $report_id,
                        'imaging_type' => $imaging_type,
                        'radiologist_name' => $this->get_radiologist_name($radiologist_id),
                        'priority' => $this->determine_imaging_priority($findings)
                    ]);
                }
            }
        
        // Return success
        echo json_encode([
            'success' => true,
            'status' => 'success',
            'message' => 'Radiology order created successfully',
            'order_id' => $order_id,
            'test_count' => $test_count,
            'redirect' => null
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Exception in processOrderCreation: ' . $e->getMessage());
        
        echo json_encode([
            'success' => false,
            'message' => 'Server exception: ' . $e->getMessage(),
            'type' => 'exception'
        ]);
    }
}

 /**
 * AJAX: Call next patient for radiology
 */
public function call_next_patient() {
    header('Content-Type: application/json');
    
    try {
        $radiologist_id = $this->input->post('radiologist_id') ?: $this->getStaffId();
        $result = $this->Queue_model->callNextPatient('radiology', $radiologist_id);
        
        if ($result['status'] === 'success') {
            $this->send_queue_notification('radiology', 'patient_called_for_imaging', array_merge($result['patient'], [
                'radiologist_name' => $this->get_radiologist_name($radiologist_id)
            ]));
        }
        
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Remove from radiology queue
 */
public function remove_from_queue() {
    header('Content-Type: application/json');
    
    try {
        $queue_id = $this->input->post('queue_id');
        $reason = $this->input->post('reason', TRUE) ?: 'Imaging completed';
        
        $result = $this->Queue_model->skipPatient($queue_id, $reason);
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

// =============================================
    // NOTIFICATION SYSTEM
    // =============================================

    /**
     * Check for incoming patients from triage/consultation
     */
    public function check_incoming_notifications() {
        header('Content-Type: application/json');
        
        try {
            $notification_file = APPPATH . 'cache/queue_notifications_radiology.json';
            
            if (file_exists($notification_file)) {
                $notification = json_decode(file_get_contents($notification_file), true);
                unlink($notification_file);
                
                echo json_encode([
                    'status' => 'success',
                    'notification' => $notification,
                    'has_notification' => true,
                    'sound' => true
                ]);
            } else {
                echo json_encode([
                    'status' => 'success',
                    'has_notification' => false
                ]);
            }
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Send notification to departments
     */
    private function send_queue_notification($department, $action, $data = []) {
        try {
            $notification = [
                'department' => $department,
                'action' => $action,
                'data' => $data,
                'timestamp' => time(),
                'from_department' => 'radiology',
                'sound' => true
            ];
            
            $notification_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
            file_put_contents($notification_file, json_encode($notification));
            
            return true;
        } catch (Exception $e) {
            log_message('error', 'Radiology notification failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * AJAX: Get radiology queue statistics
     */
    public function get_queue_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('radiology');
            echo json_encode(['status' => 'success', 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Get current radiology queue
     */
    public function get_current_queue() {
        header('Content-Type: application/json');
        
        try {
            $queue_data = $this->Queue_model->getDepartmentQueue('radiology');
            echo json_encode(['status' => 'success', 'queue' => $queue_data]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }


private function get_radiologist_name($radiologist_id) {
    if (!$radiologist_id) return 'Radiologist';
    
    $this->db->select('name, surname');
    $this->db->where('id', $radiologist_id);
    $staff = $this->db->get('staff')->row();
    
    return $staff ? $staff->name . ' ' . $staff->surname : 'Radiologist';
}

private function determine_imaging_priority($findings) {
    $urgent_keywords = ['fracture', 'tumor', 'emergency', 'urgent', 'critical'];
    $findings_lower = strtolower($findings);
    
    foreach ($urgent_keywords as $keyword) {
        if (strpos($findings_lower, $keyword) !== false) {
            return 'urgent';
        }
    }
    
    return 'normal';
}

/**
 * Enhanced verification logging
 */
private function logVerificationAttempt($patient_id, $match_score, $success, $method, $notes = '') {
    try {
        $log_data = [
            'patient_id' => $patient_id,
            'verification_method' => $method,
            'match_score' => $match_score,
            'verification_result' => $success ? 'success' : 'failed',
            'staff_id' => $this->session->userdata('admin_id') ?? 1,
            'department' => 'radiology',
            'notes' => $notes,
            'created_at' => date('Y-m-d H:i:s'),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => substr($this->input->user_agent(), 0, 255)
        ];
        
        // Try to insert into verification log table
        if ($this->db->table_exists('radiology_verification_log')) {
            $this->db->insert('radiology_verification_log', $log_data);
        }
        
        // Always log to file as well
        log_message('info', 'Radiology verification: ' . json_encode($log_data));
        
    } catch (Exception $e) {
        log_message('error', 'Failed to log verification attempt: ' . $e->getMessage());
    }
}

/**
 * Log manual verification override
 */
public function logManualVerification() {
    if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }
    
    $patient_id = $this->input->post('patient_id');
    $patient_name = $this->input->post('patient_name');
    $verification_type = $this->input->post('verification_type');
    $user_id = $this->session->userdata('admin_id');
    
    try {
        // Insert verification log
        $log_data = [
            'patient_id' => $patient_id,
            'patient_name' => $patient_name,
            'verification_type' => $verification_type,
            'verified_by' => $user_id,
            'verification_method' => 'manual_override',
            'match_score' => null,
            'created_at' => date('Y-m-d H:i:s'),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent()
        ];
        
        $this->db->insert('radiology_verification_log', $log_data);
        
        echo json_encode(['success' => true, 'message' => 'Manual verification logged']);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Logging error: ' . $e->getMessage()]);
    }
}

    /**
     * Create new radiology order
     */
public function createRadiologyOrder() {
        // ALWAYS set JSON header first
        header('Content-Type: application/json');
        
        // Log that method was called
        log_message('debug', 'createRadiologyOrder method called');
        log_message('debug', 'POST data: ' . print_r($_POST, true));
        
        try {
            // Get POST data
            $patient_id = $this->input->post('patient_id');
            $referring_doctor_id = $this->input->post('referring_doctor_id');
            $imaging_tests = $this->input->post('imaging_tests');
            
            // Debug what we received
            $debug_data = [
                'patient_id' => $patient_id,
                'referring_doctor_id' => $referring_doctor_id,
                'imaging_tests' => $imaging_tests,
                'all_post' => $this->input->post()
            ];
            log_message('debug', 'Received data: ' . json_encode($debug_data));
            
            // Basic validation
            if (empty($patient_id)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Patient ID is required',
                    'debug' => $debug_data
                ]);
                return;
            }
            
            if (empty($referring_doctor_id)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Referring doctor is required',
                    'debug' => $debug_data
                ]);
                return;
            }
            
            if (empty($imaging_tests) || !is_array($imaging_tests)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Imaging tests are required and must be an array',
                    'debug' => $debug_data
                ]);
                return;
            }
            
            // Check if required tables exist
            if (!$this->db->table_exists('radiology_orders')) {
                echo json_encode([
                    'success' => false,
                    'message' => 'radiology_orders table does not exist'
                ]);
                return;
            }
            
            if (!$this->db->table_exists('radiology_order_tests')) {
                echo json_encode([
                    'success' => false,
                    'message' => 'radiology_order_tests table does not exist'
                ]);
                return;
            }
            
            // Try to insert - SIMPLIFIED VERSION
            $order_data = [
                'patient_id' => intval($patient_id),
                'referring_doctor_id' => intval($referring_doctor_id),
                'clinical_notes' => $this->input->post('clinical_notes') ?: '',
                'priority' => $this->input->post('priority') ?: 'routine',
                'status' => 'pending',
                'order_date' => date('Y-m-d H:i:s'),
                'verification_status' => 'pending',
                'total_amount' => floatval($this->input->post('total_amount') ?: 0),
                'created_by' => $this->session->userdata('admin_id') ?: 1,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            log_message('debug', 'Inserting order data: ' . json_encode($order_data));
            
            // Insert the order
            $this->db->insert('radiology_orders', $order_data);
            $order_id = $this->db->insert_id();
            
            if (!$order_id) {
                $db_error = $this->db->error();
                log_message('error', 'Failed to insert radiology order: ' . json_encode($db_error));
                
                echo json_encode([
                    'success' => false,
                    'message' => 'Failed to create order record',
                    'db_error' => $db_error
                ]);
                return;
            }
            
            // Insert tests
            $test_count = 0;
            foreach ($imaging_tests as $test_id) {
                $test_data = [
                    'radiology_order_id' => $order_id,
                    'test_id' => intval($test_id),
                    'status' => 'pending',
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('radiology_order_tests', $test_data);
                if ($this->db->affected_rows() > 0) {
                    $test_count++;
                }
            }
            
            log_message('debug', "Order created: ID=$order_id, Tests=$test_count");
            
            // Return success
            echo json_encode([
                'success' => true,
                'status' => 'success',
                'message' => 'Radiology order created successfully',
                'order_id' => $order_id,
                'test_count' => $test_count
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Exception in createRadiologyOrder: ' . $e->getMessage());
            log_message('error', 'Stack trace: ' . $e->getTraceAsString());
            
            echo json_encode([
                'success' => false,
                'message' => 'Server exception: ' . $e->getMessage(),
                'type' => 'exception'
            ]);
        }
    }


// Helper method to process test IDs
private function processTestIds($imaging_tests) {
    $test_ids = array();
    
    foreach ($imaging_tests as $test_identifier) {
        // If it's already a numeric ID, use it
        if (is_numeric($test_identifier)) {
            $test_ids[] = intval($test_identifier);
        } else {
            // If it's a test code, look up the ID
            $test_id = $this->getTestIdByCode($test_identifier);
            if ($test_id) {
                $test_ids[] = $test_id;
            }
        }
    }
    
    return array_unique($test_ids);
}

// Helper method to get test ID by code
private function getTestIdByCode($test_code) {
    $this->db->select('id');
    $this->db->from('radiology_tests');
    $this->db->where('test_code', $test_code);
    $this->db->or_where('id', $test_code); // In case it's already an ID
    $query = $this->db->get();
    
    if ($query->num_rows() > 0) {
        return $query->row()->id;
    }
    
    return null;
}

// Helper method to validate patient exists
private function validatePatientExists($patient_id) {
    $this->db->select('id');
    $this->db->from('patients');
    $this->db->where('id', $patient_id);
    $query = $this->db->get();
    
    return $query->num_rows() > 0;
}

// Helper method to validate doctor exists
private function validateDoctorExists($doctor_id) {
    $this->db->select('id');
    $this->db->from('staff');
    $this->db->where('id', $doctor_id);
    $query = $this->db->get();
    
    return $query->num_rows() > 0;
}

// Helper method to get current staff ID
private function getStaffId() {
    // Get from session or return default
    return $this->session->userdata('admin_id') ?: 1;
}

// Helper method to log radiology activity
private function logRadiologyActivity($patient_id, $activity_type, $description) {
    try {
        if ($this->db->table_exists('radiology_activity_log')) {
            $log_data = array(
                'patient_id' => $patient_id,
                'staff_id' => $this->getStaffId(),
                'activity_type' => $activity_type,
                'description' => $description,
                'timestamp' => date('Y-m-d H:i:s'),
                'department' => 'radiology',
                'ip_address' => $this->input->ip_address()
            );
            
            $this->db->insert('radiology_activity_log', $log_data);
        }
    } catch (Exception $e) {
        log_message('error', 'Failed to log radiology activity: ' . $e->getMessage());
        // Don't fail the main operation if logging fails
    }
}

    /**
     * Get test IDs by their keys/codes
     */
    private function getTestIdsByKeys($test_keys) {
        try {
            $test_ids = array();
            
            // Define mapping of keys to test codes
            $key_to_code_mapping = array(
                'xray' => 'XRAY_CHEST',
                'ct_scan' => 'CT_HEAD',
                'mri' => 'MRI_BRAIN',
                'ultrasound' => 'US_ABD',
                'mammography' => 'MAMMO'
            );
            
            foreach ($test_keys as $key) {
                // Check if it's already a numeric ID
                if (is_numeric($key)) {
                    $test_ids[] = intval($key);
                    continue;
                }
                
                // Map key to test code
                $test_code = isset($key_to_code_mapping[$key]) ? $key_to_code_mapping[$key] : $key;
                
                // Get test ID from database
                $this->db->select('id');
                $this->db->from('radiology_tests');
                $this->db->where('test_code', $test_code);
                $this->db->where('is_active', 1);
                $query = $this->db->get();
                
                if ($query->num_rows() > 0) {
                    $test_ids[] = $query->row()->id;
                } else {
                    // Try to find by test name if code doesn't match
                    $this->db->select('id');
                    $this->db->from('radiology_tests');
                    $this->db->like('test_name', $key);
                    $this->db->where('is_active', 1);
                    $query = $this->db->get();
                    
                    if ($query->num_rows() > 0) {
                        $test_ids[] = $query->row()->id;
                    }
                }
            }
            
            return array_unique($test_ids);
        } catch (Exception $e) {
            log_message('error', 'Error getting test IDs by keys: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Verify patient identity using fingerprint
     */
    /*public function verifyPatientIdentity() {
        $patient_id = $this->input->post('patient_id');
        $scanned_template = $this->input->post('scanned_template');
        
        if (!$patient_id || !$scanned_template) {
            echo json_encode(array('success' => false, 'message' => 'Patient ID and fingerprint template are required'));
            return;
        }
        
        try {
            $patient = $this->radiology_model->getPatientById($patient_id);
            
            if (!$patient || empty($patient['fingerprint_template'])) {
                echo json_encode(array('success' => false, 'message' => 'No fingerprint template found for this patient'));
                return;
            }
            
            // Simple template comparison (in production, use SecuGen SDK)
            $match_score = $this->compareFingerprints($scanned_template, $patient['fingerprint_template']);
            $is_verified = $match_score >= 0.8; // 80% threshold
            
            // Log verification attempt
            $verification_data = array(
                'patient_id' => $patient_id,
                'verification_method' => 'fingerprint',
                'verification_result' => $is_verified ? 'success' : 'failed',
                'match_score' => $match_score,
                'verified_by' => $this->getStaffId(),
                'verification_timestamp' => date('Y-m-d H:i:s'),
                'department' => 'radiology'
            );
            
            $this->radiology_model->logVerificationAttempt($verification_data);
            
            if ($is_verified) {
                echo json_encode(array(
                    'success' => true, 
                    'verified' => true,
                    'message' => 'Patient identity verified successfully',
                    'match_score' => $match_score
                ));
            } else {
                echo json_encode(array(
                    'success' => true, 
                    'verified' => false,
                    'message' => 'Fingerprint verification failed',
                    'match_score' => $match_score
                ));
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error verifying patient identity: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'Error during verification: ' . $e->getMessage()));
        }
    }*/

    /**
     * Get radiology queue/appointments
     */
    public function getRadiologyQueue() {
        try {
            $queue_data = $this->radiology_model->getRadiologyQueue();
            
            $formatted_queue = array();
            foreach ($queue_data as $item) {
                $priority_badge = $this->getPriorityBadge($item['priority']);
                $status_badge = $this->getStatusBadge($item['status']);
                
                $actions = '<div class="btn-group">';
                if ($item['status'] == 'pending') {
                    $actions .= '<button class="btn btn-success btn-sm" onclick="startRadiologyExam(' . $item['order_id'] . ')">
                                    <i class="fa fa-play"></i> Start
                                </button>';
                }
                if ($item['status'] == 'in_progress') {
                    $actions .= '<button class="btn btn-info btn-sm" onclick="completeRadiologyExam(' . $item['order_id'] . ')">
                                    <i class="fa fa-check"></i> Complete
                                </button>';
                }
                $actions .= '<button class="btn btn-primary btn-sm" onclick="viewRadiologyDetails(' . $item['order_id'] . ')">
                                <i class="fa fa-eye"></i> View
                            </button>';
                $actions .= '</div>';
                
                $formatted_queue[] = array(
                    'order_id' => $item['order_id'],
                    'patient_name' => $item['patient_name'],
                    'patient_id' => $item['patient_unique_id'],
                    'tests' => $item['test_names'],
                    'priority' => $priority_badge,
                    'status' => $status_badge,
                    'scheduled_time' => date('H:i', strtotime($item['scheduled_date'])),
                    'referring_doctor' => $item['doctor_name'],
                    'actions' => $actions
                );
            }
            
            echo json_encode(array('data' => $formatted_queue));
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology queue: ' . $e->getMessage());
            echo json_encode(array('data' => array(), 'error' => $e->getMessage()));
        }
    }

    /**
     * Update radiology order status
     */
    public function updateOrderStatus() {
        $order_id = $this->input->post('order_id');
        $new_status = $this->input->post('status');
        $notes = $this->input->post('notes');
        
        if (!$order_id || !$new_status) {
            echo json_encode(array('success' => false, 'message' => 'Order ID and status are required'));
            return;
        }
        
        try {
            $update_data = array(
                'status' => $new_status,
                'updated_at' => date('Y-m-d H:i:s'),
                'updated_by' => $this->getStaffId()
            );
            
            if ($new_status == 'in_progress') {
                $update_data['started_at'] = date('Y-m-d H:i:s');
                $update_data['technician_id'] = $this->getStaffId();
            } elseif ($new_status == 'completed') {
                $update_data['completed_at'] = date('Y-m-d H:i:s');
            }
            
            if ($notes) {
                $update_data['technician_notes'] = $notes;
            }
            
            $result = $this->radiology_model->updateRadiologyOrder($order_id, $update_data);
            
            if ($result) {
                // Log status change
                $this->logRadiologyActivity(null, 'status_changed', 'Order #' . $order_id . ' status changed to ' . $new_status);
                
                echo json_encode(array('success' => true, 'message' => 'Order status updated successfully'));
            } else {
                echo json_encode(array('success' => false, 'message' => 'Failed to update order status'));
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error updating order status: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'Error updating status: ' . $e->getMessage()));
        }
    }

    // =============================================
    // HELPER METHODS
    // =============================================

    /**
     * Get radiology staff (technicians, radiologists)
     */
    private function getRadiologyStaff() {
        try {
            $this->db->select('s.*, r.name as role_name');
            $this->db->from('staff s');
            $this->db->join('staff_roles sr', 'sr.staff_id = s.id', 'left');
            $this->db->join('roles r', 'r.id = sr.role_id', 'left');
            $this->db->where_in('r.name', array('Radiologist', 'Radiology Technician', 'Doctor'));
           // $this->db->or_where('s.department', 'Radiology');
            $this->db->where('s.is_active', 1);
            $this->db->order_by('s.name', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology staff: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get available imaging test types
     */
private function getImagingTypes() {
    try {
        // Get actual tests from database including is_gcc field
        $tests = $this->radiology_model->getRadiologyTests();
        $imaging_types = array();
        
        foreach ($tests as $test) {
            $imaging_types[$test['id']] = array(
                'id' => $test['id'],
                'test_name' => $test['test_name'], // Changed from 'name' to 'test_name'
                'test_code' => $test['test_code'],
                'description' => $test['description'],
                'estimated_duration' => $test['estimated_duration'], // Changed from 'duration'
                'preparation_instructions' => $test['preparation_instructions'], // Changed from 'preparation'
                'category' => $test['category'],
                'price' => $test['price'],
                'is_gcc' => $test['is_gcc'], // ADD THIS FIELD for GCC filtering
                'requires_fasting' => $test['requires_fasting'],
                'requires_contrast' => $test['requires_contrast'],
                'is_active' => $test['is_active']
            );
        }
        
        // If no tests in database, return default set with GCC flags
        if (empty($imaging_types)) {
            return array(
                '1' => array(
                    'test_name' => 'X-Ray Chest',
                    'description' => 'Standard chest radiographic imaging',
                    'estimated_duration' => 15,
                    'preparation_instructions' => 'Remove metal objects',
                    'price' => 2500.00,
                    'is_gcc' => 1, // GCC test
                    'category' => 'X-Ray'
                ),
                '2' => array(
                    'test_name' => 'CT Scan Head',
                    'description' => 'Computed Tomography of head',
                    'estimated_duration' => 30,
                    'preparation_instructions' => 'Fasting may be required',
                    'price' => 8500.00,
                    'is_gcc' => 0, // General test
                    'category' => 'CT'
                ),
                '3' => array(
                    'test_name' => 'MRI Brain',
                    'description' => 'Magnetic Resonance Imaging of brain',
                    'estimated_duration' => 45,
                    'preparation_instructions' => 'Remove all metal objects',
                    'price' => 15000.00,
                    'is_gcc' => 0, // General test
                    'category' => 'MRI'
                ),
                '4' => array(
                    'test_name' => 'Ultrasound Abdomen',
                    'description' => 'Abdominal ultrasonography',
                    'estimated_duration' => 20,
                    'preparation_instructions' => 'Fasting required',
                    'price' => 3500.00,
                    'is_gcc' => 1, // GCC test
                    'category' => 'Ultrasound'
                ),
                '5' => array(
                    'test_name' => 'ECG',
                    'description' => 'Electrocardiogram',
                    'estimated_duration' => 10,
                    'preparation_instructions' => 'None',
                    'price' => 1500.00,
                    'is_gcc' => 1, // GCC test
                    'category' => 'Cardiac'
                )
            );
        }
        
        return $imaging_types;
    } catch (Exception $e) {
        log_message('error', 'Error getting imaging types: ' . $e->getMessage());
        return array();
    }
}

public function getTestsByWorkflow() {
    header('Content-Type: application/json');
    
    $patient_type = $this->input->get('patient_type');
    
    try {
        $tests = $this->radiology_model->getRadiologyTests();
        $filtered_tests = array();
        
        foreach ($tests as $test) {
            // If patient is GCC, only show GCC tests (is_gcc = 1)
            // If patient is General/Specialized, show all tests
            if ($patient_type === 'GCC') {
                if ($test['is_gcc'] == 1) {
                    $filtered_tests[] = $test;
                }
            } else {
                // Show all tests for non-GCC patients
                $filtered_tests[] = $test;
            }
        }
        
        echo json_encode([
            'success' => true,
            'tests' => $filtered_tests,
            'patient_type' => $patient_type,
            'total_tests' => count($filtered_tests)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error getting tests: ' . $e->getMessage()
        ]);
    }
}

public function verify_patient() {
    header('Content-Type: application/json');
    
    $patient_id = $this->input->post('patient_id');
    $action = $this->input->post('action');
    
    try {
        if ($action === 'verify_fingerprint') {
            // This is now primarily for fallback when SecuGen device is unavailable
            $live_template = $this->input->post('live_template');
            $stored_template = $this->input->post('stored_template');
            $method = $this->input->post('method', TRUE) ?: 'secugen_fallback';
            
            // Validate required data
            if (empty($patient_id)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID is required'
                ]);
                return;
            }
            
            if (empty($live_template)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Live fingerprint template is required'
                ]);
                return;
            }
            
            // Get patient info
            $this->db->select('patient_name, patient_workflow, fingerprint_template');
            $this->db->where('id', $patient_id);
            $query = $this->db->get('patients');
            
            if ($query->num_rows() === 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found'
                ]);
                return;
            }
            
            $patient = $query->row_array();
            
            // Use stored template from database if not provided
            if (empty($stored_template)) {
                $stored_template = $patient['fingerprint_template'];
                
                if (empty($stored_template)) {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'No stored fingerprint template found for this patient'
                    ]);
                    return;
                }
            }
            
            // Simple fallback verification (when SecuGen device is unavailable)
            $match_result = $this->performSimpleFallbackMatching($live_template, $stored_template);
            
            $threshold = 70; // Fallback threshold
            $success = $match_result['match_score'] >= $threshold;
            
            // Log verification attempt
            $this->logVerificationAttempt($patient_id, $match_result['match_score'], $success, $method, 
                "Fallback verification. Method: $method");
            
            if ($success) {
                echo json_encode([
                    'status' => 'success',
                    'match_score' => $match_result['match_score'],
                    'message' => 'Fingerprint verification successful (fallback method)',
                    'patient_workflow' => $patient['patient_workflow'] ?? 'general',
                    'verification_method' => $method
                ]);
            } else {
                echo json_encode([
                    'status' => 'failed',
                    'match_score' => $match_result['match_score'],
                    'message' => 'Fingerprint verification failed - insufficient match score (fallback method)',
                    'threshold' => $threshold
                ]);
            }
            
        } elseif ($action === 'record_verification') {
            // Record verification result from SecuGen or other methods
            $verification_result = $this->input->post('verification_result');
            $match_score = $this->input->post('match_score');
            $verification_method = $this->input->post('verification_method');
            $additional_data = $this->input->post('additional_data');
            
            $success = ($verification_result === 'success');
            
            $this->logVerificationAttempt($patient_id, $match_score, $success, $verification_method, $additional_data);
            
            echo json_encode([
                'status' => 'success',
                'message' => 'Verification result recorded'
            ]);
            
        } elseif ($action === 'workflow_bypass') {
            // Record workflow bypass for General patients
            $this->logVerificationAttempt($patient_id, 100, true, 'workflow_bypass', 'General workflow automatic bypass');
            
            echo json_encode([
                'status' => 'success',
                'message' => 'Verification bypassed for General workflow',
                'verification_method' => 'workflow_bypass'
            ]);
            
        } elseif ($action === 'staff_override') {
            // Record staff override
            $override_reason = $this->input->post('override_reason');
            $staff_id = $this->input->post('staff_id');
            $staff_name = $this->input->post('staff_name');
            
            $this->logVerificationAttempt($patient_id, 100, true, 'staff_override', 
                "Staff override by $staff_name (ID: $staff_id). Reason: $override_reason");
            
            echo json_encode([
                'status' => 'success',
                'message' => 'Staff override verification completed',
                'verification_method' => 'staff_override'
            ]);
            
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid verification action'
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Verification error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Verification system error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get patient workflow from database
 */
public function getPatientWorkflow() {
    header('Content-Type: application/json');
    
    $patient_id = $this->input->post('patient_id');
    
    if (!$patient_id) {
        echo json_encode([
            'success' => false,
            'message' => 'Patient ID required'
        ]);
        return;
    }
    
    try {
        $this->db->select('
            id,
            patient_name,
            patient_workflow,
            patient_type,
            fingerprint_template,
            fingerprint_verified,
            created_at
        ');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            echo json_encode([
                'success' => true,
                'patient_id' => $patient_id,
                'patient_name' => $patient['patient_name'],
                'patient_workflow' => $patient['patient_workflow'],
                'patient_type' => $patient['patient_type'],
                'has_fingerprint' => !empty($patient['fingerprint_template']),
                'fingerprint_verified' => $patient['fingerprint_verified'] == 1,
                'created_at' => $patient['created_at']
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Patient not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Update patient workflow
 */
public function updatePatientWorkflow() {
    header('Content-Type: application/json');
    
    $patient_id = $this->input->post('patient_id');
    $workflow = $this->input->post('workflow');
    
    if (!$patient_id || !$workflow) {
        echo json_encode([
            'success' => false,
            'message' => 'Patient ID and workflow are required'
        ]);
        return;
    }
    
    // Validate workflow value
    $valid_workflows = ['gcc', 'general', 'specialized'];
    if (!in_array(strtolower($workflow), $valid_workflows)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid workflow. Must be: gcc, general, or specialized'
        ]);
        return;
    }
    
    try {
        // Check if patient exists
        $this->db->select('id, patient_name');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $check_query = $this->db->get();
        
        if ($check_query->num_rows() === 0) {
            echo json_encode([
                'success' => false,
                'message' => 'Patient not found'
            ]);
            return;
        }
        
        $patient = $check_query->row_array();
        
        // Update workflow
        $update_data = [
            'patient_workflow' => strtolower($workflow),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Set additional fields based on workflow
        if (strtolower($workflow) === 'gcc') {
            $update_data['requires_triage'] = 1;
            $update_data['skip_ic10_diagnosis'] = 1;
            $update_data['gcc_comprehensive_fee'] = 10500.00;
        } else {
            $update_data['requires_triage'] = 0;
            $update_data['skip_ic10_diagnosis'] = 0;
        }
        
        $this->db->where('id', $patient_id);
        $result = $this->db->update('patients', $update_data);
        
        if ($result) {
            // Log the workflow change
            log_message('info', "Patient workflow updated: ID=$patient_id, Name={$patient['patient_name']}, Workflow=$workflow");
            
            echo json_encode([
                'success' => true,
                'message' => 'Patient workflow updated successfully',
                'patient_id' => $patient_id,
                'patient_name' => $patient['patient_name'],
                'new_workflow' => strtolower($workflow)
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to update patient workflow'
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error updating patient workflow: ' . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get workflow statistics
 */
public function getWorkflowStats() {
    header('Content-Type: application/json');
    
    try {
        // Get workflow distribution
        $this->db->select('patient_workflow, COUNT(*) as count');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->group_by('patient_workflow');
        
        $query = $this->db->get();
        $workflow_stats = $query->result_array();
        
        // Get fingerprint stats by workflow
        $this->db->select('
            patient_workflow,
            SUM(CASE WHEN fingerprint_template IS NOT NULL AND fingerprint_template != "" THEN 1 ELSE 0 END) as with_fingerprint,
            SUM(CASE WHEN fingerprint_template IS NULL OR fingerprint_template = "" THEN 1 ELSE 0 END) as without_fingerprint
        ');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->group_by('patient_workflow');
        
        $fingerprint_query = $this->db->get();
        $fingerprint_stats = $fingerprint_query->result_array();
        
        echo json_encode([
            'success' => true,
            'workflow_distribution' => $workflow_stats,
            'fingerprint_stats' => $fingerprint_stats,
            'generated_at' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error getting workflow stats: ' . $e->getMessage()
        ]);
    }
}

/**
 * Enhanced searchPatients method with better workflow handling
 */
public function searchPatientsEnhanced() {
    header('Content-Type: application/json');
    
    $search = $this->input->get('search', TRUE);
    
    try {
        $this->db->select('
            id,
            patient_name,
            patient_unique_id,
            guardian_name,
            gender,
            age,
            month,
            mobileno,
            dob,
            patient_workflow,
            patient_type,
            fingerprint_template,
            fingerprint_verified,
            created_at
        ');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('patient_name', $search);
            $this->db->or_like('patient_unique_id', $search);
            $this->db->or_like('mobileno', $search);
            $this->db->or_like('guardian_name', $search);
            $this->db->group_end();
        }
        
        $this->db->limit(20);
        $this->db->order_by('patient_name');
        
        $query = $this->db->get();
        $patients = $query->result_array();
        
        // Enhanced patient data processing
        foreach ($patients as &$patient) {
            $patient['has_fingerprint'] = !empty($patient['fingerprint_template']);
            
            // Ensure patient_workflow is set with fallback
            if (empty($patient['patient_workflow'])) {
                $patient['patient_workflow'] = 'general';
            }
            
            // Normalize workflow values
            $patient['patient_workflow'] = strtolower(trim($patient['patient_workflow']));
            
            // Set patient_type for compatibility
            $patient['patient_type'] = strtoupper($patient['patient_workflow']);
            
            // Remove sensitive data
            unset($patient['fingerprint_template']);
            
            // Add debug info
            $patient['debug_info'] = [
                'original_workflow' => $patient['patient_workflow'],
                'normalized_type' => $patient['patient_type'],
                'has_fingerprint' => $patient['has_fingerprint']
            ];
        }
        
        echo json_encode($patients);
        
    } catch (Exception $e) {
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

    /**
     * Compare fingerprint templates
     */
    private function compareFingerprints($template1, $template2) {
        // Simplified comparison - in production use SecuGen SDK
        $similarity = similar_text($template1, $template2, $percent);
        return $percent / 100;
    }

    /**
     * Get priority badge HTML
     */
    private function getPriorityBadge($priority) {
        switch ($priority) {
            case 'urgent':
                return '<span class="badge badge-danger">Urgent</span>';
            case 'stat':
                return '<span class="badge badge-warning">STAT</span>';
            case 'routine':
                return '<span class="badge badge-info">Routine</span>';
            default:
                return '<span class="badge badge-secondary">Normal</span>';
        }
    }

    /**
     * Get status badge HTML
     */
    private function getStatusBadge($status) {
        switch ($status) {
            case 'pending':
                return '<span class="badge badge-warning">Pending</span>';
            case 'in_progress':
                return '<span class="badge badge-info">In Progress</span>';
            case 'completed':
                return '<span class="badge badge-success">Completed</span>';
            case 'cancelled':
                return '<span class="badge badge-danger">Cancelled</span>';
            default:
                return '<span class="badge badge-secondary">' . ucfirst($status) . '</span>';
        }
    }

    /**
     * Log radiology activities
     */
    /*private function logRadiologyActivity($patient_id, $activity_type, $description) {
        try {
            $log_data = array(
                'patient_id' => $patient_id,
                'activity_type' => $activity_type,
                'description' => $description,
                'staff_id' => $this->getStaffId(),
                'timestamp' => date('Y-m-d H:i:s'),
                'department' => 'radiology'
            );
            
            $this->radiology_model->logActivity($log_data);
        } catch (Exception $e) {
            log_message('error', 'Error logging radiology activity: ' . $e->getMessage());
        }
    }*/

    /**
     * Test controller functionality
     */
public function test() {
        echo "<h1>Radiology Controller Test</h1>";
        echo "<p>Controller is working properly!</p>";
        echo "<p>Queue integration: " . (class_exists('Queue_model') ? 'ACTIVE' : 'MISSING') . "</p>";
        echo "<p>Notification system: " . (is_writable(APPPATH . 'cache/') ? 'READY' : 'NOT WRITABLE') . "</p>";
        
        // Test database tables
        $tables_to_check = ['radiology_reports', 'radiology_completions'];
        foreach ($tables_to_check as $table) {
            echo "<p>Table $table: " . ($this->db->table_exists($table) ? 'EXISTS' : 'MISSING') . "</p>";
        }
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('radiology');
            echo "<p>Queue statistics: Working</p>";
        } catch (Exception $e) {
            echo "<p>Queue statistics: Error - " . $e->getMessage() . "</p>";
        }
    }
}