<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Pharmacy extends Admin_Controller
{
    public function __construct()
    {
        parent::__construct();
        
        // Load required libraries and models
        $this->load->library(['datatables', 'system_notification', 'customlib']);
        $this->load->model([
            'pharmacy_model', 
            'patient_model', 
            'medicine_category_model',
            'staff_model',
            'transaction_model'
        ]);
        $this->load->helper(['customfield_helper', 'custom']);
        
        // Configuration items
        $this->config->load("payroll");
        $this->marital_status = $this->config->item('marital_status');
        $this->payment_mode = $this->config->item('payment_mode');
        $this->blood_group = $this->config->item('bloodgroup');
        $this->time_format = $this->customlib->getHospitalTimeFormat();
    }

    /**
     * MAIN PHARMACY DASHBOARD
     * URL: /pharmacy or /pharmacy/index
     * VIEW: admin/pharmacy/pharmacyDashboard.php (MAIN COMPREHENSIVE DASHBOARD)
     */
    public function index()
    {
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_view')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Pharmacy Management System - Dashboard';
        $data['medicineCategory'] = $this->medicine_category_model->getMedicineCategory();
        $data['payment_mode'] = $this->payment_mode;
        
        // Get dashboard statistics
        $data['dashboard_stats'] = $this->pharmacy_model->getPharmacyDashboardStats();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/pharmacyDashboard', $data);  // MAIN DASHBOARD VIEW
        $this->load->view('layout/footer', $data);
    }

    /**
     * MEDICINE STOCK MANAGEMENT PAGE
     * URL: /pharmacy/medicines or /pharmacy/medicineStock
     * VIEW: admin/pharmacy/medicineStock.php
     */
    public function medicineStock()
    {
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_view')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Medicine Stock Management';
        $data['medicineCategory'] = $this->medicine_category_model->getMedicineCategory();
        $data['unitname'] = $this->pharmacy_model->getpharmacyunit();
        $data['company'] = $this->pharmacy_model->getcomapnyname();
        
        // Get medicine inventory with stock info
        $result = $this->pharmacy_model->getPharmacy();
        $resultlist = array();
        $i = 0;
        
        foreach ($result as $pharmacy) {
            $available_qty = $this->pharmacy_model->getMedicineAvailableQuantity($pharmacy['id']);
            $resultlist[$i] = $pharmacy;
            $resultlist[$i]["total_qty"] = $available_qty['total_qty'];
            $i++;
        }
        
        $data['resultlist'] = $resultlist;
        $data['result'] = $result;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/medicineStock', $data);
        $this->load->view('layout/footer', $data);
    }
    
    /**
 * Dispense prescription medications
 */
public function dispensePrescription()
{
    if (!$this->rbac->hasPrivilege('pharmacy', 'can_add')) {
        access_denied();
    }

    $prescription_id = $this->input->post('prescription_id');
    $items_to_dispense = $this->input->post('items'); // Array of items with quantities
    
    if (empty($prescription_id) || empty($items_to_dispense)) {
        echo json_encode(['status' => 'error', 'message' => 'Missing required data']);
        return;
    }

    $this->db->trans_start();

    $total_dispensed_cost = 0;
    $all_items_dispensed = true;

    foreach ($items_to_dispense as $item) {
        $item_id = $item['item_id'];
        $quantity_to_dispense = $item['quantity'];
        $batch_id = $item['batch_id'] ?? null;
        
        // Get prescription item details
        $prescription_item = $this->pharmacy_model->getPrescriptionItem($item_id);
        
        if (!$prescription_item) {
            continue;
        }

        // Check stock availability
        $available_stock = $this->pharmacy_model->getAvailableStock($prescription_item['drug_id'], $batch_id);
        
        if ($available_stock < $quantity_to_dispense) {
            echo json_encode([
                'status' => 'error', 
                'message' => 'Insufficient stock for ' . $prescription_item['drug_name']
            ]);
            return;
        }

        // Calculate dispensing cost
        $unit_cost = $prescription_item['unit_price'];
        $total_item_cost = $quantity_to_dispense * $unit_cost;
        $total_dispensed_cost += $total_item_cost;

        // Update prescription item
        $new_dispensed_quantity = $prescription_item['quantity_dispensed'] + $quantity_to_dispense;
        $item_status = ($new_dispensed_quantity >= $prescription_item['quantity_prescribed']) ? 'dispensed' : 'partially_dispensed';
        
        if ($item_status !== 'dispensed') {
            $all_items_dispensed = false;
        }

        $this->pharmacy_model->updatePrescriptionItem($item_id, array(
            'quantity_dispensed' => $new_dispensed_quantity,
            'status' => $item_status,
            'dispensed_by' => $this->session->userdata('admin_id'),
            'dispensed_at' => date('Y-m-d H:i:s')
        ));

        // Log dispensing
        $dispensing_log = array(
            'prescription_id' => $prescription_id,
            'prescription_item_id' => $item_id,
            'patient_id' => $prescription_item['patient_id'],
            'drug_id' => $prescription_item['drug_id'],
            'quantity_dispensed' => $quantity_to_dispense,
            'unit_price' => $unit_cost,
            'total_cost' => $total_item_cost,
            'batch_number' => $batch_id,
            'dispensed_by' => $this->session->userdata('admin_id'),
            'dispensed_at' => date('Y-m-d H:i:s'),
            'payment_method' => $this->input->post('payment_method') ?: 'cash',
            'payment_status' => 'paid',
            'stock_reduced' => 1
        );

        $this->pharmacy_model->addDispensingLog($dispensing_log);

        // Reduce stock
        $this->pharmacy_model->reduceStock($prescription_item['drug_id'], $quantity_to_dispense, $batch_id);
    }

    // Update prescription status
    $prescription_status = $all_items_dispensed ? 'dispensed' : 'partially_dispensed';
    $prescription_update = array(
        'status' => $prescription_status,
        'dispensed_cost' => $total_dispensed_cost,
        'dispensed_by' => $this->session->userdata('admin_id'),
        'dispensed_at' => date('Y-m-d H:i:s')
    );

    $this->pharmacy_model->updatePrescription($prescription_id, $prescription_update);

    $this->db->trans_complete();

    if ($this->db->trans_status() === FALSE) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to dispense prescription']);
    } else {
        echo json_encode([
            'status' => 'success',
            'message' => 'Prescription dispensed successfully',
            'dispensed_cost' => $total_dispensed_cost,
            'prescription_status' => $prescription_status
        ]);
    }
}

/**
 * Get pending prescriptions for pharmacy
 */
public function getPendingPrescriptions()
{
    if (!$this->rbac->hasPrivilege('pharmacy', 'can_view')) {
        access_denied();
    }

    $prescriptions = $this->pharmacy_model->getPendingPrescriptions();
    
    // Enrich with patient and doctor details
    foreach ($prescriptions as &$prescription) {
        $prescription['items'] = $this->pharmacy_model->getPrescriptionItems($prescription['id']);
        $prescription['patient_details'] = $this->patient_model->getPatientById($prescription['patient_id']);
        $prescription['doctor_details'] = $this->staff_model->getStaffById($prescription['doctor_id']);
    }

    echo json_encode(['status' => 'success', 'prescriptions' => $prescriptions]);
}

/**
 * Generate prescription report
 */
public function generatePrescriptionReport()
{
    if (!$this->rbac->hasPrivilege('reports', 'can_view')) {
        access_denied();
    }

    $start_date = $this->input->post('start_date') ?: date('Y-m-01');
    $end_date = $this->input->post('end_date') ?: date('Y-m-d');
    $doctor_id = $this->input->post('doctor_id');
    $status = $this->input->post('status');

    $report_data = $this->consultation_model->getPrescriptionReport($start_date, $end_date, $doctor_id, $status);
    
    echo json_encode(['status' => 'success', 'report' => $report_data]);
}

    /**
     * PHARMACY BILLING PAGE - FIXED METHOD NAME
     * URL: /pharmacy/billing (NEW PREFERRED URL)
     * VIEW: admin/pharmacy/pharmacyBill.php
     */
    public function billing()
    {
        if (!$this->rbac->hasPrivilege('pharmacy_bill', 'can_view')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Pharmacy Billing';
        $data['doctors'] = $this->staff_model->getStaffbyrole(3);
        $data['medicineCategory'] = $this->medicine_category_model->getMedicineCategory();
        $data['patients'] = $this->patient_model->getPatientListall();
        $data['payment_mode'] = $this->payment_mode;
        $data['marital_status'] = $this->marital_status;
        $data['bloodgroup'] = $this->blood_group;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/pharmacyBill', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * LEGACY BILLING METHOD - for backward compatibility
     * URL: /pharmacy/bill (LEGACY URL)
     * ACTION: Redirects to new billing() method OR calls it directly
     */
    public function bill()
    {
        // Option 1: Redirect to new URL (recommended)
        // redirect('pharmacy/billing');
        
        // Option 2: Call the billing method directly (keeps same URL)
        $this->billing();
    }

    /**
     * PHARMACY SETTINGS PAGE
     * URL: /pharmacy/settings
     * VIEW: admin/pharmacy/pharmacySettings.php
     */
    public function settings()
    {
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_edit')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Pharmacy Settings & Configuration';
        $data['medicineCategory'] = $this->medicine_category_model->getMedicineCategory();
        $data['unitname'] = $this->pharmacy_model->getpharmacyunit();
        $data['company'] = $this->pharmacy_model->getcomapnyname();
        $data['payment_mode'] = $this->payment_mode;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/pharmacySettings', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * PHARMACY REPORTS PAGE - RENAMED METHOD
     * URL: /pharmacy/reports (NEW PREFERRED URL)
     * VIEW: admin/pharmacy/billReport.php
     */
    public function reports()
    {
        if (!$this->rbac->hasPrivilege('pharmacy_bill', 'can_view')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'Reports');
        $this->session->set_userdata('sub_menu', 'reports/pharmacy');
        $this->session->set_userdata('subsub_menu', 'reports/pharmacy/billreport');
        
        $data['title'] = 'Pharmacy Reports';
        $data['payment_mode'] = $this->payment_mode;
        $data['searchlist'] = $this->config->item('search_type');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/billReport', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * LEGACY REPORTS METHOD - for backward compatibility
     * URL: /pharmacy/billreport (LEGACY URL)
     * ACTION: Redirects to new reports() method OR calls it directly
     */
    public function billreport()
    {
        // Option 1: Redirect to new URL (recommended)
        // redirect('pharmacy/reports');
        
        // Option 2: Call the reports method directly (keeps same URL)
        $this->reports();
    }

    // ==========================================
    // AJAX METHODS (No view changes needed)
    // ==========================================

    /**
     * Get dashboard statistics via AJAX
     */
    public function getDashboardStats()
    {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->pharmacy_model->getPharmacyDashboardStats();
            echo json_encode(['status' => 1, 'stats' => $stats]);
        } catch (Exception $e) {
            log_message('error', 'Pharmacy dashboard stats error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error loading dashboard statistics']);
        }
    }

    /**
     * Search patients for billing
     */
    public function searchPatients()
    {
        header('Content-Type: application/json');
        
        $search_term = $this->input->get('q');
        $page = $this->input->get('page', true) ?: 1;
        
        if (empty($search_term)) {
            echo json_encode(['patients' => [], 'total_count' => 0]);
            return;
        }
        
        try {
            $patients = $this->patient_model->searchPatientsForPharmacy($search_term, $page);
            echo json_encode(['patients' => $patients['data'], 'total_count' => $patients['total']]);
        } catch (Exception $e) {
            log_message('error', 'Patient search error: ' . $e->getMessage());
            echo json_encode(['patients' => [], 'total_count' => 0]);
        }
    }

    /**
     * Search medicines for billing
     */
    public function searchMedicines()
    {
        header('Content-Type: application/json');
        
        $search_term = $this->input->get('q');
        
        if (empty($search_term)) {
            echo json_encode(['medicines' => []]);
            return;
        }
        
        try {
            $medicines = $this->pharmacy_model->searchMedicines($search_term);
            echo json_encode(['medicines' => $medicines]);
        } catch (Exception $e) {
            log_message('error', 'Medicine search error: ' . $e->getMessage());
            echo json_encode(['medicines' => []]);
        }
    }

    /**
     * Get medicine batches for dropdown
     */
    public function getMedicineBatches()
    {
        header('Content-Type: application/json');
        
        $medicine_id = $this->input->post('medicine_id');
        
        if (empty($medicine_id)) {
            echo json_encode(['status' => 0, 'message' => 'Medicine ID required']);
            return;
        }
        
        try {
            $batches = $this->pharmacy_model->getMedicineBatches($medicine_id);
            echo json_encode(['status' => 1, 'batches' => $batches]);
        } catch (Exception $e) {
            log_message('error', 'Get medicine batches error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error loading batches']);
        }
    }

    /**
     * Get batch details for pricing
     */
    public function getBatchDetails()
    {
        header('Content-Type: application/json');
        
        $batch_id = $this->input->post('batch_id');
        
        if (empty($batch_id)) {
            echo json_encode(['status' => 0, 'message' => 'Batch ID required']);
            return;
        }
        
        try {
            $batch = $this->pharmacy_model->getBatchDetails($batch_id);
            echo json_encode(['status' => 1, 'batch' => $batch]);
        } catch (Exception $e) {
            log_message('error', 'Get batch details error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error loading batch details']);
        }
    }
    
    /**
 * NEW METHOD: Search drugs from medicines table for prescription selection
 * URL: /admin/pharmacy/searchDrugsForPrescription
 */
public function searchDrugsForPrescription()
{
    header('Content-Type: application/json');
    
    // Get search parameters
    $search_term = $this->input->post('search_term') ?: $this->input->post('search');
    $category_id = $this->input->post('category_id');
    $search_type = $this->input->post('search_type') ?: 'name';
    
    // Validate search term
    if (empty($search_term) || strlen(trim($search_term)) < 2) {
        echo json_encode([
            'status' => 0,
            'message' => 'Please enter at least 2 characters to search',
            'drugs' => []
        ]);
        return;
    }
    
    try {
        // Search in medicines table with your schema
        $this->db->select('m.*, mc.medicine_category as category_name');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        
        // Only get active medicines
        $this->db->where('m.is_active', 'yes');
        
        // Search conditions
        $this->db->group_start();
        $this->db->like('m.medicine_name', $search_term);
        $this->db->or_like('m.medicine_company', $search_term);
        $this->db->or_like('m.medicine_composition', $search_term);
        $this->db->group_end();
        
        // Category filter if provided
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(50); // Limit results for performance
        
        $medicines = $this->db->get()->result();
        
        // Format response for prescription interface
        $drugs = [];
        foreach ($medicines as $medicine) {
            // Since this is just a medicines catalog, we'll show all as available
            // You can add stock checking logic here if you have a separate stock table
            
            $drugs[] = [
                'id' => $medicine->id,
                'medicine_name' => $medicine->medicine_name,
                'name' => $medicine->medicine_name, // Alternative key
                'company_name' => $medicine->medicine_company ?: '',
                'brand_name' => $medicine->medicine_company ?: '', // Alternative key
                'composition' => $medicine->medicine_composition ?: '',
                'generic_name' => $medicine->medicine_composition ?: '', // Alternative key
                'unit' => $medicine->unit ?: 'Tablet',
                'unit_name' => $medicine->unit ?: 'Tablet', // Alternative key
                'strength' => $medicine->unit ?: 'Tablet',
                'form' => $medicine->unit ?: 'Tablet',
                'category' => $medicine->category_name ?: '',
                'category_id' => $medicine->medicine_category_id,
                'is_active' => $medicine->is_active,
                'available_qty' => 'Available', // Since no stock table, show as available
                'stock_status' => 'available',
                'stock_class' => 'text-success',
                'stock_text' => 'Available',
                'available' => true,
                'created_at' => $medicine->created_at,
                'updated_at' => $medicine->updated_at
            ];
        }
        
        echo json_encode([
            'status' => 1,
            'message' => count($drugs) . ' medicines found for prescription',
            'drugs' => $drugs,
            'search_term' => $search_term,
            'total_found' => count($drugs)
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Prescription medicines search error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading medicines. Please try again.',
            'drugs' => [],
            'error' => $e->getMessage() // Include error for debugging
        ]);
    }
}

/**
 * NEW METHOD: Browse all medicines for prescription with pagination
 * URL: /admin/pharmacy/browseDrugsForPrescription
 */
public function browseDrugsForPrescription()
{
    header('Content-Type: application/json');
    
    try {
        $category_id = $this->input->post('category_id');
        $page = (int)($this->input->post('page') ?: 1);
        $limit = 20; // Medicines per page
        $offset = ($page - 1) * $limit;
        
        // Build query for medicines table
        $this->db->select('m.*, mc.medicine_category as category_name');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        
        // Only get active medicines
        $this->db->where('m.is_active', 'yes');
        
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit($limit, $offset);
        
        $medicines = $this->db->get()->result();
        
        // Get total count for pagination
        $this->db->select('COUNT(m.id) as total');
        $this->db->from('medicines m');
        $this->db->where('m.is_active', 'yes');
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        $total_result = $this->db->get()->row();
        $total_medicines = $total_result ? $total_result->total : 0;
        
        // Format medicines for prescription
        $drugs = [];
        foreach ($medicines as $medicine) {
            $drugs[] = [
                'id' => $medicine->id,
                'medicine_name' => $medicine->medicine_name,
                'company_name' => $medicine->medicine_company ?: '',
                'composition' => $medicine->medicine_composition ?: '',
                'unit' => $medicine->unit ?: 'Tablet',
                'category' => $medicine->category_name ?: '',
                'category_id' => $medicine->medicine_category_id,
                'is_active' => $medicine->is_active,
                'available_qty' => 'Available',
                'stock_status' => 'available',
                'stock_class' => 'text-success',
                'available' => true
            ];
        }
        
        echo json_encode([
            'status' => 1,
            'drugs' => $drugs,
            'pagination' => [
                'current_page' => $page,
                'total_medicines' => $total_medicines,
                'total_pages' => ceil($total_medicines / $limit),
                'per_page' => $limit
            ]
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Browse medicines for prescription error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading medicines: ' . $e->getMessage(),
            'drugs' => []
        ]);
    }
}

/**
 * NEW METHOD: Get medicine categories for prescription filter
 * URL: /admin/pharmacy/getDrugCategories
 * (This method remains the same as it uses medicine_category table)
 */
public function getDrugCategories()
{
    header('Content-Type: application/json');
    
    try {
        $categories = $this->medicine_category_model->getMedicineCategory();
        
        echo json_encode([
            'status' => 1,
            'categories' => $categories,
            'message' => 'Categories loaded successfully'
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Get medicine categories error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading categories',
            'categories' => []
        ]);
    }
}

/**
 * NEW METHOD: Get medicine details from medicines table
 * URL: /admin/pharmacy/getDrugDetails
 */
public function getDrugDetails()
{
    header('Content-Type: application/json');
    
    $medicine_id = $this->input->post('drug_id') ?: $this->input->get('drug_id') ?: $this->input->post('medicine_id');
    
    if (empty($medicine_id)) {
        echo json_encode([
            'status' => 0,
            'message' => 'Medicine ID required'
        ]);
        return;
    }
    
    try {
        // Get medicine details from medicines table
        $this->db->select('m.*, mc.medicine_category as category_name');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        $this->db->where('m.id', $medicine_id);
        $this->db->where('m.is_active', 'yes');
        $medicine = $this->db->get()->row();
        
        if (!$medicine) {
            echo json_encode([
                'status' => 0,
                'message' => 'Medicine not found or inactive'
            ]);
            return;
        }
        
        $medicine_details = [
            'id' => $medicine->id,
            'medicine_name' => $medicine->medicine_name,
            'company_name' => $medicine->medicine_company,
            'composition' => $medicine->medicine_composition,
            'unit' => $medicine->unit,
            'category' => $medicine->category_name,
            'category_id' => $medicine->medicine_category_id,
            'is_active' => $medicine->is_active,
            'created_at' => $medicine->created_at,
            'updated_at' => $medicine->updated_at,
            'available_qty' => 'Available',
            'stock_status' => 'available'
        ];
        
        echo json_encode([
            'status' => 1,
            'medicine' => $medicine_details, // Using 'medicine' instead of 'drug'
            'drug' => $medicine_details, // Also provide as 'drug' for compatibility
            'message' => 'Medicine details loaded successfully'
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Get medicine details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading medicine details'
        ]);
    }
}

/**
 * NEW METHOD: Quick search for prescription autocomplete
 * URL: /admin/pharmacy/quickDrugSearch
 */
public function quickDrugSearch()
{
    header('Content-Type: application/json');
    
    $query = $this->input->get('q') ?: $this->input->post('q');
    
    if (empty($query) || strlen(trim($query)) < 2) {
        echo json_encode([
            'suggestions' => []
        ]);
        return;
    }
    
    try {
        $this->db->select('id, medicine_name, medicine_company, medicine_composition, unit');
        $this->db->from('medicines');
        $this->db->where('is_active', 'yes');
        $this->db->group_start();
        $this->db->like('medicine_name', $query);
        $this->db->or_like('medicine_company', $query);
        $this->db->or_like('medicine_composition', $query);
        $this->db->group_end();
        $this->db->order_by('medicine_name', 'ASC');
        $this->db->limit(10);
        
        $results = $this->db->get()->result();
        
        $suggestions = [];
        foreach ($results as $result) {
            $suggestions[] = [
                'id' => $result->id,
                'value' => $result->medicine_name,
                'label' => $result->medicine_name . 
                          ($result->medicine_company ? ' (' . $result->medicine_company . ')' : '') .
                          ($result->unit ? ' - ' . $result->unit : '') .
                          ($result->medicine_composition ? ' | ' . substr($result->medicine_composition, 0, 50) : '')
            ];
        }
        
        echo json_encode([
            'suggestions' => $suggestions
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Quick medicine search error: ' . $e->getMessage());
        echo json_encode([
            'suggestions' => []
        ]);
    }
}

/**
 * NEW METHOD: Get all active medicines count (utility method)
 * URL: /admin/pharmacy/getMedicinesCount
 */
public function getMedicinesCount()
{
    header('Content-Type: application/json');
    
    try {
        // Total medicines count
        $this->db->select('COUNT(*) as total');
        $this->db->from('medicines');
        $this->db->where('is_active', 'yes');
        $total_result = $this->db->get()->row();
        $total_medicines = $total_result ? $total_result->total : 0;
        
        // Count by category
        $this->db->select('mc.medicine_category, COUNT(m.id) as count');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.medicine_category_id');
        $this->db->order_by('count', 'DESC');
        $category_counts = $this->db->get()->result();
        
        echo json_encode([
            'status' => 1,
            'total_medicines' => $total_medicines,
            'category_breakdown' => $category_counts
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Get medicines count error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error getting medicines count'
        ]);
    }
}

/**
 * NEW METHOD: Search medicines by specific field
 * URL: /admin/pharmacy/searchMedicinesByField
 */
public function searchMedicinesByField()
{
    header('Content-Type: application/json');
    
    $search_field = $this->input->post('field'); // 'name', 'company', 'composition'
    $search_value = $this->input->post('value');
    $exact_match = $this->input->post('exact_match') === 'true';
    
    if (empty($search_field) || empty($search_value)) {
        echo json_encode([
            'status' => 0,
            'message' => 'Search field and value required',
            'medicines' => []
        ]);
        return;
    }
    
    try {
        $this->db->select('m.*, mc.medicine_category as category_name');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        $this->db->where('m.is_active', 'yes');
        
        // Map field names to database columns
        $field_map = [
            'name' => 'medicine_name',
            'company' => 'medicine_company', 
            'composition' => 'medicine_composition',
            'unit' => 'unit'
        ];
        
        $db_field = $field_map[$search_field] ?? 'medicine_name';
        
        if ($exact_match) {
            $this->db->where($db_field, $search_value);
        } else {
            $this->db->like($db_field, $search_value);
        }
        
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(100);
        
        $medicines = $this->db->get()->result();
        
        echo json_encode([
            'status' => 1,
            'medicines' => $medicines,
            'search_field' => $search_field,
            'search_value' => $search_value,
            'exact_match' => $exact_match,
            'count' => count($medicines)
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Search medicines by field error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error searching medicines: ' . $e->getMessage(),
            'medicines' => []
        ]);
    }
}

public function testMedicinesTable()
{
    header('Content-Type: application/json');
    
    try {
        // Check if medicines table exists and get structure
        $table_exists = $this->db->table_exists('medicines');
        
        if (!$table_exists) {
            echo json_encode([
                'status' => 0,
                'message' => 'Medicines table does not exist',
                'table_exists' => false
            ]);
            return;
        }
        
        // Get table structure
        $fields = $this->db->field_data('medicines');
        
        // Count total medicines
        $this->db->select('COUNT(*) as total');
        $this->db->from('medicines');
        $total_result = $this->db->get()->row();
        $total_medicines = $total_result ? $total_result->total : 0;
        
        // Count active medicines
        $this->db->select('COUNT(*) as active_total');
        $this->db->from('medicines');
        $this->db->where('is_active', 'yes');
        $active_result = $this->db->get()->row();
        $active_medicines = $active_result ? $active_result->active_total : 0;
        
        // Get sample medicines
        $this->db->select('*');
        $this->db->from('medicines');
        $this->db->where('is_active', 'yes');
        $this->db->limit(5);
        $sample_medicines = $this->db->get()->result();
        
        // Check medicine_category table
        $category_table_exists = $this->db->table_exists('medicine_category');
        $categories = [];
        if ($category_table_exists) {
            $this->db->select('*');
            $this->db->from('medicine_category');
            $this->db->limit(5);
            $categories = $this->db->get()->result();
        }
        
        // Test a simple search
        $test_search_results = [];
        if ($active_medicines > 0) {
            $this->db->select('*');
            $this->db->from('medicines');
            $this->db->where('is_active', 'yes');
            $this->db->like('medicine_name', 'a'); // Search for medicines containing 'a'
            $this->db->limit(3);
            $test_search_results = $this->db->get()->result();
        }
        
        echo json_encode([
            'status' => 1,
            'message' => 'Medicines table test completed',
            'table_info' => [
                'table_exists' => $table_exists,
                'total_medicines' => $total_medicines,
                'active_medicines' => $active_medicines,
                'fields' => $fields
            ],
            'sample_medicines' => $sample_medicines,
            'categories_info' => [
                'table_exists' => $category_table_exists,
                'sample_categories' => $categories
            ],
            'test_search' => [
                'search_term' => 'medicines containing "a"',
                'results_count' => count($test_search_results),
                'results' => $test_search_results
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Error testing medicines table: ' . $e->getMessage(),
            'error_details' => $e->getTraceAsString()
        ]);
    }
}

/**
 * TEMPORARY TEST METHOD: Test prescription search specifically
 * URL: /admin/pharmacy/testPrescriptionSearch
 */
public function testPrescriptionSearch()
{
    header('Content-Type: application/json');
    
    try {
        // Test search with a common term
        $search_term = 'a'; // Simple search that should return results
        
        $this->db->select('m.*, mc.medicine_category as category_name');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        $this->db->where('m.is_active', 'yes');
        $this->db->group_start();
        $this->db->like('m.medicine_name', $search_term);
        $this->db->or_like('m.medicine_company', $search_term);
        $this->db->or_like('m.medicine_composition', $search_term);
        $this->db->group_end();
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(10);
        
        $medicines = $this->db->get()->result();
        
        // Format like the real search
        $drugs = [];
        foreach ($medicines as $medicine) {
            $drugs[] = [
                'id' => $medicine->id,
                'medicine_name' => $medicine->medicine_name,
                'company_name' => $medicine->medicine_company ?: '',
                'composition' => $medicine->medicine_composition ?: '',
                'unit' => $medicine->unit ?: 'Tablet',
                'category' => $medicine->category_name ?: '',
                'available' => true
            ];
        }
        
        echo json_encode([
            'status' => 1,
            'message' => 'Test search completed successfully',
            'search_term' => $search_term,
            'results_count' => count($drugs),
            'drugs' => $drugs,
            'sql_query' => $this->db->last_query()
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Test search failed: ' . $e->getMessage(),
            'error_details' => $e->getTraceAsString()
        ]);
    }
}

public function saveBill()
{
    header('Content-Type: application/json');
    
    $this->form_validation->set_rules('patient_id', 'Patient', 'required');
    $this->form_validation->set_rules('bill_date', 'Bill Date', 'required');
    
    if ($this->form_validation->run() == false) {
        echo json_encode([
            'status' => 0,
            'message' => 'Validation failed: ' . validation_errors()
        ]);
        return;
    }
    
    try {
        // Use the correct table name: pharmacy_bill_basic
        $bill_data = [
            'date' => $this->input->post('bill_date'),
            'patient_id' => $this->input->post('patient_id'),
            'total' => $this->input->post('total') ?: 0,
            'discount_percentage' => $this->input->post('discount_percentage') ?: 0,
            'discount' => $this->input->post('discount') ?: 0,
            'net_amount' => $this->input->post('net_amount') ?: 0,
            'generated_by' => $this->session->userdata('admin_id') ?: 1,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Direct database insert to correct table
        $this->db->insert('pharmacy_bill_basic', $bill_data);
        $bill_id = $this->db->insert_id();
        
        if ($bill_id) {
            echo json_encode([
                'status' => 1,
                'message' => 'Bill saved successfully',
                'bill_id' => $bill_id
            ]);
        } else {
            echo json_encode([
                'status' => 0,
                'message' => 'Failed to save bill'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
}

/**
 * DEBUGGING METHOD: Check pharmacy bill tables and structure
 * URL: /admin/pharmacy/debugBillTables
 */
public function debugBillTables()
{
    header('Content-Type: application/json');
    
    try {
        $debug_info = [];
        
        // Check common pharmacy bill table names
        $possible_tables = [
            'pharmacy_bills',
            'pharmacy_bill',
            'bills',
            'pharmacy_billing',
            'pharmacy_sales',
            'pharmacy_transactions'
        ];
        
        $existing_tables = [];
        foreach ($possible_tables as $table) {
            if ($this->db->table_exists($table)) {
                $existing_tables[] = $table;
                
                // Get table structure
                $fields = $this->db->field_data($table);
                $debug_info[$table] = [
                    'exists' => true,
                    'fields' => $fields,
                    'record_count' => $this->db->count_all($table)
                ];
                
                // Get sample records
                $this->db->limit(3);
                $sample_records = $this->db->get($table)->result();
                $debug_info[$table]['sample_records'] = $sample_records;
            } else {
                $debug_info[$table] = ['exists' => false];
            }
        }
        
        // Check if pharmacy_model exists and its methods
        $model_info = [
            'model_loaded' => isset($this->pharmacy_model),
            'model_methods' => []
        ];
        
        if (isset($this->pharmacy_model)) {
            $model_info['model_methods'] = get_class_methods($this->pharmacy_model);
        }
        
        echo json_encode([
            'status' => 1,
            'existing_tables' => $existing_tables,
            'table_details' => $debug_info,
            'model_info' => $model_info,
            'database_name' => $this->db->database
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Error debugging tables: ' . $e->getMessage(),
            'error_details' => $e->getTraceAsString()
        ]);
    }
}

/**
 * DEBUGGING METHOD: Test direct database insert
 * URL: /admin/pharmacy/testDirectBillInsert
 */
public function testDirectBillInsert()
{
    header('Content-Type: application/json');
    
    try {
        // Determine the correct table name
        $table_name = null;
        $possible_tables = ['pharmacy_bills', 'pharmacy_bill', 'bills', 'pharmacy_billing'];
        
        foreach ($possible_tables as $table) {
            if ($this->db->table_exists($table)) {
                $table_name = $table;
                break;
            }
        }
        
        if (!$table_name) {
            echo json_encode([
                'status' => 0,
                'message' => 'No pharmacy bill table found',
                'checked_tables' => $possible_tables
            ]);
            return;
        }
        
        // Test direct insert
        $test_data = [
            'patient_id' => 1, // Adjust as needed
            'bill_date' => date('Y-m-d'),
            'total_amount' => 100.00,
            'discount_percentage' => 0,
            'payment_method' => 'cash',
            'created_by' => $this->session->userdata('admin_id') ?: 1,
            'created_at' => date('Y-m-d H:i:s'),
            'status' => 'completed'
        ];
        
        $this->db->insert($table_name, $test_data);
        $insert_id = $this->db->insert_id();
        $db_error = $this->db->error();
        
        if ($insert_id) {
            // Verify the record was inserted
            $this->db->where('id', $insert_id);
            $inserted_record = $this->db->get($table_name)->row();
            
            echo json_encode([
                'status' => 1,
                'message' => 'Direct insert successful',
                'table_used' => $table_name,
                'insert_id' => $insert_id,
                'inserted_record' => $inserted_record,
                'test_data' => $test_data
            ]);
        } else {
            echo json_encode([
                'status' => 0,
                'message' => 'Direct insert failed',
                'table_used' => $table_name,
                'db_error' => $db_error,
                'last_query' => $this->db->last_query()
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Test insert failed: ' . $e->getMessage(),
            'error_details' => $e->getTraceAsString()
        ]);
    }
}

/**
 * DEBUGGING METHOD: Check pharmacy model methods
 * URL: /admin/pharmacy/debugPharmacyModel
 */
public function debugPharmacyModel()
{
    header('Content-Type: application/json');
    
    try {
        $model_debug = [
            'model_loaded' => isset($this->pharmacy_model),
            'model_class' => null,
            'model_methods' => [],
            'model_file_exists' => false
        ];
        
        if (isset($this->pharmacy_model)) {
            $model_debug['model_class'] = get_class($this->pharmacy_model);
            $model_debug['model_methods'] = get_class_methods($this->pharmacy_model);
            
            // Check if specific methods exist
            $required_methods = ['saveBill', 'getBillById', 'getBillByNumber'];
            $model_debug['required_methods'] = [];
            
            foreach ($required_methods as $method) {
                $model_debug['required_methods'][$method] = method_exists($this->pharmacy_model, $method);
            }
        }
        
        // Check if model file exists
        $model_file = APPPATH . 'models/Pharmacy_model.php';
        $model_debug['model_file_exists'] = file_exists($model_file);
        $model_debug['model_file_path'] = $model_file;
        
        echo json_encode([
            'status' => 1,
            'model_debug' => $model_debug
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Error debugging model: ' . $e->getMessage()
        ]);
    }
}

/**
 * DEBUGGING METHOD: Show recent database queries
 * URL: /admin/pharmacy/showRecentQueries
 */
public function showRecentQueries()
{
    header('Content-Type: application/json');
    
    // Enable query logging temporarily
    $this->db->save_queries = TRUE;
    
    try {
        // Perform a test query
        $this->db->select('COUNT(*) as total');
        $this->db->from('medicines');
        $test_result = $this->db->get()->row();
        
        // Get all queries
        $queries = $this->db->queries;
        
        echo json_encode([
            'status' => 1,
            'recent_queries' => $queries,
            'last_query' => $this->db->last_query(),
            'test_result' => $test_result,
            'database_info' => [
                'database' => $this->db->database,
                'hostname' => $this->db->hostname,
                'username' => $this->db->username
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Error showing queries: ' . $e->getMessage()
        ]);
    }
}

/**
 * SIMPLE ALTERNATIVE: Create a basic bill saving method that works
 * URL: /admin/pharmacy/saveBillSimple
 */
public function saveBillSimple()
{
    header('Content-Type: application/json');
    
    try {
        // Get form data
        $patient_id = $this->input->post('patient_id');
        $bill_date = $this->input->post('bill_date');
        $medicines = $this->input->post('medicines'); // This might be JSON
        $total_amount = $this->input->post('total_amount') ?: 0;
        $discount = $this->input->post('discount') ?: 0;
        $payment_method = $this->input->post('payment_method') ?: 'cash';
        
        // Validate required fields
        if (empty($patient_id) || empty($bill_date)) {
            echo json_encode([
                'status' => 0,
                'message' => 'Patient ID and Bill Date are required'
            ]);
            return;
        }
        
        // Find the correct bill table
        $bill_table = null;
        $possible_tables = ['pharmacy_bills', 'pharmacy_bill', 'bills'];
        
        foreach ($possible_tables as $table) {
            if ($this->db->table_exists($table)) {
                $bill_table = $table;
                break;
            }
        }
        
        if (!$bill_table) {
            // Create a basic pharmacy_bills table if none exists
            $sql = "CREATE TABLE IF NOT EXISTS `pharmacy_bills` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` int(11) NOT NULL,
                `bill_date` date NOT NULL,
                `total_amount` decimal(10,2) DEFAULT 0.00,
                `discount` decimal(10,2) DEFAULT 0.00,
                `net_amount` decimal(10,2) DEFAULT 0.00,
                `payment_method` varchar(50) DEFAULT 'cash',
                `medicines_data` text,
                `created_by` int(11) DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
            
            $this->db->query($sql);
            $bill_table = 'pharmacy_bills';
        }
        
        // Calculate net amount
        $net_amount = $total_amount - $discount;
        
        // Prepare bill data
        $bill_data = [
            'patient_id' => $patient_id,
            'bill_date' => $bill_date,
            'total_amount' => $total_amount,
            'discount' => $discount,
            'net_amount' => $net_amount,
            'payment_method' => $payment_method,
            'medicines_data' => is_array($medicines) ? json_encode($medicines) : $medicines,
            'created_by' => $this->session->userdata('admin_id') ?: 1,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Insert the bill
        $this->db->insert($bill_table, $bill_data);
        $bill_id = $this->db->insert_id();
        $db_error = $this->db->error();
        
        if ($bill_id) {
            echo json_encode([
                'status' => 1,
                'message' => 'Bill saved successfully using simple method',
                'bill_id' => $bill_id,
                'table_used' => $bill_table,
                'bill_data' => $bill_data
            ]);
        } else {
            echo json_encode([
                'status' => 0,
                'message' => 'Failed to save bill',
                'db_error' => $db_error,
                'last_query' => $this->db->last_query()
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0,
            'message' => 'Error in simple save: ' . $e->getMessage(),
            'error_trace' => $e->getTraceAsString()
        ]);
    }
}

public function saveBillQuickFix()
{
    header('Content-Type: application/json');
    
    // Get your actual column names
    $fields = $this->db->field_data('pharmacy_bills');
    $columns = [];
    foreach ($fields as $field) {
        $columns[] = $field->name;
    }
    
    // Build data using only existing columns
    $data = ['patient_id' => $this->input->post('patient_id')];
    
    // Add other fields only if they exist
    if (in_array('total', $columns)) $data['total'] = $this->input->post('total') ?: 0;
    if (in_array('amount', $columns)) $data['amount'] = $this->input->post('total') ?: 0;
    if (in_array('bill_date', $columns)) $data['bill_date'] = $this->input->post('bill_date');
    if (in_array('date', $columns)) $data['date'] = $this->input->post('bill_date');
    if (in_array('discount', $columns)) $data['discount'] = $this->input->post('discount') ?: 0;
    if (in_array('payment_mode', $columns)) $data['payment_mode'] = $this->input->post('payment_method') ?: 'cash';
    if (in_array('payment_method', $columns)) $data['payment_method'] = $this->input->post('payment_method') ?: 'cash';
    if (in_array('note', $columns)) $data['note'] = json_encode($this->input->post('medicines'));
    if (in_array('created_at', $columns)) $data['created_at'] = date('Y-m-d H:i:s');
    
    $this->db->insert('pharmacy_bills', $data);
    $bill_id = $this->db->insert_id();
    
    echo json_encode([
        'status' => $bill_id ? 1 : 0,
        'message' => $bill_id ? 'Bill saved successfully' : 'Failed to save',
        'bill_id' => $bill_id,
        'used_columns' => array_keys($data),
        'available_columns' => $columns
    ]);
}

    /**
     * Save new medicine
     */
    public function saveMedicine()
    {
        header('Content-Type: application/json');
        
        $this->form_validation->set_rules('medicine_name', 'Medicine Name', 'required|trim');
        $this->form_validation->set_rules('category', 'Category', 'required');
        $this->form_validation->set_rules('batch_number', 'Batch Number', 'required|trim');
        $this->form_validation->set_rules('quantity', 'Quantity', 'required|numeric|greater_than[0]');
        $this->form_validation->set_rules('purchase_price', 'Purchase Price', 'required|numeric|greater_than[0]');
        $this->form_validation->set_rules('sale_price', 'Sale Price', 'required|numeric|greater_than[0]');
        $this->form_validation->set_rules('expiry_date', 'Expiry Date', 'required');
        
        if ($this->form_validation->run() == false) {
            echo json_encode([
                'status' => 0,
                'message' => 'Validation failed',
                'errors' => validation_errors()
            ]);
            return;
        }
        
        try {
            $medicine_data = [
                'medicine_name' => $this->input->post('medicine_name'),
                'category' => $this->input->post('category'),
                'batch_number' => $this->input->post('batch_number'),
                'quantity' => $this->input->post('quantity'),
                'unit' => $this->input->post('unit'),
                'purchase_price' => $this->input->post('purchase_price'),
                'sale_price' => $this->input->post('sale_price'),
                'manufacturing_date' => $this->input->post('manufacturing_date'),
                'expiry_date' => $this->input->post('expiry_date'),
                'supplier_name' => $this->input->post('supplier_name'),
                'min_stock_alert' => $this->input->post('min_stock_alert') ?: 10,
                'description' => $this->input->post('description'),
                'created_by' => $this->session->userdata('admin_id'),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $medicine_id = $this->pharmacy_model->saveMedicine($medicine_data);
            
            if ($medicine_id) {
                echo json_encode([
                    'status' => 1,
                    'message' => 'Medicine added successfully',
                    'medicine_id' => $medicine_id
                ]);
            } else {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Error adding medicine'
                ]);
            }
        } catch (Exception $e) {
            log_message('error', 'Save medicine error: ' . $e->getMessage());
            echo json_encode([
                'status' => 0,
                'message' => 'Error adding medicine: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Filter medicines based on criteria
     */
    public function filterMedicines()
    {
        header('Content-Type: application/json');
        
        $filters = [
            'category' => $this->input->post('category'),
            'stock_status' => $this->input->post('stock_status'),
            'medicine_name' => $this->input->post('medicine_name')
        ];
        
        try {
            $medicines = $this->pharmacy_model->filterMedicines($filters);
            echo json_encode(['status' => 1, 'medicines' => $medicines]);
        } catch (Exception $e) {
            log_message('error', 'Filter medicines error: ' . $e->getMessage());
            echo json_encode(['status' => 0, 'message' => 'Error filtering medicines']);
        }
    }

    /**
     * Get medicine details for viewing
     */
    public function getMedicineDetails()
    {
        $medicine_id = $this->input->post('medicine_id');
        
        if (empty($medicine_id)) {
            echo 'Invalid medicine ID';
            return;
        }
        
        try {
            $data['medicine'] = $this->pharmacy_model->getMedicineById($medicine_id);
            $data['batches'] = $this->pharmacy_model->getMedicineBatches($medicine_id);
            $data['stock_history'] = $this->pharmacy_model->getMedicineStockHistory($medicine_id);
            
            $this->load->view('admin/pharmacy/medicineDetails', $data);
        } catch (Exception $e) {
            log_message('error', 'Get medicine details error: ' . $e->getMessage());
            echo 'Error loading medicine details';
        }
    }

    /**
     * Print bill
     */
    public function printBill($bill_id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy_bill', 'can_view')) {
            access_denied();
        }
        
        try {
            $data['bill'] = $this->pharmacy_model->getBillById($bill_id);
            $data['bill_items'] = $this->pharmacy_model->getBillItems($bill_id);
            $data['patient'] = $this->patient_model->getPatientById($data['bill']->patient_id);
            $data['hospital_settings'] = $this->customlib->getHospitalSettings();
            
            $this->load->view('admin/pharmacy/printBill', $data);
        } catch (Exception $e) {
            log_message('error', 'Print bill error: ' . $e->getMessage());
            show_error('Error loading bill for printing');
        }
    }

    /**
     * View bill details
     */
    public function viewBill($bill_no)
    {
        if (!$this->rbac->hasPrivilege('pharmacy_bill', 'can_view')) {
            access_denied();
        }
        
        try {
            $data['bill'] = $this->pharmacy_model->getBillByNumber($bill_no);
            $data['bill_items'] = $this->pharmacy_model->getBillItems($data['bill']->id);
            $data['patient'] = $this->patient_model->getPatientById($data['bill']->patient_id);
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/pharmacy/viewBill', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'View bill error: ' . $e->getMessage());
            show_error('Error loading bill details');
        }
    }

    /**
     * Medicine categories management
     */
    public function categories()
    {
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_edit')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Medicine Categories';
        $data['categories'] = $this->medicine_category_model->getMedicineCategory();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/categories', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Expiring medicines report
     */
    public function expiringMedicines()
    {
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_view')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Expiring Medicines';
        $data['expiring_medicines'] = $this->pharmacy_model->getExpiringMedicines();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/expiringMedicines', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Medicine pricing configuration
     */
    public function pricingConfig()
    {
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_edit')) {
            access_denied();
        }
        
        $this->session->set_userdata('top_menu', 'pharmacy');
        
        $data['title'] = 'Medicine Pricing Configuration';
        $data['pricing_rules'] = $this->pharmacy_model->getPricingRules();
        $data['medicineCategory'] = $this->medicine_category_model->getMedicineCategory();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/pharmacy/pricingConfig', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Save pricing rule
     */
    public function savePricingRule()
    {
        header('Content-Type: application/json');
        
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_edit')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $this->form_validation->set_rules('category_id', 'Category', 'required');
        $this->form_validation->set_rules('markup_percentage', 'Markup Percentage', 'required|numeric');
        
        if ($this->form_validation->run() == false) {
            echo json_encode([
                'status' => 0,
                'message' => 'Validation failed',
                'errors' => validation_errors()
            ]);
            return;
        }
        
        try {
            $rule_data = [
                'category_id' => $this->input->post('category_id'),
                'markup_percentage' => $this->input->post('markup_percentage'),
                'min_price' => $this->input->post('min_price'),
                'max_price' => $this->input->post('max_price'),
                'is_active' => 1,
                'created_by' => $this->session->userdata('admin_id'),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $result = $this->pharmacy_model->savePricingRule($rule_data);
            
            if ($result) {
                echo json_encode([
                    'status' => 1,
                    'message' => 'Pricing rule saved successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Error saving pricing rule'
                ]);
            }
        } catch (Exception $e) {
            log_message('error', 'Save pricing rule error: ' . $e->getMessage());
            echo json_encode([
                'status' => 0,
                'message' => 'Error saving pricing rule: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Update medicine stock
     */
    public function updateStock()
    {
        header('Content-Type: application/json');
        
        if (!$this->rbac->hasPrivilege('pharmacy', 'can_edit')) {
            echo json_encode(['status' => 0, 'message' => 'Access denied']);
            return;
        }
        
        $this->form_validation->set_rules('medicine_id', 'Medicine', 'required');
        $this->form_validation->set_rules('quantity', 'Quantity', 'required|numeric');
        $this->form_validation->set_rules('action', 'Action', 'required|in_list[add,subtract]');
        
        if ($this->form_validation->run() == false) {
            echo json_encode([
                'status' => 0,
                'message' => 'Validation failed',
                'errors' => validation_errors()
            ]);
            return;
        }
        
        try {
            $stock_data = [
                'medicine_id' => $this->input->post('medicine_id'),
                'quantity' => $this->input->post('quantity'),
                'action' => $this->input->post('action'),
                'reason' => $this->input->post('reason'),
                'updated_by' => $this->session->userdata('admin_id'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            $result = $this->pharmacy_model->updateStock($stock_data);
            
            if ($result) {
                echo json_encode([
                    'status' => 1,
                    'message' => 'Stock updated successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Error updating stock'
                ]);
            }
        } catch (Exception $e) {
            log_message('error', 'Update stock error: ' . $e->getMessage());
            echo json_encode([
                'status' => 0,
                'message' => 'Error updating stock: ' . $e->getMessage()
            ]);
        }
    }
}