<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Payslip Controller
 * Handles payslip generation and PDF creation
 */
class Payslip extends Admin_Controller {

    function __construct() {
        parent::__construct();
        $this->load->model('hr_payroll_model');
        $this->load->model('hr_model');
        $this->load->library(['pdf', 'email']);
        $this->load->helper(['url', 'download']);
    }

    /**
     * Generate and display payslip
     */
    public function generate($payroll_id) {
        try {
            $payslip_data = $this->hr_payroll_model->generatePayslipData($payroll_id);
            
            if (!$payslip_data) {
                show_404();
                return;
            }

            // Generate PDF
            $this->generatePayslipPDF($payslip_data);
            
        } catch (Exception $e) {
            log_message('error', 'Payslip generation error: ' . $e->getMessage());
            show_error('Failed to generate payslip: ' . $e->getMessage());
        }
    }

    /**
     * Generate payslip PDF
     */
    private function generatePayslipPDF($payslip_data) {
        // Initialize PDF
        $this->pdf->SetCreator('Hospital Management System');
        $this->pdf->SetAuthor('HR Department');
        $this->pdf->SetTitle('Payslip - ' . $payslip_data['payroll']->name . ' ' . $payslip_data['payroll']->surname);
        
        $this->pdf->AddPage();
        $this->pdf->SetFont('helvetica', '', 10);

        // Generate HTML content
        $html = $this->generatePayslipHTML($payslip_data);
        
        // Write HTML to PDF
        $this->pdf->writeHTML($html, true, false, true, false, '');
        
        // Output PDF
        $filename = 'payslip_' . $payslip_data['payroll']->employee_id . '_' . 
                   date('Y_m', mktime(0, 0, 0, $payslip_data['payroll']->month, 1, $payslip_data['payroll']->year)) . '.pdf';
        
        $this->pdf->Output($filename, 'I'); // 'I' for inline display, 'D' for download
    }

    /**
     * Generate payslip HTML content matching the Kenya Teachers format
     */
    private function generatePayslipHTML($payslip_data) {
        $payroll = $payslip_data['payroll'];
        $earnings = $payslip_data['earnings'];
        $deductions = $payslip_data['deductions'];
        $component_names = $payslip_data['component_names'];

        // Calculate taxable pay
        $taxable_pay = $this->calculateTaxablePay($earnings, $deductions);
        
        // Get organization details
        $organization = $this->getOrganizationDetails();

        $html = '
        <style>
            body { font-family: Arial, sans-serif; font-size: 10pt; }
            .header { text-align: center; margin-bottom: 20px; }
            .logo { width: 50px; height: 50px; }
            .payslip-title { font-size: 14pt; font-weight: bold; margin: 10px 0; }
            .employee-info { margin-bottom: 15px; }
            .employee-name { font-size: 12pt; font-weight: bold; }
            .section { margin-bottom: 15px; }
            .section-title { font-weight: bold; margin-bottom: 5px; border-bottom: 1px solid #000; }
            .amount-table { width: 100%; }
            .amount-table td { padding: 2px 5px; }
            .amount-right { text-align: right; }
            .total-line { border-top: 1px solid #000; font-weight: bold; }
            .net-pay { font-size: 12pt; font-weight: bold; margin-top: 10px; }
            .two-column { width: 48%; display: inline-block; vertical-align: top; }
            .left-column { margin-right: 4%; }
        </style>

        <div class="header">
            <img src="' . base_url() . 'assets/images/logo.png" class="logo" alt="Logo">
            <div class="payslip-title">' . strtoupper($organization['name']) . '</div>
            <div>' . date('F Y', mktime(0, 0, 0, $payroll->month, 1, $payroll->year)) . ' PAYSLIP</div>
        </div>

        <div class="employee-info">
            <div class="employee-name">EMPLOYEE NAME: ' . strtoupper($payroll->name . ' ' . $payroll->surname) . '</div>
            <div>EMPLOYEE NO: ' . $payroll->employee_id . '</div>
        </div>

        <div class="two-column left-column">
            <div class="section">
                <div class="section-title">EARNINGS</div>
                <table class="amount-table">
                    ' . $this->generateEarningsRows($earnings, $component_names) . '
                </table>
            </div>

            <div class="section">
                <div class="section-title">TAX CALCULATION</div>
                <table class="amount-table">
                    <tr>
                        <td>Defined contribution</td>
                        <td class="amount-right">' . number_format($deductions['NSSF'] ?? 0, 2) . '</td>
                    </tr>
                    <tr>
                        <td>Taxable Pay</td>
                        <td class="amount-right">' . number_format($taxable_pay, 2) . '</td>
                    </tr>
                    <tr>
                        <td>Tax charged</td>
                        <td class="amount-right">' . number_format($deductions['PAYE'] ?? 0, 2) . '</td>
                    </tr>
                    <tr>
                        <td>Relief</td>
                        <td class="amount-right">' . number_format($this->getTaxRelief(), 2) . '</td>
                    </tr>
                </table>
            </div>
        </div>

        <div class="two-column">
            <div class="section">
                <div class="section-title">DEDUCTIONS</div>
                <table class="amount-table">
                    ' . $this->generateDeductionsRows($deductions, $component_names) . '
                    <tr class="total-line">
                        <td>Total Deductions</td>
                        <td class="amount-right">' . number_format($payroll->total_deductions, 2) . '</td>
                    </tr>
                </table>
            </div>

            <div class="net-pay">
                <table class="amount-table">
                    <tr class="total-line">
                        <td><strong>NET PAY</strong></td>
                        <td class="amount-right"><strong>' . number_format($payroll->net_salary, 2) . '</strong></td>
                    </tr>
                </table>
            </div>
        </div>

        <div style="margin-top: 30px; font-size: 8pt; text-align: center;">
            Generated on ' . date('d/m/Y H:i:s') . ' | ' . $organization['name'] . ' Payroll System
        </div>';

        return $html;
    }

    /**
     * Generate earnings rows for payslip
     */
    private function generateEarningsRows($earnings, $component_names) {
        $rows = '';
        $total_earnings = 0;

        foreach ($earnings as $code => $amount) {
            if ($amount > 0) {
                $name = isset($component_names[$code]) ? $component_names[$code] : $code;
                $rows .= '<tr>';
                $rows .= '<td>' . $name . '</td>';
                $rows .= '<td class="amount-right">' . number_format($amount, 2) . '</td>';
                $rows .= '</tr>';
                $total_earnings += $amount;
            }
        }

        // Add total row if there are multiple earnings
        if (count(array_filter($earnings)) > 1) {
            $rows .= '<tr class="total-line">';
            $rows .= '<td>Total Earnings</td>';
            $rows .= '<td class="amount-right">' . number_format($total_earnings, 2) . '</td>';
            $rows .= '</tr>';
        }

        return $rows;
    }

    /**
     * Generate deductions rows for payslip
     */
    private function generateDeductionsRows($deductions, $component_names) {
        $rows = '';
        
        // Order deductions properly (statutory first, then others)
        $ordered_deductions = $this->orderDeductions($deductions);

        foreach ($ordered_deductions as $code => $amount) {
            if ($amount > 0) {
                $name = isset($component_names[$code]) ? $component_names[$code] : $code;
                $rows .= '<tr>';
                $rows .= '<td>' . $name . '</td>';
                $rows .= '<td class="amount-right">' . number_format($amount, 2) . '</td>';
                $rows .= '</tr>';
            }
        }

        return $rows;
    }

    /**
     * Order deductions to match standard format
     */
    private function orderDeductions($deductions) {
        $order = ['PAYE', 'NSSF', 'NHIF', 'AHL', 'LOAN', 'ADVANCE'];
        $ordered = [];
        
        // Add in specified order
        foreach ($order as $code) {
            if (isset($deductions[$code]) && $deductions[$code] > 0) {
                $ordered[$code] = $deductions[$code];
            }
        }
        
        // Add any remaining deductions
        foreach ($deductions as $code => $amount) {
            if (!in_array($code, $order) && $amount > 0) {
                $ordered[$code] = $amount;
            }
        }
        
        return $ordered;
    }

    /**
     * Calculate taxable pay
     */
    private function calculateTaxablePay($earnings, $deductions) {
        $gross_pay = array_sum($earnings);
        $non_taxable_deductions = ($deductions['NSSF'] ?? 0) + ($deductions['NHIF'] ?? 0);
        return $gross_pay - $non_taxable_deductions;
    }

    /**
     * Get tax relief amount (Kenya standard relief)
     */
    private function getTaxRelief() {
        return 1408; // Monthly tax relief as of 2024
    }

    /**
     * Get organization details
     */
    private function getOrganizationDetails() {
        // You can load this from database or config
        return [
            'name' => $this->config->item('hospital_name') ?: 'Hospital Management System',
            'address' => $this->config->item('hospital_address') ?: '',
            'phone' => $this->config->item('hospital_phone') ?: '',
            'email' => $this->config->item('hospital_email') ?: ''
        ];
    }

    /**
     * Email payslip to staff
     */
    public function email($payroll_id) {
        header('Content-Type: application/json');
        
        try {
            $payslip_data = $this->hr_payroll_model->generatePayslipData($payroll_id);
            
            if (!$payslip_data) {
                echo json_encode(['status' => 'error', 'message' => 'Payslip not found']);
                return;
            }

            $payroll = $payslip_data['payroll'];
            
            if (!$payroll->email) {
                echo json_encode(['status' => 'error', 'message' => 'Staff email not found']);
                return;
            }

            // Generate PDF content
            $pdf_content = $this->generatePayslipPDFContent($payslip_data);
            
            // Email configuration
            $this->email->from($this->config->item('smtp_from_email'), $this->config->item('hospital_name'));
            $this->email->to($payroll->email);
            $this->email->subject('Payslip for ' . date('F Y', mktime(0, 0, 0, $payroll->month, 1, $payroll->year)));
            
            $message = "Dear " . $payroll->name . ",\n\n";
            $message .= "Please find attached your payslip for " . date('F Y', mktime(0, 0, 0, $payroll->month, 1, $payroll->year)) . ".\n\n";
            $message .= "Best regards,\n";
            $message .= "HR Department";
            
            $this->email->message($message);
            
            // Attach PDF
            $filename = 'payslip_' . $payroll->employee_id . '_' . 
                       date('Y_m', mktime(0, 0, 0, $payroll->month, 1, $payroll->year)) . '.pdf';
            $this->email->attach($pdf_content, 'attachment', $filename, 'application/pdf');
            
            if ($this->email->send()) {
                echo json_encode(['status' => 'success', 'message' => 'Payslip emailed successfully']);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to send email']);
            }
            
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Generate PDF content for email attachment
     */
    private function generatePayslipPDFContent($payslip_data) {
        $this->pdf->SetCreator('Hospital Management System');
        $this->pdf->SetAuthor('HR Department');
        $this->pdf->SetTitle('Payslip');
        
        $this->pdf->AddPage();
        $this->pdf->SetFont('helvetica', '', 10);

        $html = $this->generatePayslipHTML($payslip_data);
        $this->pdf->writeHTML($html, true, false, true, false, '');
        
        return $this->pdf->Output('', 'S'); // 'S' returns the document as a string
    }

    /**
     * Bulk generate payslips for multiple staff
     */
    public function bulkGenerate() {
        if ($this->input->method() == 'post') {
            header('Content-Type: application/json');
            
            try {
                $month = $this->input->post('month');
                $year = $this->input->post('year');
                $department_id = $this->input->post('department_id');
                
                $payroll_list = $this->hr_payroll_model->getPayrollList($month, $year, $department_id);
                
                if (empty($payroll_list)) {
                    echo json_encode(['status' => 'error', 'message' => 'No payroll records found']);
                    return;
                }

                // Create ZIP file with all payslips
                $this->load->library('zip');
                
                foreach ($payroll_list as $payroll) {
                    $payslip_data = $this->hr_payroll_model->generatePayslipData($payroll->id);
                    if ($payslip_data) {
                        $pdf_content = $this->generatePayslipPDFContent($payslip_data);
                        $filename = 'payslip_' . $payroll->employee_id . '_' . 
                                   date('Y_m', mktime(0, 0, 0, $month, 1, $year)) . '.pdf';
                        $this->zip->add_data($filename, $pdf_content);
                    }
                }

                $zip_filename = 'payslips_' . date('Y_m', mktime(0, 0, 0, $month, 1, $year)) . '.zip';
                $this->zip->download($zip_filename);
                
            } catch (Exception $e) {
                echo json_encode(['status' => 'error', 'message' => 'Error: ' . $e->getMessage()]);
            }
        } else {
            // Load bulk generation form
            $data['title'] = 'Bulk Generate Payslips';
            $data['departments'] = $this->hr_model->getDepartments();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/hr/payroll/bulk_generate', $data);
            $this->load->view('layout/footer', $data);
        }
    }
}