<?php

defined('BASEPATH') OR exit('No direct script access allowed');

class Patient_billing extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->library('form_validation');
    }

    public function index() {
        // Temporarily bypass permission check - uncomment after setting up permissions
        /*
        if (!$this->rbac->hasPrivilege('patient', 'can_view')) {
            access_denied();
        }
        */

        $this->session->set_userdata('top_menu', 'Patient');
        $this->session->set_userdata('sub_menu', 'patient/patient_billing');
        
        $data['title'] = 'Patient Management & Billing';
        $data['bloodgroups'] = $this->getBloodGroups();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/patient/patient_billing', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get all patients with billing summary - CORRECTED VERSION
     */
    private function getAllPatientsWithBillingSummary() {
        try {
            $tables = $this->db->list_tables();
            
            if (!in_array('patients', $tables)) {
                return [];
            }
            
            // Get patients with blood group info
            $this->db->select('
                p.id,
                p.patient_name,
                p.id as patient_unique_id,
                p.age,
                p.gender,
                p.mobileno,
                p.email,
                p.address,
                p.created_at as admission_date,
                p.is_active,
                bg.name as blood_group_name
            ');
            $this->db->from('patients p');
            $this->db->join('blood_group bg', 'bg.id = p.blood_group_id', 'left');
            $this->db->where('p.is_active', 'yes');
            $this->db->order_by('p.patient_name', 'ASC');
            
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $patients = $query->result_array();
                
                // Calculate billing totals for each patient
                foreach ($patients as &$patient) {
                    $billing = $this->calculatePatientBilling($patient['id'], $tables);
                    $patient = array_merge($patient, $billing);
                }
                
                return $patients;
            }
            
            return [];
        } catch (Exception $e) {
            log_message('error', 'Error in getAllPatientsWithBillingSummary: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get dashboard statistics - CORRECTED VERSION
     */
    public function getDashboardStats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->calculateDashboardStats();
            
            echo json_encode([
                'status' => 'success',
                'stats' => $stats
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get detailed billing information for a specific patient
     */
    public function getPatientBillingDetails() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            
            if (empty($patient_id)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID is required'
                ]);
                return;
            }

            $billing_details = $this->getPatientDetailedBilling($patient_id);
            
            if ($billing_details !== false) {
                echo json_encode([
                    'status' => 'success',
                    'billing_details' => $billing_details
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to load billing details'
                ]);
            }
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Patient search endpoint for AJAX search - CORRECTED VERSION
     */
    public function patient_search() {
        header('Content-Type: application/json');
        
        try {
            $search_term = $this->input->get('q') ?: $this->input->post('q');
            $page = $this->input->get('page') ?: 1;
            $per_page = 30;
            $offset = ($page - 1) * $per_page;

            if (empty($search_term) || strlen($search_term) < 2) {
                echo json_encode([
                    'items' => [],
                    'total_count' => 0,
                    'incomplete_results' => false
                ]);
                return;
            }

            // Get patients with search functionality
            $this->db->select('
                p.id, 
                p.patient_name, 
                p.mobileno, 
                p.age,
                p.gender,
                p.guardian_name,
                p.email,
                p.address,
                bg.name as blood_group_name
            ');
            $this->db->from('patients p');
            $this->db->join('blood_group bg', 'bg.id = p.blood_group_id', 'left');
            
            // Search conditions
            $this->db->group_start();
            $this->db->like('p.patient_name', $search_term);
            $this->db->or_like('p.mobileno', $search_term);
            if ($this->db->field_exists('identification_number', 'patients')) {
                $this->db->or_like('p.identification_number', $search_term);
            }
            if ($this->db->field_exists('insurance_id', 'patients')) {
                $this->db->or_like('p.insurance_id', $search_term);
            }
            $this->db->or_like('p.guardian_name', $search_term);
            $this->db->or_like('p.email', $search_term);
            $this->db->group_end();
            
            // Only active patients
            $this->db->where('p.is_active', 'yes');

            // Get total count for pagination
            $total_query = clone $this->db;
            $total_count = $total_query->count_all_results('', false);

            // Apply pagination and ordering
            $this->db->order_by('p.patient_name', 'ASC');
            $this->db->limit($per_page, $offset);

            $query = $this->db->get();
            $patients = $query ? $query->result_array() : [];

            // Format results for Select2
            $items = [];
            foreach ($patients as $patient) {
                $display_text = $patient['patient_name'];
                
                // Add additional info for better identification
                $additional_info = [];
                if (!empty($patient['id'])) {
                    $additional_info[] = 'ID: ' . $patient['id'];
                }
                if (!empty($patient['mobileno'])) {
                    $additional_info[] = 'Phone: ' . $patient['mobileno'];
                }
                if (!empty($patient['age'])) {
                    $additional_info[] = $patient['age'] . 'y';
                }
                if (!empty($patient['gender'])) {
                    $additional_info[] = $patient['gender'];
                }
                
                if (!empty($additional_info)) {
                    $display_text .= ' (' . implode(', ', $additional_info) . ')';
                }

                $items[] = [
                    'id' => $patient['id'],
                    'text' => $display_text,
                    'patient_name' => $patient['patient_name'],
                    'patient_id' => $patient['id'],
                    'mobileno' => $patient['mobileno'] ?: '',
                    'age' => $patient['age'] ?: '',
                    'gender' => $patient['gender'] ?: '',
                    'guardian_name' => $patient['guardian_name'] ?: '',
                    'blood_group_name' => $patient['blood_group_name'] ?: '',
                    'email' => $patient['email'] ?: '',
                    'address' => $patient['address'] ?: ''
                ];
            }

            echo json_encode([
                'items' => $items,
                'total_count' => $total_count,
                'incomplete_results' => ($offset + $per_page) < $total_count
            ]);

        } catch (Exception $e) {
            log_message('error', "Patient search error: " . $e->getMessage());
            echo json_encode([
                'items' => [],
                'total_count' => 0,
                'error' => 'Search failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Add a new patient
     */
    public function addPatient() {
        header('Content-Type: application/json');
        
        try {
            // Set validation rules
            $this->form_validation->set_rules('patient_name', 'Patient Name', 'trim|required|xss_clean');
            $this->form_validation->set_rules('age', 'Age', 'trim|required|numeric|xss_clean');
            $this->form_validation->set_rules('gender', 'Gender', 'trim|required|xss_clean');
            $this->form_validation->set_rules('mobileno', 'Mobile Number', 'trim|required|xss_clean');
            $this->form_validation->set_rules('email', 'Email', 'trim|valid_email|xss_clean');
            $this->form_validation->set_rules('address', 'Address', 'trim|xss_clean');

            if ($this->form_validation->run() == FALSE) {
                echo json_encode([
                    'status' => 'error',
                    'message' => strip_tags(validation_errors())
                ]);
                return;
            }

            // Check if mobile number already exists
            $this->db->where('mobileno', $this->input->post('mobileno'));
            $existing = $this->db->get('patients')->num_rows();
            
            if ($existing > 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Mobile number already exists'
                ]);
                return;
            }

            // Prepare patient data
            $patient_data = array(
                'patient_name' => $this->input->post('patient_name'),
                'age' => $this->input->post('age'),
                'gender' => $this->input->post('gender'),
                'mobileno' => $this->input->post('mobileno'),
                'email' => $this->input->post('email'),
                'address' => $this->input->post('address'),
                'blood_group_id' => $this->input->post('blood_group_id'),
                'patient_unique_id' => $this->generatePatientUniqueId(),
                'created_at' => date('Y-m-d H:i:s'),
                'is_active' => 'yes'
            );

            $this->db->insert('patients', $patient_data);
            $patient_id = $this->db->insert_id();

            if ($patient_id) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Patient added successfully',
                    'patient_id' => $patient_id
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to add patient'
                ]);
            }
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get blood groups - works with existing blood_group table
     */
    private function getBloodGroups() {
        try {
            $this->db->select('id, name');
            $this->db->where('is_blood_group', 'yes');
            $query = $this->db->get('blood_group');
            
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
        } catch (Exception $e) {
            log_message('error', 'Error loading blood groups: ' . $e->getMessage());
        }
        
        // Fallback to default blood groups
        return [
            ['id' => '1', 'name' => 'A+'],
            ['id' => '2', 'name' => 'A-'],
            ['id' => '3', 'name' => 'B+'],
            ['id' => '4', 'name' => 'B-'],
            ['id' => '5', 'name' => 'AB+'],
            ['id' => '6', 'name' => 'AB-'],
            ['id' => '7', 'name' => 'O+'],
            ['id' => '8', 'name' => 'O-']
        ];
    }

    /**
     * Calculate billing totals for a patient - CORRECTED for your schema
     */
    private function calculatePatientBilling($patient_id, $tables = null) {
        if (!$tables) {
            $tables = $this->db->list_tables();
        }
        
        $billing = [
            'consultation_total' => 0,
            'lab_total' => 0,
            'pharmacy_total' => 0,
            'radiology_total' => 0,
            'total_billing' => 0,
            'last_visit' => null
        ];
        
        $last_visits = [];
        
        try {
            // CONSULTATION TOTALS - Use consultation_financials table
            if (in_array('consultation_financials', $tables) && in_array('consultations', $tables)) {
                $this->db->select_sum('cf.total_amount', 'total_amount');
                $this->db->select_max('c.date', 'last_date');
                $this->db->from('consultation_financials cf');
                $this->db->join('consultations c', 'c.id = cf.consultation_id', 'inner');
                $this->db->where('c.patient_id', $patient_id);
                $result = $this->db->get()->row();
                
                $billing['consultation_total'] = $result->total_amount ?: 0;
                if ($result->last_date) {
                    $last_visits[] = $result->last_date;
                }
            }
            
            // LABORATORY TOTALS - Use pathology_billing table
            if (in_array('pathology_billing', $tables)) {
                $this->db->select_sum('net_amount', 'total_amount');
                $this->db->select_max('date', 'last_date');
                $this->db->where('patient_id', $patient_id);
                $result = $this->db->get('pathology_billing')->row();
                
                $billing['lab_total'] = $result->total_amount ?: 0;
                if ($result->last_date) {
                    $last_visits[] = $result->last_date;
                }
            }
            
            // PHARMACY TOTALS - Use pharmacy_bill_basic table
            if (in_array('pharmacy_bill_basic', $tables)) {
                $this->db->select_sum('net_amount', 'total_amount');
                $this->db->select_max('date', 'last_date');
                $this->db->where('patient_id', $patient_id);
                $result = $this->db->get('pharmacy_bill_basic')->row();
                
                $billing['pharmacy_total'] = $result->total_amount ?: 0;
                if ($result->last_date) {
                    $last_visits[] = $result->last_date;
                }
            }
            
            // RADIOLOGY TOTALS - Use radiology_billing table (this exists in your schema)
            if (in_array('radiology_billing', $tables)) {
                $this->db->select_sum('net_amount', 'total_amount');
                $this->db->select_max('date', 'last_date');
                $this->db->where('patient_id', $patient_id);
                $result = $this->db->get('radiology_billing')->row();
                
                $billing['radiology_total'] = $result->total_amount ?: 0;
                if ($result->last_date) {
                    $last_visits[] = $result->last_date;
                }
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error calculating patient billing: ' . $e->getMessage());
        }
        
        // Calculate total and last visit
        $billing['total_billing'] = $billing['consultation_total'] + 
                                  $billing['lab_total'] + 
                                  $billing['pharmacy_total'] + 
                                  $billing['radiology_total'];
        
        if (!empty($last_visits)) {
            $billing['last_visit'] = max($last_visits);
        }
        
        return $billing;
    }

    /**
     * Calculate dashboard statistics - CORRECTED VERSION
     */
    private function calculateDashboardStats() {
        $stats = [
            'total_patients' => 0,
            'total_revenue' => 0,
            'today_consultations' => 0,
            'pending_labs' => 0
        ];
        
        $tables = $this->db->list_tables();
        
        try {
            // Total patients
            if (in_array('patients', $tables)) {
                $this->db->where('is_active', 'yes');
                $stats['total_patients'] = $this->db->count_all_results('patients');
            }
            
            // Total revenue calculation
            $total_revenue = 0;
            
            // CONSULTATION REVENUE - Use consultation_financials
            if (in_array('consultation_financials', $tables)) {
                $this->db->select_sum('total_amount');
                $result = $this->db->get('consultation_financials')->row();
                $total_revenue += $result->total_amount ?: 0;
            }
            
            // LABORATORY REVENUE - Use pathology_billing
            if (in_array('pathology_billing', $tables)) {
                $this->db->select_sum('net_amount');
                $result = $this->db->get('pathology_billing')->row();
                $total_revenue += $result->net_amount ?: 0;
            }
            
            // PHARMACY REVENUE - Use pharmacy_bill_basic
            if (in_array('pharmacy_bill_basic', $tables)) {
                $this->db->select_sum('net_amount');
                $result = $this->db->get('pharmacy_bill_basic')->row();
                $total_revenue += $result->net_amount ?: 0;
            }
            
            // RADIOLOGY REVENUE - Use radiology_billing
            if (in_array('radiology_billing', $tables)) {
                $this->db->select_sum('net_amount');
                $result = $this->db->get('radiology_billing')->row();
                $total_revenue += $result->net_amount ?: 0;
            }
            
            $stats['total_revenue'] = $total_revenue;
            
            // Today's consultations
            if (in_array('consultations', $tables)) {
                $today = date('Y-m-d');
                $this->db->where('DATE(date)', $today);
                $stats['today_consultations'] = $this->db->count_all_results('consultations');
            }
            
            // Pending lab results - count records without report results
            if (in_array('pathology_billing', $tables)) {
                // Count pathology bills without corresponding reports
                $this->db->select('pb.id');
                $this->db->from('pathology_billing pb');
                $this->db->join('pathology_report pr', 'pr.pathology_bill_id = pb.id', 'left');
                $this->db->where('pr.id IS NULL');
                $stats['pending_labs'] = $this->db->count_all_results();
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error calculating dashboard stats: ' . $e->getMessage());
        }
        
        return $stats;
    }

    /**
     * Get detailed billing for a patient - FULLY CORRECTED for your schema
     */
    private function getPatientDetailedBilling($patient_id) {
        $billing_details = [
            'consultations' => [],
            'laboratory' => [],
            'pharmacy' => [],
            'radiology' => [],
            'totals' => [
                'consultations' => 0,
                'laboratory' => 0,
                'pharmacy' => 0,
                'radiology' => 0,
                'grand_total' => 0
            ]
        ];
        
        $tables = $this->db->list_tables();
        
        try {
            // CONSULTATIONS - Use consultations + consultation_financials
            if (in_array('consultations', $tables) && in_array('consultation_financials', $tables)) {
                $this->db->select('
                    c.id,
                    c.date as date,
                    cf.total_amount as amount,
                    c.symptoms,
                    c.diagnosis,
                    c.doctor,
                    CONCAT(COALESCE(staff.name, ""), " ", COALESCE(staff.surname, "")) as doctor_name,
                    cf.payment_status as status,
                    "Consultation" as type
                ');
                $this->db->from('consultations c');
                $this->db->join('consultation_financials cf', 'cf.consultation_id = c.id', 'inner');
                $this->db->join('staff', 'staff.id = c.doctor', 'left');
                $this->db->where('c.patient_id', $patient_id);
                $this->db->order_by('c.date', 'DESC');
                
                $query = $this->db->get();
                if ($query) {
                    $consultations = $query->result_array();
                    $billing_details['consultations'] = $consultations;
                    $billing_details['totals']['consultations'] = array_sum(array_column($consultations, 'amount'));
                }
            }
            
            // LABORATORY - Use pathology_billing (simplified, no joins to avoid errors)
            if (in_array('pathology_billing', $tables)) {
                $this->db->select('
                    pb.id,
                    pb.date,
                    pb.net_amount as amount,
                    pb.doctor_name,
                    "Lab Test" as test_name,
                    "Pending" as result,
                    "Pending" as status
                ');
                $this->db->from('pathology_billing pb');
                $this->db->where('pb.patient_id', $patient_id);
                $this->db->order_by('pb.date', 'DESC');
                
                $query = $this->db->get();
                if ($query) {
                    $laboratory = $query->result_array();
                    
                    // Try to get test names from pathology table if it exists
                    if (in_array('pathology', $tables)) {
                        foreach ($laboratory as &$lab) {
                            // Try to get test name if pathology_id exists in pathology_billing
                            if ($this->db->field_exists('pathology_id', 'pathology_billing')) {
                                $this->db->select('test_name');
                                $this->db->from('pathology');
                                $this->db->join('pathology_billing pb2', 'pb2.pathology_id = pathology.id', 'inner');
                                $this->db->where('pb2.id', $lab['id']);
                                $pathology_query = $this->db->get();
                                
                                if ($pathology_query && $pathology_query->num_rows() > 0) {
                                    $pathology_result = $pathology_query->row();
                                    $lab['test_name'] = $pathology_result->test_name;
                                }
                            }
                            
                            // Check if there's a report for this lab
                            if (in_array('pathology_report', $tables)) {
                                $this->db->select('id');
                                $this->db->from('pathology_report');
                                $this->db->where('pathology_bill_id', $lab['id']);
                                $report_query = $this->db->get();
                                
                                if ($report_query && $report_query->num_rows() > 0) {
                                    $lab['status'] = 'Completed';
                                }
                            }
                        }
                    }
                    
                    $billing_details['laboratory'] = $laboratory;
                    $billing_details['totals']['laboratory'] = array_sum(array_column($laboratory, 'amount'));
                }
            }
            
            // PHARMACY - Use pharmacy_bill_basic
            if (in_array('pharmacy_bill_basic', $tables)) {
                $this->db->select('
                    pbb.id,
                    pbb.date,
                    pbb.net_amount as amount,
                    "Pharmacy Bill" as medications,
                    COALESCE(pbb.discount, 0) as discount,
                    "Dispensed" as status
                ');
                $this->db->from('pharmacy_bill_basic pbb');
                $this->db->where('pbb.patient_id', $patient_id);
                $this->db->order_by('pbb.date', 'DESC');
                
                $query = $this->db->get();
                if ($query) {
                    $pharmacy = $query->result_array();
                    $billing_details['pharmacy'] = $pharmacy;
                    $billing_details['totals']['pharmacy'] = array_sum(array_column($pharmacy, 'amount'));
                }
            }
            
            // RADIOLOGY - Use radiology_billing (simplified)
            if (in_array('radiology_billing', $tables)) {
                $this->db->select('
                    rb.id,
                    rb.date,
                    rb.net_amount as amount,
                    rb.doctor_name,
                    "Radiology Test" as scan_name,
                    COALESCE(rb.note, "") as result,
                    "Pending" as status
                ');
                $this->db->from('radiology_billing rb');
                $this->db->where('rb.patient_id', $patient_id);
                $this->db->order_by('rb.date', 'DESC');
                
                $query = $this->db->get();
                if ($query) {
                    $radiology = $query->result_array();
                    
                    // Try to get test names from radio table if relationship exists
                    if (in_array('radio', $tables)) {
                        foreach ($radiology as &$rad) {
                            // Try to get test name if radio_id exists in radiology_billing
                            if ($this->db->field_exists('radio_id', 'radiology_billing')) {
                                $this->db->select('test_name');
                                $this->db->from('radio');
                                $this->db->join('radiology_billing rb2', 'rb2.radio_id = radio.id', 'inner');
                                $this->db->where('rb2.id', $rad['id']);
                                $radio_query = $this->db->get();
                                
                                if ($radio_query && $radio_query->num_rows() > 0) {
                                    $radio_result = $radio_query->row();
                                    $rad['scan_name'] = $radio_result->test_name;
                                }
                            }
                            
                            // Check if there's a report for this radiology
                            if (in_array('radiology_report', $tables)) {
                                $this->db->select('id');
                                $this->db->from('radiology_report');
                                $this->db->where('radiology_bill_id', $rad['id']);
                                $report_query = $this->db->get();
                                
                                if ($report_query && $report_query->num_rows() > 0) {
                                    $rad['status'] = 'Completed';
                                }
                            }
                        }
                    }
                    
                    $billing_details['radiology'] = $radiology;
                    $billing_details['totals']['radiology'] = array_sum(array_column($radiology, 'amount'));
                }
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error getting patient billing details: ' . $e->getMessage());
        }
        
        // Calculate grand total
        $billing_details['totals']['grand_total'] = 
            $billing_details['totals']['consultations'] +
            $billing_details['totals']['laboratory'] +
            $billing_details['totals']['pharmacy'] +
            $billing_details['totals']['radiology'];
        
        return $billing_details;
    }

    /**
     * Generate unique patient ID
     */
    private function generatePatientUniqueId() {
        // Get the last patient ID
        $this->db->select_max('id');
        $result = $this->db->get('patients')->row();
        $last_id = $result->id ?: 0;
        
        return 'PAT' . sprintf('%06d', $last_id + 1);
    }

    /**
     * Test method to verify table structure
     */
    public function test() {
        echo "<h2>Patient Billing Controller - Database Schema Aligned Test</h2>";
        echo "<p><strong>Status:</strong> Corrected to match your database schema</p>";
        
        // Test database connections
        $tables = $this->db->list_tables();
        echo "<p><strong>Database tables:</strong> " . count($tables) . " found</p>";
        
        // Show relevant billing tables and their column counts
        $billing_tables = [];
        foreach ($tables as $table) {
            if (strpos($table, 'consult') !== false || 
                strpos($table, 'radiology') !== false || 
                strpos($table, 'pathology') !== false || 
                strpos($table, 'pharmacy') !== false ||
                $table == 'patients') {
                
                $fields = $this->db->list_fields($table);
                $billing_tables[] = $table . ' (' . count($fields) . ' columns)';
                
                // Show key fields for verification
                if ($table == 'consultations') {
                    echo "<p><strong>Consultations table fields:</strong> " . implode(', ', $fields) . "</p>";
                }
            }
        }
        echo "<p><strong>Billing tables:</strong> " . implode(', ', $billing_tables) . "</p>";
        
        // Test blood groups
        $bloodgroups = $this->getBloodGroups();
        echo "<p><strong>Blood groups:</strong> " . count($bloodgroups) . " available</p>";
        
        // Test patients
        $this->db->where('is_active', 'yes');
        $patient_count = $this->db->count_all_results('patients');
        echo "<p><strong>Active patients:</strong> $patient_count</p>";
        
        // Test dashboard stats
        $stats = $this->calculateDashboardStats();
        echo "<p><strong>Dashboard Stats:</strong></p>";
        echo "<ul>";
        echo "<li>Total Patients: " . $stats['total_patients'] . "</li>";
        echo "<li>Total Revenue: $" . number_format($stats['total_revenue'], 2) . "</li>";
        echo "<li>Today's Consultations: " . $stats['today_consultations'] . "</li>";
        echo "<li>Pending Labs: " . $stats['pending_labs'] . "</li>";
        echo "</ul>";
        
        echo "<hr>";
        echo "<p><a href='" . base_url('admin/patient_billing') . "' class='btn btn-primary'>Go to Patient Billing System</a></p>";
        echo "<p><a href='" . base_url('admin/patient_billing/debug_schema') . "' class='btn btn-info'>Debug Schema Alignment</a></p>";
    }
    
    /**
     * Debug method to verify schema alignment
     */
    public function debug_schema() {
        $patient_id = 55; // Change this to test with a specific patient
        
        echo "<h3>Schema-Aligned Billing Debug - Patient ID: $patient_id</h3>";
        echo "<p><em>Testing billing calculations aligned with your actual database schema</em></p>";
        
        $tables = $this->db->list_tables();
        
        // Test consultations table structure
        if (in_array('consultations', $tables)) {
            echo "<h4>✅ CONSULTATIONS Table Structure:</h4>";
            $fields = $this->db->list_fields('consultations');
            echo "<div style='background: #e8f5e8; padding: 10px; margin: 10px 0;'>";
            echo "<strong>Available columns:</strong> " . implode(', ', $fields) . "<br>";
            
            // Check specific fields that caused errors
            $has_doctor_id = in_array('doctor_id', $fields);
            $has_doctor = in_array('doctor', $fields);
            echo "<strong>Has 'doctor_id' column:</strong> " . ($has_doctor_id ? 'YES' : 'NO') . "<br>";
            echo "<strong>Has 'doctor' column:</strong> " . ($has_doctor ? 'YES' : 'NO') . "<br>";
            echo "</div>";
            
            // Test consultation query with correct column
            $doctor_column = $has_doctor_id ? 'doctor_id' : 'doctor';
            
            $this->db->select("c.id, c.patient_id, c.$doctor_column, c.date, c.symptoms, c.diagnosis");
            $this->db->from('consultations c');
            $this->db->where('c.patient_id', $patient_id);
            $this->db->limit(3);
            $result = $this->db->get();
            
            if ($result && $result->num_rows() > 0) {
                echo "<p><strong>Sample consultation records:</strong></p>";
                echo "<pre>" . print_r($result->result_array(), true) . "</pre>";
            } else {
                echo "<p><em>No consultation records found for this patient</em></p>";
            }
        }
        
        // Test pathology_billing table structure
        if (in_array('pathology_billing', $tables)) {
            echo "<h4>✅ PATHOLOGY_BILLING Table Structure:</h4>";
            $fields = $this->db->list_fields('pathology_billing');
            echo "<div style='background: #e8f5e8; padding: 10px; margin: 10px 0;'>";
            echo "<strong>Available columns:</strong> " . implode(', ', $fields) . "<br>";
            
            // Check for payment_status column that caused CASE WHEN error
            $has_payment_status = in_array('payment_status', $fields);
            echo "<strong>Has 'payment_status' column:</strong> " . ($has_payment_status ? 'YES' : 'NO') . "<br>";
            echo "</div>";
            
            // Test pathology query without problematic CASE WHEN
            $this->db->select('pb.id, pb.date, pb.net_amount, pb.patient_id');
            if ($has_payment_status) {
                $this->db->select('pb.payment_status', false);
            }
            $this->db->from('pathology_billing pb');
            $this->db->where('pb.patient_id', $patient_id);
            $this->db->limit(3);
            $result = $this->db->get();
            
            if ($result && $result->num_rows() > 0) {
                echo "<p><strong>Sample pathology billing records:</strong></p>";
                echo "<pre>" . print_r($result->result_array(), true) . "</pre>";
            } else {
                echo "<p><em>No pathology billing records found for this patient</em></p>";
            }
        }
        
        // Test radiology_billing table structure
        if (in_array('radiology_billing', $tables)) {
            echo "<h4>✅ RADIOLOGY_BILLING Table Structure:</h4>";
            $fields = $this->db->list_fields('radiology_billing');
            echo "<div style='background: #e8f5e8; padding: 10px; margin: 10px 0;'>";
            echo "<strong>Available columns:</strong> " . implode(', ', $fields) . "<br>";
            
            // Check for columns that might not exist
            $has_referring_doctor_id = in_array('referring_doctor_id', $fields);
            $has_reffering_doctor_id = in_array('reffering_doctor_id', $fields);
            $has_test_name = in_array('test_name', $fields);
            $has_order_date = in_array('order_date', $fields);
            $has_total_amount = in_array('total_amount', $fields);
            
            echo "<strong>Has 'referring_doctor_id':</strong> " . ($has_referring_doctor_id ? 'YES' : 'NO') . "<br>";
            echo "<strong>Has 'reffering_doctor_id':</strong> " . ($has_reffering_doctor_id ? 'YES' : 'NO') . "<br>";
            echo "<strong>Has 'test_name':</strong> " . ($has_test_name ? 'YES' : 'NO') . "<br>";
            echo "<strong>Has 'order_date':</strong> " . ($has_order_date ? 'YES' : 'NO') . "<br>";
            echo "<strong>Has 'total_amount':</strong> " . ($has_total_amount ? 'YES' : 'NO') . "<br>";
            echo "</div>";
            
            // Test safe radiology query
            $this->db->select('rb.id, rb.date, rb.net_amount, rb.patient_id, rb.doctor_name');
            $this->db->from('radiology_billing rb');
            $this->db->where('rb.patient_id', $patient_id);
            $this->db->limit(3);
            $result = $this->db->get();
            
            if ($result && $result->num_rows() > 0) {
                echo "<p><strong>Sample radiology billing records:</strong></p>";
                echo "<pre>" . print_r($result->result_array(), true) . "</pre>";
            } else {
                echo "<p><em>No radiology billing records found for this patient</em></p>";
            }
        }
        
        // Test the corrected billing calculation
        echo "<hr><h4>🧮 Schema-Corrected Billing Calculation:</h4>";
        $billing = $this->calculatePatientBilling($patient_id);
        echo "<div style='background: #e3f2fd; padding: 15px; margin: 10px 0; border-left: 4px solid #2196f3;'>";
        echo "<table style='width: 100%; border-collapse: collapse;'>";
        echo "<tr><td style='padding: 5px; border: 1px solid #ddd;'><strong>Consultation Total:</strong></td><td style='padding: 5px; border: 1px solid #ddd;'>$" . number_format($billing['consultation_total'], 2) . "</td></tr>";
        echo "<tr><td style='padding: 5px; border: 1px solid #ddd;'><strong>Laboratory Total:</strong></td><td style='padding: 5px; border: 1px solid #ddd;'>$" . number_format($billing['lab_total'], 2) . "</td></tr>";
        echo "<tr><td style='padding: 5px; border: 1px solid #ddd;'><strong>Pharmacy Total:</strong></td><td style='padding: 5px; border: 1px solid #ddd;'>$" . number_format($billing['pharmacy_total'], 2) . "</td></tr>";
        echo "<tr><td style='padding: 5px; border: 1px solid #ddd;'><strong>Radiology Total:</strong></td><td style='padding: 5px; border: 1px solid #ddd;'>$" . number_format($billing['radiology_total'], 2) . "</td></tr>";
        echo "<tr style='background: #f5f5f5; font-weight: bold;'><td style='padding: 8px; border: 1px solid #ddd;'><strong>GRAND TOTAL:</strong></td><td style='padding: 8px; border: 1px solid #ddd;'>$" . number_format($billing['total_billing'], 2) . "</td></tr>";
        echo "<tr><td style='padding: 5px; border: 1px solid #ddd;'><strong>Last Visit:</strong></td><td style='padding: 5px; border: 1px solid #ddd;'>" . ($billing['last_visit'] ?: 'N/A') . "</td></tr>";
        echo "</table>";
        echo "</div>";
        
        // Show corrections made
        echo "<hr><h4>📝 Schema Corrections Applied:</h4>";
        echo "<div style='background: #e8f5e8; padding: 15px; margin: 10px 0; border-left: 4px solid #4caf50;'>";
        echo "<ul>";
        echo "<li><strong>Consultations:</strong> Uses correct 'doctor' column instead of 'doctor_id'</li>";
        echo "<li><strong>Pathology:</strong> Removed problematic CASE WHEN statement for payment_status</li>";
        echo "<li><strong>Radiology:</strong> Removed non-existent columns like 'reffering_doctor_id' and 'order_date'</li>";
        echo "<li><strong>All tables:</strong> Uses only columns that actually exist in your schema</li>";
        echo "</ul>";
        echo "</div>";
        
        echo "<hr>";
        echo "<div style='text-align: center; margin: 20px 0;'>";
        echo "<a href='" . base_url('admin/patient_billing') . "' class='btn btn-primary' style='padding: 10px 20px; margin: 0 10px; text-decoration: none; background: #2196f3; color: white; border-radius: 4px;'>← Back to Patient Billing System</a>";
        echo "<a href='" . base_url('admin/patient_billing/test') . "' class='btn btn-success' style='padding: 10px 20px; margin: 0 10px; text-decoration: none; background: #4caf50; color: white; border-radius: 4px;'>Run Full System Test</a>";
        echo "</div>";
    }

    /**
     * Get all patients endpoint for AJAX - with corrected billing
     */
    public function getAllPatients() {
        header('Content-Type: application/json');
        
        try {
            $patients = $this->getAllPatientsWithBillingSummary();
            
            echo json_encode([
                'status' => 'success',
                'patients' => $patients,
                'count' => count($patients)
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get all patients with billing - endpoint called by the view
     */
    public function getAllPatientsWithBilling() {
        header('Content-Type: application/json');
        
        try {
            $status = $this->input->get('status');
            $sort = $this->input->get('sort');
            
            $patients = $this->getAllPatientsWithBillingSummary();
            
            // Apply filters if needed
            if ($status === 'active') {
                $patients = array_filter($patients, function($p) { return $p['is_active'] === 'yes'; });
            } elseif ($status === 'inactive') {
                $patients = array_filter($patients, function($p) { return $p['is_active'] !== 'yes'; });
            }
            
            // Apply sorting
            if ($sort === 'total_billing') {
                usort($patients, function($a, $b) {
                    return floatval($b['total_billing']) - floatval($a['total_billing']);
                });
            } elseif ($sort === 'last_visit') {
                usort($patients, function($a, $b) {
                    return strcmp($b['last_visit'] ?: '0000-00-00', $a['last_visit'] ?: '0000-00-00');
                });
            } else {
                // Default sort by name
                usort($patients, function($a, $b) {
                    return strcmp($a['patient_name'], $b['patient_name']);
                });
            }
            
            echo json_encode([
                'status' => 'success',
                'patients' => array_values($patients),
                'count' => count($patients)
            ]);
        } catch (Exception $e) {
            log_message('error', 'Error in getAllPatientsWithBilling: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Error: ' . $e->getMessage(),
                'patients' => []
            ]);
        }
    }

    /**
     * Export patient billing data to CSV
     */
    public function exportBillingData() {
        try {
            $patients = $this->getAllPatientsWithBillingSummary();
            
            // Set CSV headers
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="patient_billing_export_' . date('Y-m-d') . '.csv"');
            
            $output = fopen('php://output', 'w');
            
            // CSV column headers
            fputcsv($output, [
                'Patient ID',
                'Patient Name',
                'Age',
                'Gender',
                'Mobile',
                'Email',
                'Blood Group',
                'Consultation Total',
                'Laboratory Total',
                'Pharmacy Total',
                'Radiology Total',
                'Grand Total',
                'Last Visit'
            ]);
            
            // Output patient data
            foreach ($patients as $patient) {
                fputcsv($output, [
                    $patient['id'],
                    $patient['patient_name'],
                    $patient['age'],
                    $patient['gender'],
                    $patient['mobileno'],
                    $patient['email'],
                    $patient['blood_group_name'],
                    number_format($patient['consultation_total'], 2),
                    number_format($patient['lab_total'], 2),
                    number_format($patient['pharmacy_total'], 2),
                    number_format($patient['radiology_total'], 2),
                    number_format($patient['total_billing'], 2),
                    $patient['last_visit']
                ]);
            }
            
            fclose($output);
        } catch (Exception $e) {
            echo "Error exporting data: " . $e->getMessage();
        }
    }

    /**
     * Print patient bill - generates printable billing statement
     */
    public function printBill($patient_id = null) {
        try {
            if (!$patient_id) {
                $patient_id = $this->input->get('patient_id');
            }
            
            if (empty($patient_id)) {
                show_404();
                return;
            }

            // Get patient basic info
            $this->db->select('p.*, bg.name as blood_group_name');
            $this->db->from('patients p');
            $this->db->join('blood_group bg', 'bg.id = p.blood_group_id', 'left');
            $this->db->where('p.id', $patient_id);
            $patient = $this->db->get()->row_array();

            if (!$patient) {
                show_404();
                return;
            }

            // Get billing summary
            $billing_summary = $this->calculatePatientBilling($patient_id);
            
            // Get detailed billing
            $billing_details = $this->getPatientDetailedBilling($patient_id);

            $data = [
                'patient' => $patient,
                'billing_summary' => $billing_summary,
                'billing_details' => $billing_details,
                'currency_symbol' => $this->customlib->getHospitalCurrencyFormat(),
                'hospital_name' => 'Medisure 360',
                'generated_date' => date('d/m/Y'),
                'generated_time' => date('H:i:s')
            ];

            // Load print view
            $this->load->view('admin/patient/print_patient_bill', $data);

        } catch (Exception $e) {
            log_message('error', 'Error generating print bill: ' . $e->getMessage());
            show_404();
        }
    }
    public function getPatientFinancialReport() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            
            if (empty($patient_id)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID is required'
                ]);
                return;
            }

            // Get patient basic info
            $this->db->select('p.*, bg.name as blood_group_name');
            $this->db->from('patients p');
            $this->db->join('blood_group bg', 'bg.id = p.blood_group_id', 'left');
            $this->db->where('p.id', $patient_id);
            $patient = $this->db->get()->row_array();

            if (!$patient) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found'
                ]);
                return;
            }

            // Get billing summary..
            $billing_summary = $this->calculatePatientBilling($patient_id);
            
            // Get detailed billing
            $billing_details = $this->getPatientDetailedBilling($patient_id);

            $report = [
                'patient_info' => $patient,
                'billing_summary' => $billing_summary,
                'billing_details' => $billing_details,
                'report_generated' => date('Y-m-d H:i:s')
            ];

            echo json_encode([
                'status' => 'success',
                'report' => $report
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error generating report: ' . $e->getMessage()
            ]);
        }
    }
}
?>