<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * P9 Form Controller
 * Handles generation of P9 tax certificates for Kenya Revenue Authority (KRA)
 */
class P9 extends Admin_Controller {

    function __construct() {
        parent::__construct();
        $this->load->model(['hr_payroll_model', 'hr_model', 'p9_model']);
        $this->load->library(['pdf', 'email']);
        $this->load->helper(['url', 'download', 'file']);
    }

    /**
     * P9 Management Dashboard
     */
    public function index() {
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'p9');

        $data['title'] = 'P9 Tax Certificates';
        $data['departments'] = $this->hr_model->getDepartments();
        $data['current_year'] = date('Y');
        $data['available_years'] = $this->p9_model->getAvailableYears();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/p9/index', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Generate P9 for individual staff
     */
    public function generate($staff_id, $year = null) {
        try {
            if (!$year) {
                $year = date('Y');
            }

            $p9_data = $this->p9_model->generateP9Data($staff_id, $year);
            
            if (!$p9_data) {
                show_404();
                return;
            }

            // Generate PDF
            $this->generateP9PDF($p9_data);
            
        } catch (Exception $e) {
            log_message('error', 'P9 generation error: ' . $e->getMessage());
            show_error('Failed to generate P9: ' . $e->getMessage());
        }
    }

    /**
     * Generate P9 PDF
     */
    private function generateP9PDF($p9_data) {
        // Initialize PDF with A4 size
        $this->pdf->SetCreator('Hospital Management System');
        $this->pdf->SetAuthor('HR Department');
        $this->pdf->SetTitle('P9 Tax Certificate - ' . $p9_data['staff']->name . ' ' . $p9_data['staff']->surname);
        
        $this->pdf->AddPage();
        $this->pdf->SetFont('helvetica', '', 9);

        // Generate HTML content
        $html = $this->generateP9HTML($p9_data);
        
        // Write HTML to PDF
        $this->pdf->writeHTML($html, true, false, true, false, '');
        
        // Output PDF
        $filename = 'P9_' . $p9_data['staff']->employee_id . '_' . $p9_data['year'] . '.pdf';
        $this->pdf->Output($filename, 'I'); // 'I' for inline display
    }

    /**
     * Generate P9 HTML content following KRA format
     */
    private function generateP9HTML($p9_data) {
        $staff = $p9_data['staff'];
        $year = $p9_data['year'];
        $monthly_data = $p9_data['monthly_data'];
        $totals = $p9_data['totals'];
        $employer = $this->getEmployerDetails();

        $html = '
        <style>
            body { font-family: Arial, sans-serif; font-size: 8pt; }
            .header { text-align: center; margin-bottom: 20px; }
            .title { font-size: 12pt; font-weight: bold; }
            .form-title { font-size: 10pt; font-weight: bold; margin: 10px 0; }
            .employer-section, .employee-section { margin-bottom: 15px; }
            .section-title { font-weight: bold; background-color: #f0f0f0; padding: 3px; margin-bottom: 5px; }
            .info-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .info-table td { padding: 2px 5px; border: 1px solid #ccc; }
            .monthly-table { width: 100%; border-collapse: collapse; font-size: 7pt; }
            .monthly-table th, .monthly-table td { padding: 2px 3px; border: 1px solid #000; text-align: center; }
            .monthly-table th { background-color: #f0f0f0; font-weight: bold; }
            .amount { text-align: right; }
            .total-row { font-weight: bold; background-color: #f9f9f9; }
            .signature-section { margin-top: 30px; }
            .signature-table { width: 100%; }
            .signature-table td { padding: 20px 10px; vertical-align: top; }
        </style>

        <div class="header">
            <div class="title">REPUBLIC OF KENYA</div>
            <div class="title">KENYA REVENUE AUTHORITY</div>
            <div class="form-title">TAX DEDUCTION CARD YEAR ' . $year . '</div>
            <div class="form-title">(To be completed by Employer)</div>
        </div>

        <!-- Employer Details -->
        <div class="employer-section">
            <div class="section-title">EMPLOYER DETAILS</div>
            <table class="info-table">
                <tr>
                    <td style="width: 25%;"><strong>Employer Name:</strong></td>
                    <td style="width: 50%;">' . strtoupper($employer['name']) . '</td>
                    <td style="width: 25%;"><strong>Employer PIN:</strong></td>
                    <td>' . ($employer['pin'] ?? 'P000000000A') . '</td>
                </tr>
                <tr>
                    <td><strong>Employer Address:</strong></td>
                    <td colspan="3">' . ($employer['address'] ?? '') . '</td>
                </tr>
            </table>
        </div>

        <!-- Employee Details -->
        <div class="employee-section">
            <div class="section-title">EMPLOYEE DETAILS</div>
            <table class="info-table">
                <tr>
                    <td style="width: 25%;"><strong>Employee Name:</strong></td>
                    <td style="width: 50%;">' . strtoupper($staff->name . ' ' . $staff->surname) . '</td>
                    <td style="width: 25%;"><strong>Employee PIN:</strong></td>
                    <td>' . ($staff->kra_pin ?? 'A000000000A') . '</td>
                </tr>
                <tr>
                    <td><strong>Employee No:</strong></td>
                    <td>' . $staff->employee_id . '</td>
                    <td><strong>Designation:</strong></td>
                    <td>' . ($staff->role_name ?? '') . '</td>
                </tr>
                <tr>
                    <td><strong>Residence:</strong></td>
                    <td>' . ($staff->address ?? '') . '</td>
                    <td><strong>Department:</strong></td>
                    <td>' . ($staff->department_name ?? '') . '</td>
                </tr>
            </table>
        </div>

        <!-- Monthly Breakdown -->
        <div class="section-title">MONTHLY BREAKDOWN</div>
        <table class="monthly-table">
            <thead>
                <tr>
                    <th rowspan="2">Month</th>
                    <th rowspan="2">Basic Salary</th>
                    <th rowspan="2">Benefits Non-Cash</th>
                    <th rowspan="2">Value of Quarters</th>
                    <th rowspan="2">Total Gross Pay</th>
                    <th rowspan="2">30% of Cash Pay</th>
                    <th rowspan="2">Actual Contribution</th>
                    <th rowspan="2">Owner Occupier Interest</th>
                    <th rowspan="2">Retirement Contribution</th>
                    <th rowspan="2">Insurance Relief</th>
                    <th rowspan="2">Taxable Pay</th>
                    <th rowspan="2">Tax Charged</th>
                    <th rowspan="2">Monthly Relief</th>
                    <th rowspan="2">Insurance Relief</th>
                    <th rowspan="2">PAYE Tax</th>
                </tr>
                <tr>
                    <th>A</th>
                    <th>B</th>
                    <th>C</th>
                    <th>D</th>
                    <th>E1</th>
                    <th>E2</th>
                    <th>E3</th>
                    <th>F</th>
                    <th>G</th>
                    <th>H</th>
                    <th>J</th>
                    <th>K</th>
                    <th>L</th>
                    <th>M</th>
                </tr>
            </thead>
            <tbody>';

        // Generate monthly rows
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        foreach ($months as $month_num => $month_name) {
            $month_data = isset($monthly_data[$month_num]) ? $monthly_data[$month_num] : null;
            
            $html .= '<tr>';
            $html .= '<td>' . $month_name . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['basic_salary'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['benefits_non_cash'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['value_of_quarters'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['gross_pay'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['thirty_percent_cash'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['actual_contribution'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['owner_occupier'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['retirement_contribution'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['insurance_relief'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['taxable_pay'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['tax_charged'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['monthly_relief'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['insurance_relief_month'], 2) : '0.00') . '</td>';
            $html .= '<td class="amount">' . ($month_data ? number_format($month_data['paye_tax'], 2) : '0.00') . '</td>';
            $html .= '</tr>';
        }

        // Totals row
        $html .= '<tr class="total-row">';
        $html .= '<td><strong>TOTALS</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['basic_salary'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['benefits_non_cash'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['value_of_quarters'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['gross_pay'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['thirty_percent_cash'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['actual_contribution'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['owner_occupier'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['retirement_contribution'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['insurance_relief'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['taxable_pay'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['tax_charged'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['monthly_relief'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['insurance_relief_total'], 2) . '</strong></td>';
        $html .= '<td class="amount"><strong>' . number_format($totals['paye_tax'], 2) . '</strong></td>';
        $html .= '</tr>';

        $html .= '</tbody>
        </table>

        <!-- Signature Section -->
        <div class="signature-section">
            <table class="signature-table">
                <tr>
                    <td style="width: 50%;">
                        <strong>EMPLOYEE SIGNATURE:</strong><br>
                        <br><br>
                        ________________________<br>
                        Date: ________________
                    </td>
                    <td style="width: 50%;">
                        <strong>EMPLOYER SIGNATURE:</strong><br>
                        <br><br>
                        ________________________<br>
                        Date: ________________<br>
                        Official Stamp:
                    </td>
                </tr>
            </table>
        </div>

        <div style="margin-top: 20px; font-size: 7pt; text-align: center;">
            Generated on ' . date('d/m/Y H:i:s') . ' | ' . $employer['name'] . ' Payroll System<br>
            This is a computer generated document and does not require a signature
        </div>';

        return $html;
    }

    /**
     * Bulk P9 generation
     */
    public function bulkGenerate() {
        if ($this->input->method() == 'post') {
            header('Content-Type: application/json');
            
            try {
                $year = $this->input->post('year');
                $department_id = $this->input->post('department_id');
                $delivery_method = $this->input->post('delivery_method'); // email, download, both
                
                $result = $this->p9_model->generateBulkP9($year, $department_id, $delivery_method);
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Bulk P9 generation failed: ' . $e->getMessage()
                ]);
            }
        } else {
            // Load bulk generation form
            $data['title'] = 'Bulk Generate P9 Forms';
            $data['departments'] = $this->hr_model->getDepartments();
            $data['available_years'] = $this->p9_model->getAvailableYears();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/hr/p9/bulk_generate', $data);
            $this->load->view('layout/footer', $data);
        }
    }

    /**
     * Email P9 to staff
     */
    public function email($staff_id, $year) {
        header('Content-Type: application/json');
        
        try {
            $result = $this->p9_model->emailP9($staff_id, $year);
            echo json_encode($result);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Email failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get P9 list for DataTables
     */
    public function getP9List() {
        header('Content-Type: application/json');
        
        try {
            $year = $this->input->post('year') ?: date('Y');
            $department_id = $this->input->post('department_id');
            
            $list = $this->p9_model->getP9ListDatatables($year, $department_id);
            $data = array();
            $no = $_POST['start'];

            foreach ($list as $item) {
                $no++;
                $row = array();
                $row[] = $no;
                $row[] = $item->employee_id;
                $row[] = $item->name . ' ' . $item->surname;
                $row[] = $item->department_name;
                $row[] = $item->kra_pin ?: 'Not Set';
                $row[] = 'KSh ' . number_format($item->total_gross, 2);
                $row[] = 'KSh ' . number_format($item->total_paye, 2);
                
                // Status
                $status = $this->p9_model->getP9Status($item->id, $year);
                $status_badge = '<span class="label label-' . ($status['generated'] ? 'success' : 'warning') . '">' . 
                               ($status['generated'] ? 'Generated' : 'Pending') . '</span>';
                $row[] = $status_badge;
                
                // Actions
                $actions = '<div class="btn-group btn-group-xs">';
                $actions .= '<a href="' . base_url() . 'admin/p9/generate/' . $item->id . '/' . $year . '" class="btn btn-primary" target="_blank" title="Generate P9"><i class="fa fa-file-pdf-o"></i></a>';
                $actions .= '<button type="button" class="btn btn-success" onclick="emailP9(' . $item->id . ', ' . $year . ')" title="Email P9"><i class="fa fa-envelope"></i></button>';
                $actions .= '<button type="button" class="btn btn-info" onclick="viewP9Details(' . $item->id . ', ' . $year . ')" title="View Details"><i class="fa fa-eye"></i></button>';
                $actions .= '</div>';
                
                $row[] = $actions;
                $data[] = $row;
            }

            $output = array(
                "draw" => $_POST['draw'],
                "recordsTotal" => $this->p9_model->countAllP9($year, $department_id),
                "recordsFiltered" => $this->p9_model->countFilteredP9($year, $department_id),
                "data" => $data,
            );

            echo json_encode($output);
            
        } catch (Exception $e) {
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    /**
     * P9 Analytics Dashboard
     */
    public function analytics() {
        $data['title'] = 'P9 Analytics';
        $data['available_years'] = $this->p9_model->getAvailableYears();
        $data['current_year'] = date('Y');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/p9/analytics', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Export P9 data for KRA submission
     */
    public function exportForKRA() {
        $year = $this->input->get('year') ?: date('Y');
        $department_id = $this->input->get('department_id');
        
        try {
            $kra_data = $this->p9_model->generateKRAExport($year, $department_id);
            
            // Generate CSV file
            $filename = 'P9_KRA_Export_' . $year . '_' . date('YmdHis') . '.csv';
            
            $this->load->helper('csv');
            array_to_csv($kra_data['data'], $filename);
            
        } catch (Exception $e) {
            show_error('KRA export failed: ' . $e->getMessage());
        }
    }

    /**
     * Get employer details
     */
    private function getEmployerDetails() {
        return [
            'name' => $this->config->item('hospital_name') ?: 'Hospital Management System',
            'address' => $this->config->item('hospital_address') ?: '',
            'pin' => $this->config->item('employer_pin') ?: 'P000000000A',
            'phone' => $this->config->item('hospital_phone') ?: '',
            'email' => $this->config->item('hospital_email') ?: ''
        ];
    }

    /**
     * Setup P9 system
     */
    public function setup() {
        try {
            $this->p9_model->createP9Tables();
            $this->session->set_flashdata('success', 'P9 system setup completed successfully');
        } catch (Exception $e) {
            $this->session->set_flashdata('error', 'Setup failed: ' . $e->getMessage());
        }
        
        redirect('admin/p9');
    }
}