<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Marketing Module Controller
 * Provides comprehensive marketing analytics and campaign management
 * 
 * @author Marketing Analytics System
 * @version 1.0
 */
class Marketing extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        
        // Load required models
        $this->load->model('Marketing_model');
        $this->load->model('Patient_model');
        $this->load->model('Financial_model');
        
        // Check permissions
        $this->auth->check_privilege('marketing_module', 'can_view');
        
        // Load helpers
        $this->load->helper('date');
        $this->load->library('form_validation');
    }

    /**
     * Marketing Dashboard - Main overview
     */
    public function index() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'dashboard');
        
        $data['title'] = 'Marketing Dashboard';
        
        // Get date range
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Key Performance Indicators
        $data['kpi'] = $this->Marketing_model->get_kpi_metrics($date_from, $date_to);
        
        // Patient acquisition sources
        $data['acquisition_sources'] = $this->Marketing_model->get_acquisition_sources($date_from, $date_to);
        
        // Revenue by department
        $data['revenue_by_dept'] = $this->Marketing_model->get_revenue_by_department($date_from, $date_to);
        
        // Campaign performance
        $data['campaigns'] = $this->Marketing_model->get_active_campaigns();
        
        // Patient demographics
        $data['demographics'] = $this->Marketing_model->get_patient_demographics();
        
        // Referral statistics
        $data['referral_stats'] = $this->Marketing_model->get_referral_statistics($date_from, $date_to);
        
        // Service utilization
        $data['service_utilization'] = $this->Marketing_model->get_service_utilization($date_from, $date_to);
        
        // Monthly trends
        $data['monthly_trends'] = $this->Marketing_model->get_monthly_trends();
        
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Patient Acquisition Analytics
     */
    public function acquisition() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'acquisition');
        
        $data['title'] = 'Patient Acquisition Analytics';
        
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Acquisition funnel
        $data['funnel'] = $this->Marketing_model->get_acquisition_funnel($date_from, $date_to);
        
        // Source performance
        $data['source_performance'] = $this->Marketing_model->get_source_performance($date_from, $date_to);
        
        // Conversion rates
        $data['conversion_rates'] = $this->Marketing_model->get_conversion_rates($date_from, $date_to);
        
        // Geographic distribution
        $data['geographic_data'] = $this->Marketing_model->get_geographic_distribution();
        
        // Channel ROI
        $data['channel_roi'] = $this->Marketing_model->calculate_channel_roi($date_from, $date_to);
        
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/acquisition', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Campaign Management
     */
    public function campaigns() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'campaigns');
        
        $data['title'] = 'Marketing Campaigns';
        
        // Get all campaigns
        $data['campaigns'] = $this->Marketing_model->get_all_campaigns();
        
        // Campaign performance metrics
        foreach ($data['campaigns'] as &$campaign) {
            $campaign->metrics = $this->Marketing_model->get_campaign_metrics($campaign->id);
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/campaigns', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Create new campaign
     */
    public function create_campaign() {
        $this->form_validation->set_rules('campaign_name', 'Campaign Name', 'required|trim');
        $this->form_validation->set_rules('campaign_type', 'Campaign Type', 'required');
        $this->form_validation->set_rules('start_date', 'Start Date', 'required');
        $this->form_validation->set_rules('budget', 'Budget', 'numeric');
        
        if ($this->form_validation->run() == FALSE) {
            $this->session->set_flashdata('error', validation_errors());
            redirect('admin/marketing/campaigns');
        }
        
        $campaign_data = [
            'campaign_name' => $this->input->post('campaign_name'),
            'campaign_type' => $this->input->post('campaign_type'),
            'description' => $this->input->post('description'),
            'start_date' => $this->input->post('start_date'),
            'end_date' => $this->input->post('end_date'),
            'budget' => $this->input->post('budget'),
            'target_audience' => $this->input->post('target_audience'),
            'channels' => json_encode($this->input->post('channels')),
            'objectives' => $this->input->post('objectives'),
            'status' => 'active',
            'created_by' => $this->session->userdata('admin')['id'],
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $campaign_id = $this->Marketing_model->create_campaign($campaign_data);
        
        if ($campaign_id) {
            $this->session->set_flashdata('success', 'Campaign created successfully');
            
            // Log activity
            $this->Marketing_model->log_activity('campaign_created', $campaign_id);
        } else {
            $this->session->set_flashdata('error', 'Failed to create campaign');
        }
        
        redirect('admin/marketing/campaigns');
    }

    /**
     * Revenue Analytics
     */
    public function revenue_analytics() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'revenue');
        
        $data['title'] = 'Revenue Analytics';
        
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Revenue breakdown
        $data['revenue_breakdown'] = $this->Marketing_model->get_revenue_breakdown($date_from, $date_to);
        
        // Service profitability
        $data['service_profitability'] = $this->Marketing_model->analyze_service_profitability($date_from, $date_to);
        
        // Patient lifetime value
        $data['patient_ltv'] = $this->Marketing_model->calculate_patient_lifetime_value();
        
        // Revenue forecasting
        $data['revenue_forecast'] = $this->Marketing_model->forecast_revenue();
        
        // Payer mix analysis
        $data['payer_mix'] = $this->Marketing_model->analyze_payer_mix($date_from, $date_to);
        
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/revenue_analytics', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Patient Segmentation
     */
    public function segmentation() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'segmentation');
        
        $data['title'] = 'Patient Segmentation';
        
        // Segment patients
        $data['segments'] = $this->Marketing_model->segment_patients();
        
        // Behavioral patterns
        $data['behavioral_patterns'] = $this->Marketing_model->analyze_patient_behavior();
        
        // Service preferences
        $data['service_preferences'] = $this->Marketing_model->get_service_preferences_by_segment();
        
        // Retention analysis
        $data['retention_analysis'] = $this->Marketing_model->analyze_retention_by_segment();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/segmentation', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Competitor Analysis
     */
    public function competitor_analysis() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'competitors');
        
        $data['title'] = 'Competitor Analysis';
        
        // Market share estimation
        $data['market_share'] = $this->Marketing_model->estimate_market_share();
        
        // Service comparison
        $data['service_comparison'] = $this->Marketing_model->compare_services();
        
        // Pricing analysis
        $data['pricing_analysis'] = $this->Marketing_model->analyze_competitive_pricing();
        
        // SWOT analysis
        $data['swot'] = $this->Marketing_model->get_swot_analysis();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/competitor_analysis', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Marketing Reports
     */
    public function reports() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'reports');
        
        $data['title'] = 'Marketing Reports';
        
        // Available report types
        $data['report_types'] = [
            'executive_summary' => 'Executive Summary',
            'acquisition_report' => 'Patient Acquisition Report',
            'campaign_performance' => 'Campaign Performance Report',
            'revenue_analysis' => 'Revenue Analysis Report',
            'patient_satisfaction' => 'Patient Satisfaction Report',
            'referral_analysis' => 'Referral Analysis Report',
            'roi_analysis' => 'ROI Analysis Report',
            'market_trends' => 'Market Trends Report'
        ];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/reports', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Generate Report
     */
    public function generate_report() {
        $report_type = $this->input->post('report_type');
        $date_from = $this->input->post('date_from');
        $date_to = $this->input->post('date_to');
        $format = $this->input->post('format'); // pdf, excel, csv
        
        $report_data = $this->Marketing_model->generate_report($report_type, $date_from, $date_to);
        
        switch ($format) {
            case 'pdf':
                $this->generate_pdf_report($report_data, $report_type);
                break;
            case 'excel':
                $this->generate_excel_report($report_data, $report_type);
                break;
            case 'csv':
                $this->generate_csv_report($report_data, $report_type);
                break;
            default:
                $this->display_report($report_data, $report_type);
        }
    }

    /**
     * Patient Feedback & Reviews
     */
    public function feedback() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'feedback');
        
        $data['title'] = 'Patient Feedback & Reviews';
        
        // Get feedback data
        $data['feedback'] = $this->Marketing_model->get_patient_feedback();
        
        // Sentiment analysis
        $data['sentiment'] = $this->Marketing_model->analyze_feedback_sentiment();
        
        // NPS Score
        $data['nps_score'] = $this->Marketing_model->calculate_nps_score();
        
        // Service ratings
        $data['service_ratings'] = $this->Marketing_model->get_service_ratings();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/feedback', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Marketing Calendar
     */
    public function calendar() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'calendar');
        
        $data['title'] = 'Marketing Calendar';
        
        // Get scheduled campaigns and events
        $data['events'] = $this->Marketing_model->get_marketing_events();
        
        // Health awareness days
        $data['awareness_days'] = $this->Marketing_model->get_health_awareness_days();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/calendar', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Email Marketing
     */
    public function email_marketing() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'email');
        
        $data['title'] = 'Email Marketing';
        
        // Email campaigns
        $data['email_campaigns'] = $this->Marketing_model->get_email_campaigns();
        
        // Subscriber lists
        $data['subscriber_lists'] = $this->Marketing_model->get_subscriber_lists();
        
        // Email templates
        $data['templates'] = $this->Marketing_model->get_email_templates();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/email_marketing', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Social Media Analytics
     */
    public function social_media() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'social');
        
        $data['title'] = 'Social Media Analytics';
        
        // Social media metrics
        $data['social_metrics'] = $this->Marketing_model->get_social_media_metrics();
        
        // Engagement rates
        $data['engagement'] = $this->Marketing_model->calculate_engagement_rates();
        
        // Content performance
        $data['content_performance'] = $this->Marketing_model->analyze_content_performance();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/social_media', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * API Endpoints for AJAX calls
     */
    
    /**
     * Get real-time KPI data
     */
    public function get_kpi_data() {
        header('Content-Type: application/json');
        
        $period = $this->input->get('period') ?: 'month';
        $kpi_data = $this->Marketing_model->get_realtime_kpi($period);
        
        echo json_encode([
            'status' => 'success',
            'data' => $kpi_data
        ]);
    }

    /**
     * Track campaign conversion
     */
    public function track_conversion() {
        header('Content-Type: application/json');
        
        $campaign_id = $this->input->post('campaign_id');
        $patient_id = $this->input->post('patient_id');
        $conversion_type = $this->input->post('conversion_type');
        
        $result = $this->Marketing_model->track_conversion($campaign_id, $patient_id, $conversion_type);
        
        echo json_encode([
            'status' => $result ? 'success' : 'error',
            'message' => $result ? 'Conversion tracked' : 'Failed to track conversion'
        ]);
    }

    /**
     * Update campaign status
     */
    public function update_campaign_status() {
        header('Content-Type: application/json');
        
        $campaign_id = $this->input->post('campaign_id');
        $status = $this->input->post('status');
        
        $result = $this->Marketing_model->update_campaign_status($campaign_id, $status);
        
        echo json_encode([
            'status' => $result ? 'success' : 'error',
            'message' => $result ? 'Status updated' : 'Failed to update status'
        ]);
    }

    /**
     * Get patient journey map
     */
    public function patient_journey($patient_id = null) {
        header('Content-Type: application/json');
        
        if (!$patient_id) {
            echo json_encode(['status' => 'error', 'message' => 'Patient ID required']);
            return;
        }
        
        $journey = $this->Marketing_model->get_patient_journey($patient_id);
        
        echo json_encode([
            'status' => 'success',
            'data' => $journey
        ]);
    }

    /**
     * Export marketing data
     */
    public function export_data() {
        $data_type = $this->input->get('type');
        $format = $this->input->get('format');
        $date_from = $this->input->get('date_from');
        $date_to = $this->input->get('date_to');
        
        $data = $this->Marketing_model->export_marketing_data($data_type, $date_from, $date_to);
        
        switch ($format) {
            case 'csv':
                $this->export_csv($data, $data_type);
                break;
            case 'excel':
                $this->export_excel($data, $data_type);
                break;
            case 'json':
                header('Content-Type: application/json');
                echo json_encode($data);
                break;
        }
    }

    /**
     * Marketing settings
     */
    public function settings() {
        $this->session->set_userdata('top_menu', 'Marketing');
        $this->session->set_userdata('sub_menu', 'settings');
        
        $data['title'] = 'Marketing Settings';
        
        // Get current settings
        $data['settings'] = $this->Marketing_model->get_settings();
        
        // Attribution models
        $data['attribution_models'] = [
            'first_touch' => 'First Touch',
            'last_touch' => 'Last Touch',
            'linear' => 'Linear',
            'time_decay' => 'Time Decay',
            'position_based' => 'Position Based'
        ];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/marketing/settings', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Update marketing settings
     */
    public function update_settings() {
        $settings = $this->input->post('settings');
        
        $result = $this->Marketing_model->update_settings($settings);
        
        if ($result) {
            $this->session->set_flashdata('success', 'Settings updated successfully');
        } else {
            $this->session->set_flashdata('error', 'Failed to update settings');
        }
        
        redirect('admin/marketing/settings');
    }

    /**
     * Private helper methods
     */
    
    private function generate_pdf_report($data, $type) {
        $this->load->library('pdf');
        
        $html = $this->load->view('admin/marketing/reports/' . $type, $data, true);
        
        $this->pdf->createPDF($html, $type . '_' . date('Y-m-d'), true);
    }

    private function generate_excel_report($data, $type) {
        $this->load->library('excel');
        
        // Excel generation logic
        $filename = $type . '_' . date('Y-m-d') . '.xlsx';
        
        // Implementation depends on your Excel library
        $this->excel->generate($data, $filename);
    }

    private function export_csv($data, $filename) {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Write headers
        if (!empty($data)) {
            fputcsv($output, array_keys((array)$data[0]));
        }
        
        // Write data
        foreach ($data as $row) {
            fputcsv($output, (array)$row);
        }
        
        fclose($output);
    }
}