<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Lab_setup extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('lab_setup_model');
        $this->load->model('laboratory_model');
        $this->load->model('staff_model');
        $this->load->model('setting_model');
        $this->load->library('form_validation');
        $this->load->library('datatables');
        $this->load->helper('customfield_helper');
        $this->load->helper('custom');
        
        // Get system currency
        $this->setSystemCurrency();
    }

    private function setSystemCurrency() {
        try {
            $settings = $this->setting_model->get();
            $this->system_currency = isset($settings[0]['currency_symbol']) ? $settings[0]['currency_symbol'] : '$';
        } catch (Exception $e) {
            $this->system_currency = '$';
        }
    }

    // =============================================
    // DASHBOARD
    // =============================================

    public function index() {
        redirect('admin/lab_setup/dashboard');
    }

    public function dashboard() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/dashboard');

        $data['title'] = 'Laboratory Setup Dashboard';
        $data['currency_symbol'] = $this->system_currency;
        
        try {
            $data['stats'] = $this->lab_setup_model->getSetupStats();
        } catch (Exception $e) {
            $data['stats'] = array(
                'total_categories' => 0,
                'total_tests' => 0,
                'total_units' => 0,
                'total_reference_ranges' => 0,
                'total_lab_results' => 0,
                'expensive_tests' => array(),
                'tests_by_category' => array()
            );
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // CATEGORIES MANAGEMENT
    // =============================================

    public function categories() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/categories');

        $data['title'] = 'Lab Test Categories';
        $data['categories'] = $this->lab_setup_model->getTestCategories();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/categories', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add_category() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('category_name', 'Category Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('description', 'Description', 'trim|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'category_name' => form_error('category_name'),
                'description' => form_error('description'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'category_name' => $this->input->post('category_name'),
                'description' => $this->input->post('description'),
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            );
            
            $result = $this->lab_setup_model->addTestCategory($data);
            if ($result) {
                $array = array('status' => 'success', 'error' => '', 'message' => 'Category added successfully.');
            } else {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to add category.');
            }
        }
        echo json_encode($array);
    }

    public function edit_category($id) {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
            access_denied();
        }
        
        $data['category'] = $this->lab_setup_model->getTestCategory($id);
        $this->load->view('admin/lab_setup/edit_category_modal', $data);
    }

    public function update_category($id) {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
            access_denied();
        }

        $this->form_validation->set_rules('category_name', 'Category Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('description', 'Description', 'trim|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'category_name' => form_error('category_name'),
                'description' => form_error('description'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $existing = $this->lab_setup_model->checkCategoryExists($this->input->post('category_name'), $id);
            
            if ($existing) {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Category name already exists.');
            } else {
                $data = array(
                    'category_name' => $this->input->post('category_name'),
                    'description' => $this->input->post('description'),
                    'updated_at' => date('Y-m-d H:i:s')
                );
                
                $result = $this->lab_setup_model->updateTestCategory($id, $data);
                if ($result) {
                    $array = array('status' => 'success', 'error' => '', 'message' => 'Category updated successfully.');
                } else {
                    $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to update category.');
                }
            }
        }
        echo json_encode($array);
    }

    public function delete_category($id) {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_delete')) {
            access_denied();
        }

        $tests_count = $this->lab_setup_model->getTestsByCategoryCount($id);
        
        if ($tests_count > 0) {
            $array = array('status' => 'fail', 'message' => 'Cannot delete category. It is being used by ' . $tests_count . ' test(s).');
        } else {
            $result = $this->lab_setup_model->deleteTestCategory($id);
            if ($result) {
                $array = array('status' => 'success', 'message' => 'Category deleted successfully.');
            } else {
                $array = array('status' => 'fail', 'message' => 'Failed to delete category.');
            }
        }
        echo json_encode($array);
    }

    // =============================================
    // TESTS MANAGEMENT
    // =============================================

public function tests() {
    if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
        access_denied();
    }

    $this->session->set_userdata('top_menu', 'lab');
    $this->session->set_userdata('sub_menu', 'lab_setup/tests');

    $data['title'] = 'Lab Tests Setup';
    $data['currency_symbol'] = $this->system_currency;
    
    // Use your actual table names
    $data['categories'] = array();
    if ($this->db->table_exists('lab_test_categories')) {
        $this->db->select('id, category_name');
        $this->db->where('is_active', 1);
        $this->db->order_by('category_name');
        $data['categories'] = $this->db->get('lab_test_categories')->result_array();
    }
    
    $data['units'] = array();
    if ($this->db->table_exists('lab_units')) {
        $this->db->select('id, unit_name, unit_symbol');
        $this->db->where('is_active', 1);
        $this->db->order_by('unit_name');
        $data['units'] = $this->db->get('lab_units')->result_array();
    }

    $this->load->view('layout/header', $data);
    $this->load->view('admin/lab_setup/tests', $data);
    $this->load->view('layout/footer', $data);
}

   /* public function get_tests_datatable() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            echo json_encode(array('error' => 'Access denied'));
            return;
        }

        $requestData = $_REQUEST;
        
        $start = isset($requestData['start']) ? intval($requestData['start']) : 0;
        $length = isset($requestData['length']) ? intval($requestData['length']) : 10;
        $search = isset($requestData['search']['value']) ? $requestData['search']['value'] : '';
        
        try {
            $tests = $this->lab_setup_model->getTestsForDataTable($start, $length, $search);
            $totalTests = $this->lab_setup_model->getTestsCount($search);
            
            $data = array();
            foreach ($tests as $test) {
                $row = array();
                $row[] = htmlspecialchars($test['test_name']);
                $row[] = htmlspecialchars($test['short_name']);
                $row[] = htmlspecialchars($test['category_name']);
                $row[] = $test['unit_name'] ? htmlspecialchars($test['unit_name']) : '-';
                $row[] = $this->system_currency . number_format($test['cost'], 2);
                $row[] = $test['report_days'] ? $test['report_days'] . ' days' : '-';
                $row[] = $test['is_active'] ? '<span class="label label-success">Active</span>' : '<span class="label label-danger">Inactive</span>';
                
                $actions = '<div class="btn-group">';
                if ($this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
                    $actions .= '<a href="javascript:void(0)" onclick="editTest(' . $test['id'] . ')" class="btn btn-default btn-xs" title="Edit"><i class="fa fa-pencil"></i></a>';
                }
                if ($this->rbac->hasPrivilege('lab_setup', 'can_delete')) {
                    $actions .= '<a href="javascript:void(0)" onclick="deleteTest(' . $test['id'] . ')" class="btn btn-default btn-xs" title="Delete"><i class="fa fa-trash"></i></a>';
                }
                $actions .= '<a href="javascript:void(0)" onclick="viewTest(' . $test['id'] . ')" class="btn btn-default btn-xs" title="View"><i class="fa fa-eye"></i></a>';
                $actions .= '</div>';
                
                $row[] = $actions;
                $data[] = $row;
            }

            $output = array(
                "draw" => intval($requestData['draw']),
                "recordsTotal" => $totalTests,
                "recordsFiltered" => $totalTests,
                "data" => $data
            );
            
        } catch (Exception $e) {
            $output = array(
                "draw" => intval($requestData['draw']),
                "recordsTotal" => 0,
                "recordsFiltered" => 0,
                "data" => array(),
                "error" => $e->getMessage()
            );
        }
        
        echo json_encode($output);
    }*/

    /*public function add_test() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('test_name', 'Test Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('short_name', 'Short Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('category_id', 'Category', 'trim|required|numeric|xss_clean');
        $this->form_validation->set_rules('cost', 'Cost', 'trim|required|numeric|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'test_name' => form_error('test_name'),
                'short_name' => form_error('short_name'),
                'category_id' => form_error('category_id'),
                'cost' => form_error('cost'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $existing = $this->lab_setup_model->checkTestExists($this->input->post('test_name'));
            
            if ($existing) {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Test name already exists.');
            } else {
                $data = array(
                    'test_name' => $this->input->post('test_name'),
                    'short_name' => $this->input->post('short_name'),
                    'category_id' => $this->input->post('category_id'),
                    'unit_id' => $this->input->post('unit_id') ? $this->input->post('unit_id') : null,
                    'cost' => $this->input->post('cost'),
                    'report_days' => $this->input->post('report_days') ? $this->input->post('report_days') : null,
                    'method' => $this->input->post('method'),
                    'sample_type' => $this->input->post('sample_type'),
                    'test_type' => $this->input->post('test_type'),
                    'description' => $this->input->post('description'),
                    'is_active' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                );
                
                $result = $this->lab_setup_model->addTest($data);
                if ($result) {
                    $array = array('status' => 'success', 'error' => '', 'message' => 'Test added successfully.');
                } else {
                    $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to add test.');
                }
            }
        }
        echo json_encode($array);
    }*/

    // =============================================
    // UNITS MANAGEMENT
    // =============================================

    public function units() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/units');

        $data['title'] = 'Lab Test Units';
        $data['units'] = $this->lab_setup_model->getUnits();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/units', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add_unit() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('unit_name', 'Unit Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('unit_symbol', 'Unit Symbol', 'trim|required|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'unit_name' => form_error('unit_name'),
                'unit_symbol' => form_error('unit_symbol'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'unit_name' => $this->input->post('unit_name'),
                'unit_symbol' => $this->input->post('unit_symbol'),
                'description' => $this->input->post('description'),
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            );
            
            $result = $this->lab_setup_model->addUnit($data);
            if ($result) {
                $array = array('status' => 'success', 'error' => '', 'message' => 'Unit added successfully.');
            } else {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to add unit.');
            }
        }
        echo json_encode($array);
    }

    // =============================================
    // SYSTEM INITIALIZATION
    // =============================================

    public function initialize_system() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        try {
            $this->lab_setup_model->createLabTables();
            $this->lab_setup_model->addDefaultCategories();
            $this->lab_setup_model->addDefaultUnits();
            $this->lab_setup_model->addDefaultTests();
            
            $array = array('status' => 'success', 'message' => 'Laboratory system initialized successfully!');
        } catch (Exception $e) {
            $array = array('status' => 'fail', 'message' => 'Failed to initialize system: ' . $e->getMessage());
        }
        
        echo json_encode($array);
    }

    public function test_setup() {
        echo "<h2>Laboratory Setup Test</h2>";
        
        try {
            echo "<p>✓ Database connection: OK</p>";
            
            $tables = array('lab_test_categories', 'lab_tests', 'lab_units', 'lab_reference_ranges', 'laboratory');
            
            foreach ($tables as $table) {
                if ($this->db->table_exists($table)) {
                    echo "<p>✓ Table '{$table}': EXISTS</p>";
                } else {
                    echo "<p>✗ Table '{$table}': MISSING</p>";
                }
            }
            
            $categories = $this->lab_setup_model->getTestCategories();
            echo "<p>✓ Categories found: " . count($categories) . "</p>";
            
            $units = $this->lab_setup_model->getUnits();
            echo "<p>✓ Units found: " . count($units) . "</p>";
            
            echo "<p><strong>System Status: Ready for setup</strong></p>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>✗ Error: " . $e->getMessage() . "</p>";
        }
        
        echo "<p><a href='" . base_url() . "admin/lab_setup/initialize_system' class='btn btn-primary'>Initialize System</a></p>";
        echo "<p><a href='" . base_url() . "admin/lab_setup/force_create_tables' class='btn btn-warning'>Force Create Tables</a></p>";
    }

    public function force_create_tables() {
        echo "<h2>Force Creating Laboratory Tables</h2>";
        
        try {
            $this->db->query("
                CREATE TABLE IF NOT EXISTS `lab_test_categories` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `category_name` varchar(100) NOT NULL,
                    `description` text,
                    `is_active` tinyint(1) DEFAULT 1,
                    `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `category_name` (`category_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ");
            echo "<p>✓ lab_test_categories table created</p>";

            $this->db->query("
                CREATE TABLE IF NOT EXISTS `lab_units` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `unit_name` varchar(50) NOT NULL,
                    `unit_symbol` varchar(10) NOT NULL,
                    `description` text,
                    `is_active` tinyint(1) DEFAULT 1,
                    `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `unit_name` (`unit_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ");
            echo "<p>✓ lab_units table created</p>";

            $this->db->query("
                CREATE TABLE IF NOT EXISTS `lab_tests` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `test_name` varchar(150) NOT NULL,
                    `short_name` varchar(50) NOT NULL,
                    `category_id` int(11) NOT NULL,
                    `unit_id` int(11) DEFAULT NULL,
                    `cost` decimal(10,2) NOT NULL DEFAULT 0.00,
                    `report_days` int(3) DEFAULT NULL,
                    `method` varchar(100) DEFAULT NULL,
                    `sample_type` varchar(100) DEFAULT NULL,
                    `test_type` enum('quantitative','qualitative','descriptive') DEFAULT 'qualitative',
                    `description` text,
                    `is_active` tinyint(1) DEFAULT 1,
                    `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `test_name` (`test_name`),
                    KEY `category_id` (`category_id`),
                    KEY `unit_id` (`unit_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ");
            echo "<p>✓ lab_tests table created</p>";

            $this->addDefaultData();
            echo "<p>✓ Default data added</p>";

            echo "<h3 style='color: green;'>✅ All tables created successfully!</h3>";
            echo "<p><a href='" . base_url() . "admin/lab_setup/test_setup' class='btn btn-success'>Test Again</a></p>";
            echo "<p><a href='" . base_url() . "admin/lab_setup/dashboard' class='btn btn-primary'>Go to Dashboard</a></p>";

        } catch (Exception $e) {
            echo "<p style='color: red;'>✗ Error: " . $e->getMessage() . "</p>";
        }
    }

    private function addDefaultData() {
        $categories = array(
            array('category_name' => 'Hematology', 'description' => 'Blood-related tests'),
            array('category_name' => 'Clinical Chemistry', 'description' => 'Chemical analysis tests'),
            array('category_name' => 'Microbiology', 'description' => 'Bacterial and viral tests'),
            array('category_name' => 'Immunology', 'description' => 'Immune system tests'),
            array('category_name' => 'Urine Analysis', 'description' => 'Urine-based tests'),
            array('category_name' => 'Serology', 'description' => 'Blood serum tests')
        );

        foreach ($categories as $category) {
            $this->db->where('category_name', $category['category_name']);
            $existing = $this->db->get('lab_test_categories')->num_rows();
            
            if ($existing == 0) {
                $category['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_test_categories', $category);
            }
        }

        $units = array(
            array('unit_name' => 'Grams per deciliter', 'unit_symbol' => 'g/dL'),
            array('unit_name' => 'Milligrams per deciliter', 'unit_symbol' => 'mg/dL'),
            array('unit_name' => 'Percentage', 'unit_symbol' => '%'),
            array('unit_name' => 'Count', 'unit_symbol' => 'count'),
            array('unit_name' => 'International Units per liter', 'unit_symbol' => 'IU/L')
        );

        foreach ($units as $unit) {
            $this->db->where('unit_name', $unit['unit_name']);
            $existing = $this->db->get('lab_units')->num_rows();
            
            if ($existing == 0) {
                $unit['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_units', $unit);
            }
        }
    }
    
    /**
 * Add new laboratory test
 */
public function add_test() {
    if ($this->input->method() === 'post') {
        // Validation rules
        $this->form_validation->set_rules('test_name', 'Test Name', 'required|trim');
        $this->form_validation->set_rules('short_name', 'Short Name', 'required|trim');
        $this->form_validation->set_rules('category_id', 'Category', 'required');
        $this->form_validation->set_rules('cost', 'Cost', 'required|numeric|greater_than[0]');
        
        if ($this->form_validation->run() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'error' => $this->form_validation->error_array(),
                'message' => 'Please fix the validation errors'
            ]);
            return;
        }

        // Prepare data
        $data = array(
            'test_name' => $this->input->post('test_name'),
            'short_name' => $this->input->post('short_name'),
            'category_id' => $this->input->post('category_id'),
            'unit_id' => $this->input->post('unit_id') ? $this->input->post('unit_id') : null,
            'cost' => $this->input->post('cost'),
            'report_days' => $this->input->post('report_days') ? $this->input->post('report_days') : 1,
            'test_type' => $this->input->post('test_type'),
            'method' => $this->input->post('method'),
            'sample_type' => $this->input->post('sample_type'),
            'description' => $this->input->post('description'),
            'is_gcc' => $this->input->post('is_gcc') ? 1 : 0,
            'is_active' => 1,
            'created_at' => date('Y-m-d H:i:s')
        );

        // Check for duplicate test name
        $this->db->where('test_name', $data['test_name']);
        $existing = $this->db->get('lab_tests')->row();
        
        if ($existing) {
            echo json_encode([
                'status' => 'error',
                'message' => 'A test with this name already exists'
            ]);
            return;
        }

        // Insert the test
        if ($this->db->insert('lab_tests', $data)) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Laboratory test added successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to add laboratory test'
            ]);
        }
    }
}

/**
 * Get tests data for DataTable
 */
/**
 * Get tests data for DataTables with GCC column
 */
public function get_tests_datatable() {
    // Set proper headers
    header('Content-Type: application/json; charset=utf-8');
    
    try {
        // Check if lab_tests table exists
        if (!$this->db->table_exists('lab_tests')) {
            echo json_encode([
                'draw' => intval($this->input->post('draw') ?: 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Lab tests table does not exist'
            ]);
            return;
        }

        // Get DataTables parameters
        $draw = intval($this->input->post('draw') ?: 1);
        $start = intval($this->input->post('start') ?: 0);
        $length = intval($this->input->post('length') ?: 10);
        
        // Search value
        $search_value = '';
        $search_array = $this->input->post('search');
        if ($search_array && isset($search_array['value'])) {
            $search_value = $search_array['value'];
        }

        // Order parameters
        $order_array = $this->input->post('order');
        $order_column_index = 0;
        $order_dir = 'asc';
        
        if ($order_array && isset($order_array[0])) {
            $order_column_index = intval($order_array[0]['column'] ?? 0);
            $order_dir = $order_array[0]['dir'] ?? 'asc';
        }

        // Column mapping - match exactly with your DataTable columns
        $columns = array(
            0 => 'lab_tests.test_name',
            1 => 'lab_tests.short_name', 
            2 => 'category_name',
            3 => 'unit_name',
            4 => 'lab_tests.cost',
            5 => 'lab_tests.report_days',
            6 => 'lab_tests.is_gcc',
            7 => 'lab_tests.is_active'
        );

        $order_column = $columns[$order_column_index] ?? 'lab_tests.test_name';

        // Use your actual table names
        $categories_table = 'lab_test_categories'; // You have this table
        $units_table = 'lab_units'; // You have this table

        // Build query with your actual table structure
        $this->db->select('
            lab_tests.id,
            lab_tests.test_name,
            lab_tests.short_name,
            lab_tests.cost,
            lab_tests.report_days,
            lab_tests.is_active,
            COALESCE(lab_tests.is_gcc, 0) as is_gcc,
            COALESCE(lab_test_categories.category_name, "Uncategorized") as category_name,
            lab_units.unit_name,
            lab_units.unit_symbol
        ');
        
        $this->db->from('lab_tests');
        $this->db->join('lab_test_categories', 'lab_test_categories.id = lab_tests.category_id', 'left');
        $this->db->join('lab_units', 'lab_units.id = lab_tests.unit_id', 'left');

        // Apply search if provided
        if (!empty($search_value)) {
            $this->db->group_start();
            $this->db->like('lab_tests.test_name', $search_value);
            $this->db->or_like('lab_tests.short_name', $search_value);
            $this->db->or_like('lab_test_categories.category_name', $search_value);
            $this->db->or_like('lab_units.unit_name', $search_value);
            $this->db->group_end();
        }

        // Get total count without limit
        $count_sql = $this->db->get_compiled_select();
        $total_records = $this->db->query("SELECT COUNT(*) as count FROM (" . $count_sql . ") as temp")->row()->count;

        // Reset and rebuild query for actual data with pagination
        $this->db->reset_query();
        $this->db->select('
            lab_tests.id,
            lab_tests.test_name,
            lab_tests.short_name,
            lab_tests.cost,
            lab_tests.report_days,
            lab_tests.is_active,
            COALESCE(lab_tests.is_gcc, 0) as is_gcc,
            COALESCE(lab_test_categories.category_name, "Uncategorized") as category_name,
            lab_units.unit_name,
            lab_units.unit_symbol
        ');
        
        $this->db->from('lab_tests');
        $this->db->join('lab_test_categories', 'lab_test_categories.id = lab_tests.category_id', 'left');
        $this->db->join('lab_units', 'lab_units.id = lab_tests.unit_id', 'left');

        // Apply search again
        if (!empty($search_value)) {
            $this->db->group_start();
            $this->db->like('lab_tests.test_name', $search_value);
            $this->db->or_like('lab_tests.short_name', $search_value);
            $this->db->or_like('lab_test_categories.category_name', $search_value);
            $this->db->or_like('lab_units.unit_name', $search_value);
            $this->db->group_end();
        }

        // Apply ordering and limiting
        $this->db->order_by($order_column, $order_dir);
        if ($length != -1) {
            $this->db->limit($length, $start);
        }

        $query = $this->db->get();
        
        if (!$query) {
            throw new Exception('Database query failed');
        }

        $data = array();
        foreach ($query->result() as $row) {
            $sub_array = array();
            
            // Test name
            $sub_array[] = htmlspecialchars($row->test_name);
            
            // Short name
            $sub_array[] = htmlspecialchars($row->short_name ?: 'N/A');
            
            // Category
            $sub_array[] = htmlspecialchars($row->category_name);
            
            // Unit
            $unit_display = 'N/A';
            if ($row->unit_name) {
                $unit_display = htmlspecialchars($row->unit_name);
                if ($row->unit_symbol) {
                    $unit_display .= ' (' . htmlspecialchars($row->unit_symbol) . ')';
                }
            }
            $sub_array[] = $unit_display;
            
            // Cost
            $sub_array[] = 'Ksh. ' . number_format($row->cost, 2);
            
            // Report days
            $sub_array[] = $row->report_days ? $row->report_days . ' day(s)' : '1 day';
            
            // GCC Required
            $is_gcc = intval($row->is_gcc);
            if ($is_gcc == 1) {
                $gcc_badge = '<span class="badge" style="background-color: #f39c12; color: white; font-size: 10px; padding: 3px 6px;">
                                <i class="fa fa-certificate"></i> GCC
                              </span>';
            } else {
                $gcc_badge = '<span class="badge" style="background-color: #95a5a6; color: white; font-size: 10px; padding: 3px 6px;">
                                <i class="fa fa-minus-circle"></i> Regular
                              </span>';
            }
            $sub_array[] = $gcc_badge;
            
            // Status
            if ($row->is_active == 1) {
                $status = '<span class="badge" style="background-color: #28a745; color: white;">Active</span>';
            } else {
                $status = '<span class="badge" style="background-color: #dc3545; color: white;">Inactive</span>';
            }
            $sub_array[] = $status;
            
            // Actions
            $actions = '';
            
            $actions .= '<button type="button" class="btn btn-xs btn-info" onclick="viewTest(' . $row->id . ')" title="View" style="margin-right: 2px;">
                            <i class="fa fa-eye"></i>
                         </button>';
            
            $actions .= '<button type="button" class="btn btn-xs btn-primary" onclick="editTest(' . $row->id . ')" title="Edit" style="margin-right: 2px;">
                            <i class="fa fa-edit"></i>
                         </button>';
            
            // GCC Toggle button
            $gcc_action = $is_gcc == 1 ? 'Remove GCC' : 'Require GCC';
            $gcc_icon = $is_gcc == 1 ? 'fa-minus-circle' : 'fa-certificate';
            $gcc_class = $is_gcc == 1 ? 'btn-warning' : 'btn-info';
            
            $actions .= '<button type="button" class="btn btn-xs ' . $gcc_class . '" 
                            onclick="toggleGCC(' . $row->id . ', ' . $is_gcc . ')" 
                            title="' . $gcc_action . '" style="margin-right: 2px;">
                            <i class="fa ' . $gcc_icon . '"></i>
                         </button>';
            
            $actions .= '<button type="button" class="btn btn-xs btn-danger" onclick="deleteTest(' . $row->id . ')" title="Delete">
                            <i class="fa fa-trash"></i>
                         </button>';
            
            $sub_array[] = $actions;
            $data[] = $sub_array;
        }

        $output = array(
            "draw" => $draw,
            "recordsTotal" => $total_records,
            "recordsFiltered" => $total_records,
            "data" => $data
        );

        echo json_encode($output);

    } catch (Exception $e) {
        // Log the error
        log_message('error', 'DataTables error: ' . $e->getMessage());
        
        echo json_encode([
            'draw' => intval($this->input->post('draw') ?: 1),
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => 'Error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Helper method to check if a column exists
 */
private function columnExists($table, $column) {
    try {
        $fields = $this->db->field_data($table);
        foreach ($fields as $field) {
            if ($field->name === $column) {
                return true;
            }
        }
        return false;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Debug endpoint to test if controller is working
 */
public function test_endpoint() {
    echo json_encode([
        'status' => 'success',
        'message' => 'Lab_setup controller is working',
        'timestamp' => date('Y-m-d H:i:s'),
        'base_url' => base_url(),
        'request_method' => $this->input->method(),
        'tables_exist' => [
            'lab_tests' => $this->db->table_exists('lab_tests'),
            'lab_categories' => $this->db->table_exists('lab_categories'),
            'lab_test_categories' => $this->db->table_exists('lab_test_categories'),
            'lab_units' => $this->db->table_exists('lab_units')
        ]
    ]);
}

/**
 * Toggle GCC status for a specific test
 */
public function toggle_gcc_status() {
    if ($this->input->method() === 'post') {
        $test_id = $this->input->post('test_id');
        $is_gcc = $this->input->post('is_gcc');
        
        if (!$test_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test ID is required'
            ]);
            return;
        }

        // Update the GCC status
        $data = array(
            'is_gcc' => $is_gcc ? 1 : 0,
            'updated_at' => date('Y-m-d H:i:s')
        );

        $this->db->where('id', $test_id);
        
        if ($this->db->update('lab_tests', $data)) {
            $action = $is_gcc ? 'enabled' : 'disabled';
            echo json_encode([
                'status' => 'success',
                'message' => 'GCC requirement ' . $action . ' successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update GCC status'
            ]);
        }
    }
}

/**
 * Bulk toggle GCC status for multiple tests
 */
public function bulk_toggle_gcc() {
    if ($this->input->method() === 'post') {
        $test_ids = $this->input->post('test_ids');
        $is_gcc = $this->input->post('is_gcc');
        
        if (!$test_ids || !is_array($test_ids)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'No tests selected'
            ]);
            return;
        }

        // Update GCC status for selected tests
        $data = array(
            'is_gcc' => $is_gcc ? 1 : 0,
            'updated_at' => date('Y-m-d H:i:s')
        );

        $this->db->where_in('id', $test_ids);
        
        if ($this->db->update('lab_tests', $data)) {
            $action = $is_gcc ? 'enabled' : 'disabled';
            $count = count($test_ids);
            echo json_encode([
                'status' => 'success',
                'message' => 'GCC requirement ' . $action . ' for ' . $count . ' tests successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update GCC status for selected tests'
            ]);
        }
    }
}

/**
 * Updated edit test method to include GCC field
 */
public function edit_test($id) {
    if ($this->input->method() === 'get') {
        // Load edit form
        $data['test'] = $this->db->get_where('lab_tests', array('id' => $id))->row();
        $data['categories'] = $this->db->get_where('lab_categories', array('is_active' => 1))->result_array();
        $data['units'] = $this->db->get_where('lab_units', array('is_active' => 1))->result_array();
        
        $this->load->view('admin/lab_setup/edit_test_modal', $data);
        
    } elseif ($this->input->method() === 'post') {
        // Process edit form
        $this->form_validation->set_rules('test_name', 'Test Name', 'required|trim');
        $this->form_validation->set_rules('short_name', 'Short Name', 'required|trim');
        $this->form_validation->set_rules('category_id', 'Category', 'required');
        $this->form_validation->set_rules('cost', 'Cost', 'required|numeric|greater_than[0]');
        
        if ($this->form_validation->run() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'error' => $this->form_validation->error_array(),
                'message' => 'Please fix the validation errors'
            ]);
            return;
        }

        // Prepare update data
        $data = array(
            'test_name' => $this->input->post('test_name'),
            'short_name' => $this->input->post('short_name'),
            'category_id' => $this->input->post('category_id'),
            'unit_id' => $this->input->post('unit_id') ? $this->input->post('unit_id') : null,
            'cost' => $this->input->post('cost'),
            'report_days' => $this->input->post('report_days') ? $this->input->post('report_days') : 1,
            'test_type' => $this->input->post('test_type'),
            'method' => $this->input->post('method'),
            'sample_type' => $this->input->post('sample_type'),
            'description' => $this->input->post('description'),
            'is_gcc' => $this->input->post('is_gcc') ? 1 : 0,  // NEW: GCC field
            'updated_at' => date('Y-m-d H:i:s')
        );

        // Check for duplicate test name (excluding current test)
        $this->db->where('test_name', $data['test_name']);
        $this->db->where('id !=', $id);
        $existing = $this->db->get('lab_tests')->row();
        
        if ($existing) {
            echo json_encode([
                'status' => 'error',
                'message' => 'A test with this name already exists'
            ]);
            return;
        }

        // Update the test
        $this->db->where('id', $id);
        
        if ($this->db->update('lab_tests', $data)) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Laboratory test updated successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update laboratory test'
            ]);
        }
    }
}

/**
 * Get GCC statistics
 */
public function get_gcc_stats() {
    // Total tests
    $total_tests = $this->db->count_all('lab_tests');
    
    // GCC required tests
    $this->db->where('is_gcc', 1);
    $gcc_tests = $this->db->count_all_results('lab_tests');
    
    // Non-GCC tests
    $non_gcc_tests = $total_tests - $gcc_tests;
    
    // GCC tests by category
    $this->db->select('lab_categories.category_name, COUNT(lab_tests.id) as count');
    $this->db->from('lab_tests');
    $this->db->join('lab_categories', 'lab_categories.id = lab_tests.category_id', 'left');
    $this->db->where('lab_tests.is_gcc', 1);
    $this->db->group_by('lab_tests.category_id');
    $gcc_by_category = $this->db->get()->result_array();
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'total_tests' => $total_tests,
            'gcc_tests' => $gcc_tests,
            'non_gcc_tests' => $non_gcc_tests,
            'gcc_percentage' => $total_tests > 0 ? round(($gcc_tests / $total_tests) * 100, 1) : 0,
            'gcc_by_category' => $gcc_by_category
        ]
    ]);
}

}