<?php
// Temporary debugging - remove in production
if (ENVIRONMENT === 'development') {
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
}
/**
 * Insurance Controller
 * File: application/controllers/admin/Insurance.php
 */

defined('BASEPATH') OR exit('No direct script access allowed');

class Insurance extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('Insurance_model');
        $this->load->library('upload');
        $this->load->helper(['file', 'download']);
    }

    /**
     * Insurance companies listing
     */
    public function index() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Insurance Companies';
        $data['companies'] = $this->Insurance_model->getInsuranceCompanies();
        $data['statistics'] = $this->Insurance_model->getPricingStatistics();
        
        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/index', $data);
        $this->load->view('layout/footer');*/
        
         $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }
    
    public function debug() {
        echo "<h1>Insurance Controller Debug</h1>";
        echo "<p style='color: green;'>✅ Controller is working!</p>";
        echo "<p><strong>Base URL:</strong> " . base_url() . "</p>";
        echo "<p><strong>Current Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
        echo "<p><strong>PHP Version:</strong> " . phpversion() . "</p>";
        echo "<p><strong>CodeIgniter Version:</strong> " . (defined('CI_VERSION') ? CI_VERSION : 'Unknown') . "</p>";
        
        echo "<h2>Database Connection Test</h2>";
        try {
            if ($this->db->initialize()) {
                echo "<p style='color: green;'>✅ Database connected successfully</p>";
                echo "<p><strong>Database:</strong> " . $this->db->database . "</p>";
                
                // Test basic query
                $query = $this->db->query("SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = DATABASE()");
                if ($query) {
                    $result = $query->row();
                    echo "<p><strong>Tables in database:</strong> " . $result->count . "</p>";
                }
            } else {
                echo "<p style='color: red;'>❌ Database connection failed</p>";
            }
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Database error: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>Model Loading Test</h2>";
        try {
            if (isset($this->Insurance_model)) {
                echo "<p style='color: green;'>✅ Insurance_model loaded successfully</p>";
                
                // Test if model has methods
                $methods = get_class_methods($this->Insurance_model);
                if ($methods) {
                    echo "<p><strong>Available methods:</strong> " . count($methods) . "</p>";
                    echo "<ul>";
                    foreach (array_slice($methods, 0, 10) as $method) {
                        if (substr($method, 0, 1) !== '_') {
                            echo "<li>{$method}</li>";
                        }
                    }
                    echo "</ul>";
                }
            } else {
                echo "<p style='color: red;'>❌ Insurance_model not loaded</p>";
            }
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Model error: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>Required Tables Check</h2>";
        $required_tables = ['insurance_companies', 'insurance_pricing', 'insurance_import_history'];
        foreach ($required_tables as $table) {
            try {
                if ($this->db->table_exists($table)) {
                    echo "<p style='color: green;'>✅ Table '{$table}' exists</p>";
                    
                    // Count records
                    $count_query = $this->db->query("SELECT COUNT(*) as count FROM {$table}");
                    if ($count_query) {
                        $count = $count_query->row()->count;
                        echo "<p style='margin-left: 20px;'><strong>Records:</strong> {$count}</p>";
                    }
                } else {
                    echo "<p style='color: red;'>❌ Table '{$table}' does not exist</p>";
                }
            } catch (Exception $e) {
                echo "<p style='color: red;'>❌ Error checking table '{$table}': " . $e->getMessage() . "</p>";
            }
        }
        
        echo "<h2>File System Check</h2>";
        $upload_dir = './uploads/insurance_imports/';
        if (is_dir($upload_dir)) {
            echo "<p style='color: green;'>✅ Upload directory exists</p>";
            if (is_writable($upload_dir)) {
                echo "<p style='color: green;'>✅ Upload directory is writable</p>";
            } else {
                echo "<p style='color: orange;'>⚠️ Upload directory is not writable</p>";
            }
        } else {
            echo "<p style='color: orange;'>⚠️ Upload directory does not exist (will be created when needed)</p>";
        }
        
        echo "<h2>Session Check</h2>";
        if ($this->session->userdata('admin_id')) {
            echo "<p style='color: green;'>✅ Admin session exists</p>";
            echo "<p><strong>Admin ID:</strong> " . $this->session->userdata('admin_id') . "</p>";
        } else {
            echo "<p style='color: orange;'>⚠️ No admin session found</p>";
        }
        
        echo "<hr>";
        echo "<p><strong>Next steps:</strong></p>";
        echo "<ul>";
        echo "<li>If this page loads successfully, your controller is working</li>";
        echo "<li>Try accessing: <a href='" . base_url() . "admin/insurance/dashboard'>" . base_url() . "admin/insurance/dashboard</a></li>";
        echo "<li>Check your routes.php file for conflicts</li>";
        echo "<li>Verify your .htaccess file is correctly configured</li>";
        echo "</ul>";
    }

    /**
     * Import pricing page
     */
public function import_pricing() {
    if ($this->input->method() == 'post') {
        // Handle the actual file upload and processing
        $this->load->library('upload');
        $this->load->library('excel'); // You'll need this library
        
        // Configure upload
        $config['upload_path'] = './uploads/imports/';
        $config['allowed_types'] = 'xlsx|xls|csv';
        $config['max_size'] = 10240; // 10MB
        
        $this->upload->initialize($config);
        
        if ($this->upload->do_upload('import_file')) {
            $file_data = $this->upload->data();
            // Process the file using your Excel library
            // Return JSON response with results
        } else {
            // Return error response
        }
    } else {
        // Load the import view
        $this->load->view('admin/insurance/import_pricing');
    }
}

    /**
     * Process Excel file upload and parsing
     */
    public function process_excel_upload() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            // Validate file upload
            if (!isset($_FILES['pricing_file']) || $_FILES['pricing_file']['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('No file uploaded or upload error occurred');
            }

            $file = $_FILES['pricing_file'];
            
            // Validate file type
            $allowed_types = ['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'application/vnd.ms-excel'];
            if (!in_array($file['type'], $allowed_types)) {
                throw new Exception('Invalid file type. Please upload Excel files only (.xlsx or .xls)');
            }

            // Validate file size (10MB max)
            if ($file['size'] > 10 * 1024 * 1024) {
                throw new Exception('File size too large. Maximum size is 10MB');
            }

            // Create upload directory if it doesn't exist
            $upload_path = './uploads/insurance_imports/';
            if (!is_dir($upload_path)) {
                mkdir($upload_path, 0777, true);
            }

            // Generate unique filename
            $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $new_filename = 'import_' . date('Ymd_His') . '_' . mt_rand(1000, 9999) . '.' . $file_extension;
            $file_path = $upload_path . $new_filename;

            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $file_path)) {
                throw new Exception('Failed to save uploaded file');
            }

            // Generate batch ID
            $batch_id = $this->Insurance_model->generateBatchId();

            // Create import history record
            $import_data = [
                'batch_id' => $batch_id,
                'file_name' => $file['name'],
                'file_size' => $file['size'],
                'file_path' => $file_path,
                'import_status' => 'processing',
                'started_at' => date('Y-m-d H:i:s'),
                'imported_by' => $this->session->userdata('admin_id')
            ];
            $import_id = $this->Insurance_model->createImportHistory($import_data);

            // Process Excel file
            $processing_result = $this->processExcelFile($file_path, $batch_id);

            // Update import history
            $this->Insurance_model->updateImportHistory($import_id, [
                'total_records' => $processing_result['total_records'],
                'processed_records' => $processing_result['processed_records'],
                'import_status' => 'completed',
                'completed_at' => date('Y-m-d H:i:s'),
                'summary_report' => json_encode($processing_result)
            ]);

            echo json_encode([
                'status' => 'success',
                'message' => 'File processed successfully',
                'batch_id' => $batch_id,
                'data' => $processing_result
            ]);

        } catch (Exception $e) {
            // Update import history if exists
            if (isset($import_id)) {
                $this->Insurance_model->updateImportHistory($import_id, [
                    'import_status' => 'failed',
                    'error_log' => $e->getMessage(),
                    'completed_at' => date('Y-m-d H:i:s')
                ]);
            }

            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Process Excel file and extract data
     */
    private function processExcelFile($file_path, $batch_id) {
        // Load PhpSpreadsheet library (you'll need to install this via Composer)
        require_once APPPATH . 'third_party/PhpSpreadsheet/vendor/autoload.php';
        
        $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file_path);
        $worksheets = $spreadsheet->getAllSheets();
        
        $processed_data = [];
        $total_records = 0;
        $processed_records = 0;
        $errors = [];

        foreach ($worksheets as $worksheet) {
            $sheet_name = $worksheet->getTitle();
            $sheet_data = $worksheet->toArray(null, true, true, true);
            
            // Skip header row and process data
            for ($row = 2; $row <= count($sheet_data); $row++) {
                if (empty($sheet_data[$row]['B'])) continue; // Skip empty rows
                
                $total_records++;
                
                $item_data = [
                    'itemId' => $sheet_data[$row]['A'] ?? $row - 1,
                    'itemName' => trim($sheet_data[$row]['B'] ?? ''),
                    'itemType' => $sheet_data[$row]['D'] ?? 'Stock',
                    'itemFamily' => $sheet_data[$row]['F'] ?? 'MEDICATION',
                    'itemPrice' => floatval($sheet_data[$row]['G'] ?? 0),
                    'insuranceCompany' => $sheet_name,
                    'batch_id' => $batch_id,
                    'row_number' => $row
                ];

                // Validate item data
                $validation_errors = $this->Insurance_model->validatePricingData($item_data);
                
                if (empty($validation_errors)) {
                    $item_data['isValid'] = true;
                    $item_data['validationErrors'] = [];
                    $processed_records++;
                } else {
                    $item_data['isValid'] = false;
                    $item_data['validationErrors'] = $validation_errors;
                    $errors[] = [
                        'sheet' => $sheet_name,
                        'row' => $row,
                        'item' => $item_data['itemName'],
                        'errors' => $validation_errors
                    ];
                }

                $processed_data[] = $item_data;
            }
        }

        return [
            'total_records' => $total_records,
            'processed_records' => $processed_records,
            'valid_records' => $processed_records,
            'invalid_records' => count($errors),
            'data' => $processed_data,
            'errors' => $errors,
            'insurance_companies' => array_unique(array_column($processed_data, 'insuranceCompany'))
        ];
    }

    /**
     * Save pricing data to database
     */
    public function save_pricing_data() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $pricing_data_json = $this->input->post('pricing_data');
            if (empty($pricing_data_json)) {
                throw new Exception('No pricing data provided');
            }

            $pricing_data = json_decode($pricing_data_json, true);
            if (!$pricing_data || !isset($pricing_data['items'])) {
                throw new Exception('Invalid pricing data format');
            }

            $items = $pricing_data['items'];
            $options = $pricing_data['options'] ?? [];
            $options['user_id'] = $this->session->userdata('admin_id');

            // Generate batch ID for this save operation
            $batch_id = $this->Insurance_model->generateBatchId();

            // Process the import
            $results = $this->Insurance_model->processImportBatch($batch_id, $items, $options);

            echo json_encode([
                'status' => 'success',
                'message' => 'Pricing data saved successfully',
                'batch_id' => $batch_id,
                'results' => $results
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * View pricing data
     */
    public function pricing() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        // Get filters from request
        $filters = [
            'insurance_id' => $this->input->get('insurance_id'),
            'item_type' => $this->input->get('item_type'),
            'item_family' => $this->input->get('item_family'),
            'search' => $this->input->get('search'),
            'status' => $this->input->get('status') ?: 'active',
            'limit' => 20,
            'offset' => ($this->input->get('page') - 1) * 20 ?: 0
        ];

        $data['title'] = 'Insurance Pricing';
        $data['pricing_items'] = $this->Insurance_model->getInsurancePricing($filters);
        $data['total_count'] = $this->Insurance_model->countInsurancePricing($filters);
        $data['companies'] = $this->Insurance_model->getInsuranceCompanies();
        $data['families'] = $this->Insurance_model->getItemFamilies();
        $data['filters'] = $filters;
        $data['current_page'] = $this->input->get('page') ?: 1;
        $data['total_pages'] = ceil($data['total_count'] / 20);

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/pricing_list', $data);
        $this->load->view('layout/footer');*/
        
         $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/pricing_list', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Edit pricing item
     */
    public function edit_pricing($id) {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        if ($this->input->method() === 'post') {
            $this->updatePricingItem($id);
            return;
        }

        $data['title'] = 'Edit Pricing Item';
        $data['pricing_item'] = $this->Insurance_model->getInsurancePricing(['id' => $id]);
        $data['companies'] = $this->Insurance_model->getInsuranceCompanies();
        $data['families'] = $this->Insurance_model->getItemFamilies();

        if (empty($data['pricing_item'])) {
            show_404();
        }

        $data['pricing_item'] = $data['pricing_item'][0];

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/edit_pricing', $data);
        $this->load->view('layout/footer');*/
        
         $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/edit_pricing', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Update pricing item
     */
    private function updatePricingItem($id) {
        try {
            $this->form_validation->set_rules('item_name', 'Item Name', 'required|min_length[2]|max_length[500]');
            $this->form_validation->set_rules('item_type', 'Item Type', 'required|in_list[Stock,Service,Medicine,Equipment,Procedure]');
            $this->form_validation->set_rules('item_price', 'Item Price', 'required|numeric|greater_than_equal_to[0]');
            $this->form_validation->set_rules('insurance_company_id', 'Insurance Company', 'required|integer');

            if ($this->form_validation->run() === FALSE) {
                $this->session->set_flashdata('error', validation_errors());
                redirect('admin/insurance/edit_pricing/' . $id);
                return;
            }

            $update_data = [
                'item_name' => $this->input->post('item_name'),
                'item_type' => $this->input->post('item_type'),
                'item_family' => $this->input->post('item_family'),
                'item_price' => $this->input->post('item_price'),
                'insurance_company_id' => $this->input->post('insurance_company_id'),
                'notes' => $this->input->post('notes'),
                'effective_date' => $this->input->post('effective_date'),
                'expiry_date' => $this->input->post('expiry_date'),
                'updated_by' => $this->session->userdata('admin_id'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            if ($this->Insurance_model->updatePricingItem($id, $update_data)) {
                $this->session->set_flashdata('success', 'Pricing item updated successfully');
            } else {
                $this->session->set_flashdata('error', 'Failed to update pricing item');
            }

        } catch (Exception $e) {
            $this->session->set_flashdata('error', 'Error: ' . $e->getMessage());
        }

        redirect('admin/insurance/pricing');
    }

    /**
     * Delete pricing item
     */
    public function delete_pricing() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_delete')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $id = $this->input->post('id');
            if (empty($id)) {
                throw new Exception('Pricing item ID is required');
            }

            if ($this->Insurance_model->deletePricingItem($id)) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Pricing item deleted successfully'
                ]);
            } else {
                throw new Exception('Failed to delete pricing item');
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Search pricing items (AJAX)
     */
    public function search_pricing() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        header('Content-Type: application/json');

        $search_term = $this->input->get('term');
        $filters = [
            'insurance_id' => $this->input->get('insurance_id'),
            'item_type' => $this->input->get('item_type')
        ];

        $results = $this->Insurance_model->searchPricing($search_term, $filters);

        echo json_encode($results);
    }

    /**
     * Export pricing data
     */
    public function export_pricing() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        $filters = [
            'insurance_id' => $this->input->get('insurance_id'),
            'item_type' => $this->input->get('item_type'),
            'item_family' => $this->input->get('item_family'),
            'status' => $this->input->get('status') ?: 'active'
        ];

        $pricing_data = $this->Insurance_model->getInsurancePricing($filters);

        // Generate CSV
        $csv_data = "Insurance Company,Item Name,Item Type,Item Family,Price (KES),Effective Date,Status\n";
        
        foreach ($pricing_data as $item) {
            $csv_data .= sprintf(
                '"%s","%s","%s","%s",%s,"%s","%s"' . "\n",
                $item['company_name'],
                str_replace('"', '""', $item['item_name']),
                $item['item_type'],
                $item['item_family'],
                $item['item_price'],
                $item['effective_date'],
                $item['status']
            );
        }

        $filename = 'insurance_pricing_' . date('Ymd_His') . '.csv';
        
        force_download($filename, $csv_data);
    }

    /**
     * Import history
     */
    public function import_history() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        $page = $this->input->get('page') ?: 1;
        $limit = 20;
        $offset = ($page - 1) * $limit;

        $data['title'] = 'Import History';
        $data['imports'] = $this->Insurance_model->getImportHistory($limit, $offset);
        $data['current_page'] = $page;

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/import_history', $data);
        $this->load->view('layout/footer');*/
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/import_history', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Insurance companies management
     */
    public function companies() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Insurance Companies';
        $data['companies'] = $this->Insurance_model->getInsuranceCompanies('all');

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/companies', $data);
        $this->load->view('layout/footer');*/
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/companies', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Add/Edit insurance company
     */
    public function edit_company($id = null) {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        if ($this->input->method() === 'post') {
            $this->saveCompany($id);
            return;
        }

        $data['title'] = $id ? 'Edit Insurance Company' : 'Add Insurance Company';
        $data['company'] = $id ? $this->Insurance_model->getInsuranceCompany($id) : null;

        if ($id && empty($data['company'])) {
            show_404();
        }

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/edit_company', $data);
        $this->load->view('layout/footer');*/
        
         $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/edit_company', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Save insurance company
     */
    private function saveCompany($id = null) {
        try {
            $this->form_validation->set_rules('company_name', 'Company Name', 'required|min_length[2]|max_length[255]');
            $this->form_validation->set_rules('contact_person', 'Contact Person', 'max_length[255]');
            $this->form_validation->set_rules('phone', 'Phone', 'max_length[50]');
            $this->form_validation->set_rules('email', 'Email', 'valid_email|max_length[255]');

            if ($this->form_validation->run() === FALSE) {
                $this->session->set_flashdata('error', validation_errors());
                redirect('admin/insurance/edit_company/' . ($id ?: ''));
                return;
            }

            $company_data = [
                'company_name' => $this->input->post('company_name'),
                'contact_person' => $this->input->post('contact_person'),
                'phone' => $this->input->post('phone'),
                'email' => $this->input->post('email'),
                'address' => $this->input->post('address'),
                'website' => $this->input->post('website'),
                'commission_rate' => $this->input->post('commission_rate') ?: 0,
                'payment_terms' => $this->input->post('payment_terms'),
                'notes' => $this->input->post('notes'),
                'status' => $this->input->post('status') ?: 'active',
                'updated_at' => date('Y-m-d H:i:s'),
                'updated_by' => $this->session->userdata('admin_id')
            ];

            if ($id) {
                // Update existing company
                if ($this->Insurance_model->updateInsuranceCompany($id, $company_data)) {
                    $this->session->set_flashdata('success', 'Insurance company updated successfully');
                } else {
                    throw new Exception('Failed to update insurance company');
                }
            } else {
                // Create new company
                $company_data['created_at'] = date('Y-m-d H:i:s');
                $company_data['created_by'] = $this->session->userdata('admin_id');
                
                if ($this->Insurance_model->createInsuranceCompany($company_data)) {
                    $this->session->set_flashdata('success', 'Insurance company created successfully');
                } else {
                    throw new Exception('Failed to create insurance company');
                }
            }

        } catch (Exception $e) {
            $this->session->set_flashdata('error', 'Error: ' . $e->getMessage());
        }

        redirect('admin/insurance/companies');
    }

    /**
     * Delete insurance company
     */
    public function delete_company() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_delete')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $id = $this->input->post('id');
            if (empty($id)) {
                throw new Exception('Company ID is required');
            }

            // Check if company has pricing data
            $pricing_count = $this->Insurance_model->countInsurancePricing(['insurance_id' => $id]);
            if ($pricing_count > 0) {
                throw new Exception('Cannot delete company with existing pricing data. Please remove pricing data first.');
            }

            if ($this->Insurance_model->deleteInsuranceCompany($id)) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Insurance company deleted successfully'
                ]);
            } else {
                throw new Exception('Failed to delete insurance company');
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Pricing comparison report
     */
    public function pricing_comparison() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Pricing Comparison';
        $data['duplicate_items'] = $this->Insurance_model->getDuplicateItems();
        $data['companies'] = $this->Insurance_model->getInsuranceCompanies();

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/pricing_comparison', $data);
        $this->load->view('layout/footer');*/
        
         $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/pricing_comparison', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get pricing comparison for specific item (AJAX)
     */
    public function get_item_comparison() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        header('Content-Type: application/json');

        $item_name = $this->input->get('item_name');
        $item_type = $this->input->get('item_type');

        if (empty($item_name) || empty($item_type)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Item name and type are required'
            ]);
            return;
        }

        $comparison_data = $this->Insurance_model->getPricingComparison($item_name, $item_type);

        echo json_encode([
            'status' => 'success',
            'data' => $comparison_data
        ]);
    }

    /**
     * Bulk approve pricing items
     */
    public function bulk_approve() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $item_ids = $this->input->post('item_ids');
            if (empty($item_ids) || !is_array($item_ids)) {
                throw new Exception('No items selected for approval');
            }

            $approved_by = $this->session->userdata('admin_id');
            
            if ($this->Insurance_model->approvePricingItems($item_ids, $approved_by)) {
                echo json_encode([
                    'status' => 'success',
                    'message' => count($item_ids) . ' items approved successfully'
                ]);
            } else {
                throw new Exception('Failed to approve items');
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Bulk reject pricing items
     */
    public function bulk_reject() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $item_ids = $this->input->post('item_ids');
            $reason = $this->input->post('reason');
            
            if (empty($item_ids) || !is_array($item_ids)) {
                throw new Exception('No items selected for rejection');
            }

            $rejected_by = $this->session->userdata('admin_id');
            
            if ($this->Insurance_model->rejectPricingItems($item_ids, $rejected_by, $reason)) {
                echo json_encode([
                    'status' => 'success',
                    'message' => count($item_ids) . ' items rejected successfully'
                ]);
            } else {
                throw new Exception('Failed to reject items');
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Dashboard statistics
     */
    public function dashboard() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Insurance Dashboard';
        $data['statistics'] = $this->Insurance_model->getPricingStatistics();
        $data['recent_imports'] = $this->Insurance_model->getImportHistory(10);
        $data['recent_changes'] = $this->Insurance_model->getRecentPriceChanges(10);

        /*$this->load->view('layout/header', $data);
        $this->load->view('layout/top_header');
        $this->load->view('layout/sidebar');
        $this->load->view('layout/menu');
        $this->load->view('admin/insurance/dashboard', $data);
        $this->load->view('layout/footer');*/
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/insurance/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Price change history for specific item
     */
    public function price_history($pricing_id) {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        header('Content-Type: application/json');

        $history = $this->Insurance_model->getPriceChangeHistory($pricing_id);

        echo json_encode([
            'status' => 'success',
            'data' => $history
        ]);
    }

    /**
     * Download sample import template
     */
    public function download_template() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
            access_denied();
        }

        // Create sample data
        $sample_data = [
            ['Item ID', 'Item Name', '', 'Item Type', '', 'Item Family', 'Item Price'],
            [1, 'TRIAMSOLONE TRIAM DENK 40MG/ML', '', 'Stock', '', 'MEDICATION', 1500.00],
            [2, 'HINGED KNEE BRACE', '', 'Stock', '', 'MEDICAL SUPPLIES', 2500.00],
            [3, 'Blood Test - Full Blood Count', '', 'Service', '', 'LABORATORY', 800.00],
            [4, 'X-Ray Chest', '', 'Service', '', 'RADIOLOGY', 1200.00],
            [5, 'General Consultation', '', 'Service', '', 'CONSULTATION', 2000.00]
        ];

        // Create CSV content
        $csv_content = '';
        foreach ($sample_data as $row) {
            $csv_content .= implode(',', $row) . "\n";
        }

        $filename = 'insurance_pricing_template.csv';
        force_download($filename, $csv_content);
    }

    /**
     * Validate uploaded data before saving
     */
    public function validate_import_data() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $data_json = $this->input->post('data');
            if (empty($data_json)) {
                throw new Exception('No data provided for validation');
            }

            $items = json_decode($data_json, true);
            if (!$items) {
                throw new Exception('Invalid data format');
            }

            $validation_results = [];
            $total_valid = 0;
            $total_invalid = 0;

            foreach ($items as $item) {
                $errors = $this->Insurance_model->validatePricingData($item);
                
                $result = [
                    'item_name' => $item['itemName'],
                    'is_valid' => empty($errors),
                    'errors' => $errors
                ];

                if (empty($errors)) {
                    $total_valid++;
                } else {
                    $total_invalid++;
                }

                $validation_results[] = $result;
            }

            echo json_encode([
                'status' => 'success',
                'summary' => [
                    'total' => count($items),
                    'valid' => $total_valid,
                    'invalid' => $total_invalid
                ],
                'results' => $validation_results
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * System cleanup - remove old import data
     */
    public function cleanup() {
        if (!$this->rbac->hasPrivilege('insurance', 'can_delete')) {
            access_denied();
        }

        header('Content-Type: application/json');

        try {
            $days = $this->input->post('days') ?: 30;
            $results = $this->Insurance_model->cleanupOldImports($days);

            echo json_encode([
                'status' => 'success',
                'message' => 'Cleanup completed successfully',
                'results' => $results
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * API endpoint for external systems
     */
    public function api_get_pricing() {
        // Basic API for getting pricing data
        header('Content-Type: application/json');

        $api_key = $this->input->get_request_header('X-API-Key');
        if (empty($api_key) || $api_key !== $this->config->item('insurance_api_key')) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            return;
        }

        $filters = [
            'insurance_id' => $this->input->get('insurance_id'),
            'item_type' => $this->input->get('item_type'),
            'search' => $this->input->get('search'),
            'status' => 'active',
            'limit' => min($this->input->get('limit') ?: 100, 1000)
        ];

        $pricing_data = $this->Insurance_model->getInsurancePricing($filters);

        echo json_encode([
            'status' => 'success',
            'data' => $pricing_data,
            'count' => count($pricing_data)
        ]);
    }


public function validate_all_data() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $validation_results = $this->Insurance_model->validateAllPricingData();
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Validation completed successfully',
            'results' => $validation_results
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function optimize_database() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $this->Insurance_model->optimizeDatabaseTables();
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Database optimization completed successfully'
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function rebuild_indexes() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $this->Insurance_model->rebuildIndexes();
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Index rebuild completed successfully'
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function export_system_report() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    try {
        $report_data = $this->Insurance_model->generateSystemReport();
        
        // Generate comprehensive CSV report
        $csv_content = $this->generateSystemReportCSV($report_data);
        
        $filename = 'insurance_system_report_' . date('Ymd_His') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($csv_content));
        
        echo $csv_content;

    } catch (Exception $e) {
        show_error('Error generating system report: ' . $e->getMessage());
    }
}

public function sync_external() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        // Placeholder for external system sync
        // Implement based on your external systems
        
        $sync_results = [
            'synced_companies' => 0,
            'synced_items' => 0,
            'errors' => []
        ];
        
        echo json_encode([
            'status' => 'success',
            'message' => 'External sync completed successfully',
            'results' => $sync_results
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

private function getSystemStatus() {
    return [
        'upload_dir_exists' => is_dir('./uploads/insurance_imports/'),
        'upload_dir_writable' => is_writable('./uploads/insurance_imports/'),
        'zip_extension' => extension_loaded('zip'),
        'excel_ready' => class_exists('ZipArchive'),
        'upload_dir_size' => $this->getUploadDirectorySize(),
        'pending_approvals' => $this->Insurance_model->countPendingApprovals()
    ];
}

private function getUploadDirectorySize() {
    $size = 0;
    if (is_dir('./uploads/insurance_imports/')) {
        $files = glob('./uploads/insurance_imports/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $size += filesize($file);
            }
        }
    }
    return round($size / 1024 / 1024, 2); // Return in MB
}

private function generateSystemReportCSV($data) {
    $csv = [];
    
    // Header
    $csv[] = "Insurance System Report - Generated on " . date('Y-m-d H:i:s');
    $csv[] = "";
    
    // Summary Statistics
    $csv[] = "SUMMARY STATISTICS";
    $csv[] = "Metric,Value";
    $csv[] = "Total Pricing Items," . ($data['statistics']['overall']['total_items'] ?? 0);
    $csv[] = "Total Insurance Companies," . ($data['statistics']['overall']['total_companies'] ?? 0);
    $csv[] = "Total Item Families," . ($data['statistics']['overall']['total_families'] ?? 0);
    $csv[] = "Average Price,KES " . number_format($data['statistics']['overall']['average_price'] ?? 0, 2);
    $csv[] = "Min Price,KES " . number_format($data['statistics']['overall']['min_price'] ?? 0, 2);
    $csv[] = "Max Price,KES " . number_format($data['statistics']['overall']['max_price'] ?? 0, 2);
    $csv[] = "";
    
    // Items by Company
    $csv[] = "ITEMS BY COMPANY";
    $csv[] = "Company,Total Items";
    if (!empty($data['statistics']['by_company'])) {
        foreach ($data['statistics']['by_company'] as $company) {
            $csv[] = '"' . $company['company_name'] . '",' . $company['total_items'];
        }
    }
    $csv[] = "";
    
    // Items by Type
    $csv[] = "ITEMS BY TYPE";
    $csv[] = "Type,Total Items";
    if (!empty($data['statistics']['by_type'])) {
        foreach ($data['statistics']['by_type'] as $type) {
            $csv[] = $type['item_type'] . ',' . $type['total_items'];
        }
    }
    $csv[] = "";
    
    // Recent Import Activity
    $csv[] = "RECENT IMPORT ACTIVITY";
    $csv[] = "Date,File Name,Total Records,Successful,Failed,Status";
    if (!empty($data['recent_imports'])) {
        foreach (array_slice($data['recent_imports'], 0, 20) as $import) {
            $csv[] = sprintf(
                '%s,"%s",%d,%d,%d,%s',
                date('Y-m-d H:i', strtotime($import['created_at'])),
                $import['file_name'],
                $import['total_records'],
                $import['successful_records'],
                $import['failed_records'],
                $import['import_status']
            );
        }
    }
    
    return implode("\n", $csv);
}
public function get_company($id) {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company = $this->Insurance_model->getInsuranceCompany($id);
        
        if ($company) {
            echo json_encode([
                'status' => 'success',
                'data' => $company
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Company not found'
            ]);
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function get_company_details($id) {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company = $this->Insurance_model->getCompanyWithStatistics($id);
        
        if ($company) {
            echo json_encode([
                'status' => 'success',
                'data' => $company
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Company not found'
            ]);
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function create_company() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_add')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $this->form_validation->set_rules('company_name', 'Company Name', 'required|min_length[2]|max_length[255]');
        $this->form_validation->set_rules('company_code', 'Company Code', 'max_length[50]');
        $this->form_validation->set_rules('contact_person', 'Contact Person', 'max_length[255]');
        $this->form_validation->set_rules('phone', 'Phone', 'max_length[50]');
        $this->form_validation->set_rules('email', 'Email', 'valid_email|max_length[255]');

        if ($this->form_validation->run() === FALSE) {
            throw new Exception(validation_errors());
        }

        // Handle file upload
        $logo_path = $this->handleLogoUpload();

        $company_data = [
            'company_name' => $this->input->post('company_name'),
            'company_code' => $this->input->post('company_code') ?: $this->generateCompanyCode($this->input->post('company_name')),
            'contact_person' => $this->input->post('contact_person'),
            'phone' => $this->input->post('phone'),
            'email' => $this->input->post('email'),
            'website' => $this->input->post('website'),
            'address' => $this->input->post('address'),
            'commission_rate' => $this->input->post('commission_rate') ?: 0,
            'payment_terms' => $this->input->post('payment_terms'),
            'notes' => $this->input->post('notes'),
            'status' => $this->input->post('status') ?: 'active',
            'logo_path' => $logo_path,
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $this->session->userdata('admin_id')
        ];

        $company_id = $this->Insurance_model->createInsuranceCompany($company_data);

        if ($company_id) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Insurance company created successfully',
                'company_id' => $company_id
            ]);
        } else {
            throw new Exception('Failed to create company');
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function update_company($id) {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $this->form_validation->set_rules('company_name', 'Company Name', 'required|min_length[2]|max_length[255]');
        $this->form_validation->set_rules('company_code', 'Company Code', 'max_length[50]');
        $this->form_validation->set_rules('contact_person', 'Contact Person', 'max_length[255]');
        $this->form_validation->set_rules('phone', 'Phone', 'max_length[50]');
        $this->form_validation->set_rules('email', 'Email', 'valid_email|max_length[255]');

        if ($this->form_validation->run() === FALSE) {
            throw new Exception(validation_errors());
        }

        // Handle file upload
        $logo_path = $this->handleLogoUpload();

        $company_data = [
            'company_name' => $this->input->post('company_name'),
            'company_code' => $this->input->post('company_code'),
            'contact_person' => $this->input->post('contact_person'),
            'phone' => $this->input->post('phone'),
            'email' => $this->input->post('email'),
            'website' => $this->input->post('website'),
            'address' => $this->input->post('address'),
            'commission_rate' => $this->input->post('commission_rate') ?: 0,
            'payment_terms' => $this->input->post('payment_terms'),
            'notes' => $this->input->post('notes'),
            'status' => $this->input->post('status'),
            'updated_at' => date('Y-m-d H:i:s'),
            'updated_by' => $this->session->userdata('admin_id')
        ];

        // Only update logo path if new file uploaded
        if ($logo_path) {
            $company_data['logo_path'] = $logo_path;
        }

        if ($this->Insurance_model->updateInsuranceCompany($id, $company_data)) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Insurance company updated successfully'
            ]);
        } else {
            throw new Exception('Failed to update company');
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function bulk_update_status() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company_ids = $this->input->post('company_ids');
        $status = $this->input->post('status');

        if (empty($company_ids) || !in_array($status, ['active', 'inactive', 'suspended'])) {
            throw new Exception('Invalid parameters');
        }

        $updated_count = $this->Insurance_model->bulkUpdateCompanyStatus($company_ids, $status, $this->session->userdata('admin_id'));

        echo json_encode([
            'status' => 'success',
            'message' => "{$updated_count} companies updated successfully"
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function bulk_delete_companies() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_delete')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company_ids = $this->input->post('company_ids');

        if (empty($company_ids)) {
            throw new Exception('No companies selected');
        }

        $deleted_count = $this->Insurance_model->bulkDeleteCompanies($company_ids);

        echo json_encode([
            'status' => 'success',
            'message' => "{$deleted_count} companies deleted successfully"
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function import_companies() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_add')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('No file uploaded or upload error');
        }

        $file = $_FILES['import_file'];
        $update_existing = $this->input->post('update_existing') === '1';

        // Process CSV file
        $imported_count = $this->Insurance_model->importCompaniesFromCSV(
            $file['tmp_name'], 
            $update_existing, 
            $this->session->userdata('admin_id')
        );

        echo json_encode([
            'status' => 'success',
            'message' => 'Companies imported successfully',
            'imported_count' => $imported_count
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function export_companies() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    $filters = [
        'status' => $this->input->get('status'),
        'search' => $this->input->get('search')
    ];

    $companies = $this->Insurance_model->getInsuranceCompanies('all', $filters);

    // Generate CSV
    $csv_data = "Company Name,Company Code,Contact Person,Phone,Email,Website,Address,Commission Rate,Payment Terms,Status,Created Date\n";
    
    foreach ($companies as $company) {
        $csv_data .= sprintf(
            '"%s","%s","%s","%s","%s","%s","%s",%s,"%s","%s","%s"' . "\n",
            str_replace('"', '""', $company['company_name']),
            $company['company_code'],
            str_replace('"', '""', $company['contact_person'] ?? ''),
            $company['phone'] ?? '',
            $company['email'] ?? '',
            $company['website'] ?? '',
            str_replace('"', '""', $company['address'] ?? ''),
            $company['commission_rate'],
            str_replace('"', '""', $company['payment_terms'] ?? ''),
            $company['status'],
            date('Y-m-d', strtotime($company['created_at']))
        );
    }

    $filename = 'insurance_companies_' . date('Ymd_His') . '.csv';
    
    force_download($filename, $csv_data);
}

public function download_companies_template() {
    $template_data = [
        ['Company Name', 'Company Code', 'Contact Person', 'Phone', 'Email', 'Website', 'Address'],
        ['Sample Insurance Ltd', 'SIL', 'John Doe', '+254700000000', 'contact@sample.com', 'https://sample.com', 'Nairobi, Kenya'],
        ['Example Coverage Co', 'ECC', 'Jane Smith', '+254711111111', 'info@example.co.ke', '', 'Mombasa, Kenya']
    ];

    $csv_content = '';
    foreach ($template_data as $row) {
        $csv_content .= implode(',', array_map(function($field) {
            return '"' . str_replace('"', '""', $field) . '"';
        }, $row)) . "\n";
    }

    force_download('insurance_companies_template.csv', $csv_content);
}

private function handleLogoUpload() {
    if (!isset($_FILES['logo_file']) || $_FILES['logo_file']['error'] === UPLOAD_ERR_NO_FILE) {
        return null;
    }

    $config['upload_path'] = './uploads/insurance_logos/';
    $config['allowed_types'] = 'gif|jpg|png|jpeg';
    $config['max_size'] = 2048; // 2MB
    $config['encrypt_name'] = TRUE;

    if (!is_dir($config['upload_path'])) {
        mkdir($config['upload_path'], 0777, true);
    }

    $this->load->library('upload', $config);

    if ($this->upload->do_upload('logo_file')) {
        $upload_data = $this->upload->data();
        return 'uploads/insurance_logos/' . $upload_data['file_name'];
    } else {
        throw new Exception('Logo upload failed: ' . $this->upload->display_errors());
    }
}

private function generateCompanyCode($company_name) {
    $words = explode(' ', strtoupper($company_name));
    $code = '';
    foreach ($words as $word) {
        if (!empty($word)) {
            $code .= substr($word, 0, 1);
        }
    }
    return substr($code, 0, 10);
}
}

?>