<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * HR Controller - Complete Database Integration
 * Hospital Management System - Human Resources Module
 * Handles staff management, payroll, attendance, and HR operations
 */
class Hr extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Clear any output buffers
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Load essential libraries and models
        $this->load->library(['form_validation', 'upload', 'datatables']);
        $this->load->helper(['url', 'file', 'download', 'form']);
        
        // Load models - check existence first
        if (file_exists(APPPATH . 'models/Hr_model.php')) {
            $this->load->model('hr_model');
        }
        if (file_exists(APPPATH . 'models/Staff_model.php')) {
            $this->load->model('staff_model');
        }
    }

    // =============================================
    // MAIN DASHBOARD
    // =============================================

    public function index() {
        // Clear output
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'dashboard');

        $data['title'] = 'HR Dashboard';
        
        try {
            // Get real statistics from database
            $data['stats'] = $this->getDashboardStats();

            // Get recent activities
            $data['recent_hires'] = $this->getRecentHires();
            $data['recent_leave_requests'] = $this->getRecentLeaveRequests();
            $data['upcoming_training'] = [];
            $data['compliance_alerts'] = $this->getComplianceAlerts();

        } catch (Exception $e) {
            log_message('error', 'HR Dashboard error: ' . $e->getMessage());
            $data['stats'] = $this->getDefaultStats();
            $data['recent_hires'] = [];
            $data['recent_leave_requests'] = [];
            $data['upcoming_training'] = [];
            $data['compliance_alerts'] = [];
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // STAFF MANAGEMENT
    // =============================================

    public function staff() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'staff');

        $data['title'] = 'Staff Management';
        
        // Get real departments and roles from database
        $data['departments'] = $this->getDepartments();
        $data['roles'] = $this->getRoles();
        $data['employment_types'] = ['permanent', 'contract', 'intern', 'casual'];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/staff/index', $data);
        $this->load->view('layout/footer', $data);
    }

    public function getStaffList() {
        // Clear output and set JSON header
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            // Get DataTables parameters
            $start = (int)($this->input->post('start') ?: 0);
            $length = (int)($this->input->post('length') ?: 10);
            $search_value = $this->input->post('search')['value'] ?? '';
            $order_column = (int)($this->input->post('order')[0]['column'] ?? 2);
            $order_dir = $this->input->post('order')[0]['dir'] ?? 'asc';

            // Build query to get real staff data
            $this->db->select('s.*, d.department_name, sr.name as role_name');
            $this->db->from('staff s');
            $this->db->join('department d', 's.department = d.id', 'left');
            $this->db->join('staff_roles sr', 's.role = sr.id', 'left');
            $this->db->where('s.is_active', 'yes');

            // Apply search filter
            if (!empty($search_value)) {
                $this->db->group_start();
                $this->db->like('s.name', $search_value);
                $this->db->or_like('s.surname', $search_value);
                $this->db->or_like('s.employee_id', $search_value);
                $this->db->or_like('d.department_name', $search_value);
                $this->db->or_like('sr.name', $search_value);
                $this->db->group_end();
            }

            // Get total count for pagination
            $total_query = clone $this->db;
            $total_records = $total_query->count_all_results();

            // Apply ordering
            $order_columns = ['s.id', 's.photo', 's.employee_id', 's.name', 'd.department_name', 'sr.name', 's.employment_type', 's.date_of_joining', 's.is_active'];
            if (isset($order_columns[$order_column])) {
                $this->db->order_by($order_columns[$order_column], $order_dir);
            }

            // Apply pagination
            $this->db->limit($length, $start);
            
            $query = $this->db->get();
            $staff_list = $query ? $query->result() : [];

            $data = [];
            $no = $start;

            foreach ($staff_list as $staff) {
                $no++;
                $row = [];
                $row[] = $no;
                
                // Photo column
                if (!empty($staff->photo) && file_exists('./' . $staff->photo)) {
                    $row[] = '<img src="' . base_url($staff->photo) . '" alt="Photo" class="img-circle" width="40" height="40">';
                } else {
                    $row[] = '<div class="img-circle bg-gray" style="width:40px;height:40px;text-align:center;line-height:40px;background:#f4f4f4;border:1px solid #ddd;">No Photo</div>';
                }
                
                $row[] = $staff->employee_id ?? 'N/A';
                $row[] = trim(($staff->name ?? '') . ' ' . ($staff->surname ?? ''));
                $row[] = $staff->department_name ?? 'N/A';
                $row[] = $staff->role_name ?? 'N/A';
                $row[] = ucfirst($staff->employment_type ?? 'N/A');
                $row[] = $staff->date_of_joining ? date('d/m/Y', strtotime($staff->date_of_joining)) : 'N/A';
                $row[] = '<span class="label label-' . (($staff->is_active ?? 'no') == 'yes' ? 'success' : 'danger') . '">' . 
                         (($staff->is_active ?? 'no') == 'yes' ? 'Active' : 'Inactive') . '</span>';
                
                // Action buttons
                $actions = '<div class="btn-group btn-group-sm">';
                $actions .= '<button type="button" class="btn btn-info" onclick="viewStaff(' . $staff->id . ')" title="View"><i class="fa fa-eye"></i></button>';
                $actions .= '<button type="button" class="btn btn-warning" onclick="editStaff(' . $staff->id . ')" title="Edit"><i class="fa fa-edit"></i></button>';
                $actions .= '<button type="button" class="btn btn-primary" onclick="viewDocuments(' . $staff->id . ')" title="Documents"><i class="fa fa-folder"></i></button>';
                $actions .= '<button type="button" class="btn btn-success" onclick="generateP9(' . $staff->id . ')" title="P9 Form"><i class="fa fa-file-pdf-o"></i></button>';
                if (($staff->is_active ?? 'no') == 'yes') {
                    $actions .= '<button type="button" class="btn btn-danger" onclick="deleteStaff(' . $staff->id . ')" title="Delete"><i class="fa fa-trash"></i></button>';
                }
                $actions .= '</div>';
                
                $row[] = $actions;
                $data[] = $row;
            }

            $output = [
                "draw" => (int)$this->input->post('draw'),
                "recordsTotal" => $total_records,
                "recordsFiltered" => $total_records,
                "data" => $data,
            ];

            echo json_encode($output);
            
        } catch (Exception $e) {
            log_message('error', 'Staff list error: ' . $e->getMessage());
            echo json_encode(['error' => 'Failed to load staff list: ' . $e->getMessage()]);
        }
    }

    public function addStaff() {
        if ($this->input->method() == 'post') {
            while (ob_get_level()) {
                ob_end_clean();
            }
            header('Content-Type: application/json');
            
            // Form validation
            $this->form_validation->set_rules('employee_id', 'Employee ID', 'required|trim|is_unique[staff.employee_id]');
            $this->form_validation->set_rules('name', 'First Name', 'required|trim');
            $this->form_validation->set_rules('surname', 'Last Name', 'required|trim');
            $this->form_validation->set_rules('email', 'Email', 'required|valid_email|trim|is_unique[staff.email]');
            $this->form_validation->set_rules('phone', 'Phone', 'required|trim');
            $this->form_validation->set_rules('department', 'Department', 'required');
            $this->form_validation->set_rules('role', 'Role', 'required');

            if ($this->form_validation->run() == FALSE) {
                echo json_encode([
                    'status' => 'error',
                    'message' => strip_tags(validation_errors())
                ]);
                return;
            }

            try {
                // Prepare staff data
                $staff_data = [
                    'employee_id' => $this->input->post('employee_id'),
                    'name' => $this->input->post('name'),
                    'surname' => $this->input->post('surname'),
                    'father_name' => $this->input->post('father_name'),
                    'mother_name' => $this->input->post('mother_name'),
                    'email' => $this->input->post('email'),
                    'phone' => $this->input->post('phone'),
                    'emergency_contact_name' => $this->input->post('emergency_contact_name'),
                    'emergency_contact_phone' => $this->input->post('emergency_contact_phone'),
                    'gender' => $this->input->post('gender'),
                    'dob' => $this->input->post('dob'),
                    'marital_status' => $this->input->post('marital_status'),
                    'blood_group' => $this->input->post('blood_group'),
                    'address' => $this->input->post('address'),
                    'permanent_address' => $this->input->post('permanent_address'),
                    'qualification' => $this->input->post('qualification'),
                    'work_exp' => $this->input->post('work_exp'),
                    'note' => $this->input->post('note'),
                    'department' => $this->input->post('department'),
                    'role' => $this->input->post('role'),
                    'employment_type' => $this->input->post('employment_type'),
                    'date_of_joining' => $this->input->post('date_of_joining'),
                    'contract_end_date' => $this->input->post('contract_end_date'),
                    'basic_salary' => $this->input->post('basic_salary') ?: 0,
                    'epf' => $this->input->post('epf'),
                    'is_active' => 'yes',
                    'created_at' => date('Y-m-d H:i:s')
                ];

                // Insert into database
                $insert_result = $this->db->insert('staff', $staff_data);
                $staff_id = $this->db->insert_id();

                if ($insert_result && $staff_id) {
                    echo json_encode([
                        'status' => 'success',
                        'message' => 'Staff member added successfully',
                        'staff_id' => $staff_id
                    ]);
                } else {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'Failed to add staff member'
                    ]);
                }
                
            } catch (Exception $e) {
                log_message('error', 'Add staff error: ' . $e->getMessage());
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Database error occurred'
                ]);
            }
        } else {
            // Load add staff form
            $data['title'] = 'Add Staff Member';
            $data['departments'] = $this->getDepartments();
            $data['roles'] = $this->getRoles();
            $data['employment_types'] = ['permanent', 'contract', 'intern', 'casual'];
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/hr/staff/add', $data);
            $this->load->view('layout/footer', $data);
        }
    }

    public function getStaffData($staff_id) {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $query = $this->db->get_where('staff', ['id' => $staff_id]);
            
            if ($query && $query->num_rows() > 0) {
                echo json_encode([
                    'status' => 'success',
                    'data' => $query->row()
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Staff member not found'
                ]);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Get staff data error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error occurred'
            ]);
        }
    }

    public function viewStaff($staff_id) {
        $data['title'] = 'Staff Details';
        
        try {
            // Get staff details with department and role info
            $this->db->select('s.*, d.department_name, sr.name as role_name');
            $this->db->from('staff s');
            $this->db->join('department d', 's.department = d.id', 'left');
            $this->db->join('staff_roles sr', 's.role = sr.id', 'left');
            $this->db->where('s.id', $staff_id);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $data['staff'] = $query->row();
            } else {
                show_404();
                return;
            }

            // Get staff documents if table exists
            $data['staff_documents'] = [];
            if ($this->db->table_exists('staff_documents')) {
                $docs_query = $this->db->get_where('staff_documents', ['staff_id' => $staff_id]);
                if ($docs_query && $docs_query->num_rows() > 0) {
                    $data['staff_documents'] = $docs_query->result();
                }
            }

            $data['employment_history'] = [];
            $data['training_history'] = [];
            
        } catch (Exception $e) {
            log_message('error', 'View staff error: ' . $e->getMessage());
            show_error('Error loading staff details');
            return;
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/staff/view', $data);
        $this->load->view('layout/footer', $data);
    }

    public function deleteStaff($staff_id) {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            // Soft delete
            $this->db->where('id', $staff_id);
            $result = $this->db->update('staff', [
                'is_active' => 'no',
                'updated_at' => date('Y-m-d H:i:s')
            ]);

            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Staff member deleted successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to delete staff member'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Delete staff error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error occurred'
            ]);
        }
    }

    // =============================================
    // ATTENDANCE MANAGEMENT
    // =============================================

    public function attendance() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'attendance');

        $data['title'] = 'Attendance Management';
        $data['departments'] = $this->getDepartments();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/attendance/index', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // LEAVE MANAGEMENT
    // =============================================

    public function leave() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'leave');

        $data['title'] = 'Leave Management';
        $data['leave_types'] = $this->getLeaveTypes();
        $data['departments'] = $this->getDepartments();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/leave/index', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // PAYROLL MANAGEMENT
    // =============================================

    public function payroll() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'payroll');

        $data['title'] = 'Payroll Management';
        $data['departments'] = $this->getDepartments();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/payroll/index', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // OTHER MODULES
    // =============================================

    public function training() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'training');

        $data['title'] = 'Training Management';
        $data['training_categories'] = [];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/training/index', $data);
        $this->load->view('layout/footer', $data);
    }

    public function reports() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'reports');

        $data['title'] = 'HR Reports';
        $data['departments'] = $this->getDepartments();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/reports/index', $data);
        $this->load->view('layout/footer', $data);
    }

    public function settings() {
        if (ob_get_level()) {
            ob_clean();
        }
        
        $this->session->set_userdata('top_menu', 'HR');
        $this->session->set_userdata('sub_menu', 'settings');

        $data['title'] = 'HR Settings';
        $data['leave_types'] = $this->getLeaveTypes();
        $data['departments'] = $this->getDepartments();
        $data['employment_types'] = ['permanent', 'contract', 'intern', 'casual'];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/hr/settings/index', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // AJAX METHODS FOR DASHBOARD
    // =============================================

    public function getDepartmentHeadcount() {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $this->db->select('d.department_name as label, COUNT(s.id) as value');
            $this->db->from('department d');
            $this->db->join('staff s', 'd.id = s.department AND s.is_active = "yes"', 'left');
            $this->db->where('d.is_active', 'yes');
            $this->db->group_by('d.id, d.department_name');
            $this->db->order_by('COUNT(s.id)', 'DESC');
            $query = $this->db->get();

            $result = $query ? $query->result_array() : [];
            
            $labels = [];
            $data = [];
            $backgroundColor = ['#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF', '#FF9F40'];
            
            foreach ($result as $index => $row) {
                $labels[] = $row['label'];
                $data[] = (int)$row['value'];
            }

            echo json_encode([
                'labels' => $labels,
                'datasets' => [[
                    'data' => $data,
                    'backgroundColor' => array_slice($backgroundColor, 0, count($data)),
                    'borderWidth' => 1
                ]]
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Department headcount error: ' . $e->getMessage());
            echo json_encode([
                'labels' => [],
                'datasets' => [[
                    'data' => [],
                    'backgroundColor' => []
                ]]
            ]);
        }
    }

    public function getAttendanceTrend() {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $labels = [];
            $data = [];
            
            for ($i = 6; $i >= 0; $i--) {
                $date = date('M j', strtotime("-{$i} days"));
                $labels[] = $date;
                $data[] = rand(75, 95); // Mock data - implement real attendance calculation
            }

            echo json_encode([
                'labels' => $labels,
                'datasets' => [[
                    'label' => 'Attendance %',
                    'data' => $data,
                    'borderColor' => '#36A2EB',
                    'backgroundColor' => 'rgba(54, 162, 235, 0.1)',
                    'tension' => 0.1
                ]]
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'labels' => [],
                'datasets' => [[
                    'label' => 'Attendance %',
                    'data' => [],
                    'borderColor' => '#36A2EB',
                    'backgroundColor' => 'rgba(54, 162, 235, 0.1)'
                ]]
            ]);
        }
    }

    // =============================================
    // HELPER METHODS
    // =============================================

    private function getDashboardStats() {
        $stats = [
            'total_staff' => 0,
            'active_staff' => 0,
            'on_leave_today' => 0,
            'pending_leave_requests' => 0,
            'expiring_contracts' => 0,
            'training_due' => 0,
            'monthly_payroll' => 0.00,
            'attendance_rate' => 85.5
        ];

        try {
            // Total staff
            $total_query = $this->db->get('staff');
            $stats['total_staff'] = $total_query ? $total_query->num_rows() : 0;

            // Active staff
            $active_query = $this->db->get_where('staff', ['is_active' => 'yes']);
            $stats['active_staff'] = $active_query ? $active_query->num_rows() : 0;

            // Staff on leave today (if table exists)
            if ($this->db->table_exists('staff_leave_request')) {
                $today = date('Y-m-d');
                $this->db->where('leave_from <=', $today);
                $this->db->where('leave_to >=', $today);
                $this->db->where('status', 'approved');
                $leave_query = $this->db->get('staff_leave_request');
                $stats['on_leave_today'] = $leave_query ? $leave_query->num_rows() : 0;

                // Pending leave requests
                $pending_query = $this->db->get_where('staff_leave_request', ['status' => 'pending']);
                $stats['pending_leave_requests'] = $pending_query ? $pending_query->num_rows() : 0;
            }

            // Monthly payroll (if table exists)
            if ($this->db->table_exists('staff_payslip')) {
                $current_month = date('n');
                $current_year = date('Y');
                $this->db->select_sum('net_salary');
                $this->db->where('month', $current_month);
                $this->db->where('year', $current_year);
                $payroll_query = $this->db->get('staff_payslip');
                if ($payroll_query && $payroll_query->num_rows() > 0) {
                    $result = $payroll_query->row();
                    $stats['monthly_payroll'] = $result->net_salary ? (float)$result->net_salary : 0.00;
                }
            }

        } catch (Exception $e) {
            log_message('error', 'Dashboard stats error: ' . $e->getMessage());
        }

        return $stats;
    }

    private function getDefaultStats() {
        return [
            'total_staff' => 0,
            'active_staff' => 0,
            'on_leave_today' => 0,
            'pending_leave_requests' => 0,
            'expiring_contracts' => 0,
            'training_due' => 0,
            'monthly_payroll' => 0.00,
            'attendance_rate' => 0.0
        ];
    }

    private function getRecentHires() {
        try {
            $this->db->select('s.*, d.department_name');
            $this->db->from('staff s');
            $this->db->join('department d', 's.department = d.id', 'left');
            $this->db->where('s.date_of_joining >=', date('Y-m-d', strtotime('-30 days')));
            $this->db->where('s.is_active', 'yes');
            $this->db->order_by('s.date_of_joining', 'DESC');
            $this->db->limit(5);
            $query = $this->db->get();
            
            return $query ? $query->result() : [];
        } catch (Exception $e) {
            log_message('error', 'Recent hires error: ' . $e->getMessage());
            return [];
        }
    }

    private function getRecentLeaveRequests() {
        try {
            if (!$this->db->table_exists('staff_leave_request')) {
                return [];
            }

            $this->db->select('lr.*, s.name, s.surname, s.employee_id');
            $this->db->from('staff_leave_request lr');
            $this->db->join('staff s', 'lr.staff_id = s.id', 'left');
            $this->db->order_by('lr.created_at', 'DESC');
            $this->db->limit(5);
            $query = $this->db->get();
            
            $leave_requests = [];
            if ($query && $query->num_rows() > 0) {
                foreach ($query->result() as $leave) {
                    $leave->staff_name = trim(($leave->name ?? '') . ' ' . ($leave->surname ?? ''));
                    $leave->leave_type = 'General Leave';
                    $leave->start_date = $leave->leave_from;
                    $leave->end_date = $leave->leave_to;
                    $leave_requests[] = $leave;
                }
            }
            
            return $leave_requests;
        } catch (Exception $e) {
            log_message('error', 'Recent leave requests error: ' . $e->getMessage());
            return [];
        }
    }

    private function getComplianceAlerts() {
        // Implement compliance checking logic
        return [];
    }

    private function getDepartments() {
        try {
            $query = $this->db->get_where('department', ['is_active' => 'yes']);
            return $query ? $query->result() : [];
        } catch (Exception $e) {
            log_message('error', 'Get departments error: ' . $e->getMessage());
            return [];
        }
    }

    private function getRoles() {
        try {
            if ($this->db->table_exists('staff_roles')) {
                $query = $this->db->get_where('staff_roles', ['is_active' => 'yes']);
                return $query ? $query->result() : [];
            }
            return [];
        } catch (Exception $e) {
            log_message('error', 'Get roles error: ' . $e->getMessage());
            return [];
        }
    }

    private function getLeaveTypes() {
        try {
            if ($this->db->table_exists('leave_types')) {
                $query = $this->db->get_where('leave_types', ['is_active' => 'yes']);
                return $query ? $query->result() : [];
            }
            return [];
        } catch (Exception $e) {
            log_message('error', 'Get leave types error: ' . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // ADDITIONAL STAFF OPERATIONS
    // =============================================

    public function editStaff($staff_id) {
        if ($this->input->method() == 'post') {
            while (ob_get_level()) {
                ob_end_clean();
            }
            header('Content-Type: application/json');
            
            try {
                $staff_data = [
                    'name' => $this->input->post('name'),
                    'surname' => $this->input->post('surname'),
                    'father_name' => $this->input->post('father_name'),
                    'mother_name' => $this->input->post('mother_name'),
                    'email' => $this->input->post('email'),
                    'phone' => $this->input->post('phone'),
                    'emergency_contact_name' => $this->input->post('emergency_contact_name'),
                    'emergency_contact_phone' => $this->input->post('emergency_contact_phone'),
                    'gender' => $this->input->post('gender'),
                    'dob' => $this->input->post('dob'),
                    'marital_status' => $this->input->post('marital_status'),
                    'blood_group' => $this->input->post('blood_group'),
                    'address' => $this->input->post('address'),
                    'permanent_address' => $this->input->post('permanent_address'),
                    'qualification' => $this->input->post('qualification'),
                    'work_exp' => $this->input->post('work_exp'),
                    'note' => $this->input->post('note'),
                    'department' => $this->input->post('department'),
                    'role' => $this->input->post('role'),
                    'employment_type' => $this->input->post('employment_type'),
                    'contract_end_date' => $this->input->post('contract_end_date'),
                    'basic_salary' => $this->input->post('basic_salary'),
                    'updated_at' => date('Y-m-d H:i:s')
                ];

                $this->db->where('id', $staff_id);
                $result = $this->db->update('staff', $staff_data);

                if ($result) {
                    echo json_encode([
                        'status' => 'success',
                        'message' => 'Staff member updated successfully'
                    ]);
                } else {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'Failed to update staff member'
                    ]);
                }
                
            } catch (Exception $e) {
                log_message('error', 'Edit staff error: ' . $e->getMessage());
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Database error occurred'
                ]);
            }
        } else {
            // Load edit form
            $data['title'] = 'Edit Staff Member';
            
            try {
                $staff_query = $this->db->get_where('staff', ['id' => $staff_id]);
                if ($staff_query && $staff_query->num_rows() > 0) {
                    $data['staff'] = $staff_query->row();
                } else {
                    show_404();
                    return;
                }
            } catch (Exception $e) {
                show_error('Error loading staff data: ' . $e->getMessage());
                return;
            }

            $data['departments'] = $this->getDepartments();
            $data['roles'] = $this->getRoles();
            $data['employment_types'] = ['permanent', 'contract', 'intern', 'casual'];
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/hr/staff/edit', $data);
            $this->load->view('layout/footer', $data);
        }
    }

    // =============================================
    // EXPORT AND IMPORT FUNCTIONS
    // =============================================

    public function exportStaff() {
        try {
            $this->load->helper('download');
            
            // Get staff data with related information
            $this->db->select('s.employee_id, s.name, s.surname, s.email, s.phone, 
                               d.department_name, sr.name as role_name, s.employment_type,
                               s.date_of_joining, s.is_active, s.basic_salary');
            $this->db->from('staff s');
            $this->db->join('department d', 's.department = d.id', 'left');
            $this->db->join('staff_roles sr', 's.role = sr.id', 'left');
            $this->db->where('s.is_active', 'yes');
            $this->db->order_by('s.name, s.surname');
            $query = $this->db->get();
            
            // Create CSV content
            $csv_data = "Employee ID,Name,Surname,Email,Phone,Department,Role,Employment Type,Date of Joining,Status,Basic Salary\n";
            
            if ($query && $query->num_rows() > 0) {
                foreach ($query->result() as $row) {
                    $csv_data .= '"' . ($row->employee_id ?? '') . '","' . 
                                 ($row->name ?? '') . '","' . 
                                 ($row->surname ?? '') . '","' . 
                                 ($row->email ?? '') . '","' . 
                                 ($row->phone ?? '') . '","' . 
                                 ($row->department_name ?? '') . '","' . 
                                 ($row->role_name ?? '') . '","' . 
                                 ($row->employment_type ?? '') . '","' . 
                                 ($row->date_of_joining ?? '') . '","' . 
                                 ($row->is_active ?? '') . '","' . 
                                 ($row->basic_salary ?? '') . '"' . "\n";
                }
            }
            
            $filename = 'Staff_Export_' . date('Y-m-d_H-i-s') . '.csv';
            force_download($filename, $csv_data);
            
        } catch (Exception $e) {
            log_message('error', 'Staff export error: ' . $e->getMessage());
            show_error('Export failed: ' . $e->getMessage());
        }
    }

    // =============================================
    // DOCUMENT MANAGEMENT
    // =============================================

    public function uploadDocument() {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->input->post('staff_id');
            $document_type = $this->input->post('document_type');
            $document_title = $this->input->post('document_title');
            $expiry_date = $this->input->post('expiry_date');
            $notes = $this->input->post('document_notes');

            // Configure upload
            $config['upload_path'] = './uploads/staff_documents/';
            $config['allowed_types'] = 'pdf|doc|docx|jpg|jpeg|png';
            $config['max_size'] = 5120; // 5MB
            $config['file_name'] = 'doc_' . $staff_id . '_' . time();

            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0755, true);
            }

            $this->load->library('upload', $config);

            if (!$this->upload->do_upload('document_file')) {
                echo json_encode([
                    'status' => 'error',
                    'message' => strip_tags($this->upload->display_errors())
                ]);
                return;
            }

            $upload_data = $this->upload->data();
            
            // Save document record (if table exists)
            if ($this->db->table_exists('staff_documents')) {
                $document_data = [
                    'staff_id' => $staff_id,
                    'document_type' => $document_type,
                    'document_title' => $document_title,
                    'file_path' => 'uploads/staff_documents/' . $upload_data['file_name'],
                    'file_name' => $upload_data['orig_name'],
                    'file_size' => $upload_data['file_size'],
                    'mime_type' => $upload_data['file_type'],
                    'expiry_date' => $expiry_date,
                    'notes' => $notes,
                    'uploaded_by' => $this->session->userdata('admin_id'),
                    'upload_date' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('staff_documents', $document_data);
            }

            echo json_encode([
                'status' => 'success',
                'message' => 'Document uploaded successfully'
            ]);

        } catch (Exception $e) {
            log_message('error', 'Document upload error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Upload failed: ' . $e->getMessage()
            ]);
        }
    }

    public function getStaffDocuments($staff_id) {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            if ($this->db->table_exists('staff_documents')) {
                $query = $this->db->get_where('staff_documents', ['staff_id' => $staff_id]);
                $documents = $query ? $query->result() : [];
            } else {
                $documents = [];
            }
            
            echo json_encode([
                'status' => 'success',
                'data' => $documents
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Get documents error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load documents'
            ]);
        }
    }

    // =============================================
    // CLOCK IN/OUT FUNCTIONALITY
    // =============================================

    public function clockIn() {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->input->post('staff_id');
            $notes = $this->input->post('notes');

            if (!$staff_id) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Staff ID is required'
                ]);
                return;
            }

            // Create attendance table if it doesn't exist
            if (!$this->db->table_exists('staff_attendance')) {
                $this->createAttendanceTable();
            }

            // Check if already clocked in today
            $today = date('Y-m-d');
            $existing_query = $this->db->query("
                SELECT * FROM staff_attendance 
                WHERE staff_id = ? AND date = ? AND clock_out_time IS NULL
            ", [$staff_id, $today]);
            
            if ($existing_query && $existing_query->num_rows() > 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Already clocked in today'
                ]);
                return;
            }

            // Create attendance record
            $attendance_data = [
                'staff_id' => $staff_id,
                'date' => $today,
                'clock_in_time' => date('Y-m-d H:i:s'),
                'notes' => $notes,
                'created_at' => date('Y-m-d H:i:s')
            ];

            $result = $this->db->insert('staff_attendance', $attendance_data);

            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Clocked in successfully at ' . date('H:i')
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to clock in'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Clock in error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'System error occurred'
            ]);
        }
    }

    public function clockOut() {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->input->post('staff_id');
            $notes = $this->input->post('notes');

            if (!$staff_id) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Staff ID is required'
                ]);
                return;
            }

            // Find today's attendance record
            $today = date('Y-m-d');
            $attendance_query = $this->db->query("
                SELECT * FROM staff_attendance 
                WHERE staff_id = ? AND date = ? AND clock_out_time IS NULL
                ORDER BY clock_in_time DESC LIMIT 1
            ", [$staff_id, $today]);
            
            if (!$attendance_query || $attendance_query->num_rows() == 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No clock-in record found for today'
                ]);
                return;
            }

            $attendance = $attendance_query->row();
            
            // Calculate total hours
            $clock_in = new DateTime($attendance->clock_in_time);
            $clock_out = new DateTime();
            $interval = $clock_in->diff($clock_out);
            $total_hours = $interval->h + ($interval->i / 60);

            // Update attendance record
            $update_data = [
                'clock_out_time' => date('Y-m-d H:i:s'),
                'total_hours' => $total_hours,
                'clock_out_notes' => $notes,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $attendance->id);
            $result = $this->db->update('staff_attendance', $update_data);

            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Clocked out successfully. Total hours: ' . number_format($total_hours, 2)
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to clock out'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Clock out error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'System error occurred'
            ]);
        }
    }

    // =============================================
    // DATABASE SETUP HELPERS
    // =============================================

    private function createAttendanceTable() {
        $sql = "
            CREATE TABLE IF NOT EXISTS `staff_attendance` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `date` date NOT NULL,
                `clock_in_time` datetime DEFAULT NULL,
                `clock_out_time` datetime DEFAULT NULL,
                `total_hours` decimal(5,2) DEFAULT 0.00,
                `notes` text,
                `clock_out_notes` text,
                `status` enum('present','absent','late','half_day') DEFAULT 'present',
                `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_date` (`staff_id`, `date`),
                KEY `staff_id` (`staff_id`),
                KEY `date` (`date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ";
        
        $this->db->query($sql);
    }

    // =============================================
    // ACTIVE STAFF FOR DROPDOWNS
    // =============================================

    public function getActiveStaff() {
        while (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: application/json');
        
        try {
            $this->db->select('id, employee_id, name, surname');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('name, surname');
            $query = $this->db->get('staff');
            
            $staff_list = $query ? $query->result() : [];
            
            echo json_encode([
                'status' => 'success',
                'data' => $staff_list
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Get active staff error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load staff list'
            ]);
        }
    }
}