<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Financial_setup extends Admin_Controller {

    function __construct() {
        parent::__construct();
        $this->load->model('financial_model');
        $this->load->library('customlib');
    }

    /**
     * Initialize Financial System
     * Run this once to set up the financial tables and data
     */
    public function initialize() {
        // Check if user has admin privileges
        if (!$this->rbac->hasPrivilege('system_settings', 'can_edit')) {
            access_denied();
        }

        $data['title'] = 'Financial System Setup';
        $data['status'] = '';
        $data['message'] = '';

        // If form is submitted
        if ($this->input->post('setup_financial')) {
            try {
                // Create financial tables
                $this->createFinancialTables();
                
                // Sync existing billing data with financial system
                $this->syncExistingBillingData();
                
                $data['status'] = 'success';
                $data['message'] = 'Financial system has been initialized successfully!';
                
            } catch (Exception $e) {
                $data['status'] = 'error';
                $data['message'] = 'Error initializing financial system: ' . $e->getMessage();
                log_message('error', 'Financial Setup Error: ' . $e->getMessage());
            }
        }

        // Check current system status
        $data['system_status'] = $this->checkSystemStatus();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/setup', $data);
        $this->load->view('layout/footer', $data);
    }

    private function createFinancialTables() {
        // Chart of Accounts
        if (!$this->db->table_exists('chart_of_accounts')) {
            $this->db->query("
                CREATE TABLE `chart_of_accounts` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `account_code` varchar(20) NOT NULL,
                    `account_name` varchar(255) NOT NULL,
                    `account_type` enum('asset','liability','equity','revenue','expense') NOT NULL,
                    `parent_id` int(11) DEFAULT NULL,
                    `is_active` enum('yes','no') DEFAULT 'yes',
                    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `account_code` (`account_code`),
                    KEY `parent_id` (`parent_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8
            ");
        }

        // Journal Entries
        if (!$this->db->table_exists('journal_entries')) {
            $this->db->query("
                CREATE TABLE `journal_entries` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `reference_no` varchar(50) NOT NULL,
                    `date` date NOT NULL,
                    `description` text,
                    `total_amount` decimal(15,2) NOT NULL,
                    `source_module` varchar(50) DEFAULT NULL,
                    `source_id` int(11) DEFAULT NULL,
                    `status` enum('draft','approved','posted') DEFAULT 'approved',
                    `created_by` int(11) DEFAULT NULL,
                    `approved_by` int(11) DEFAULT NULL,
                    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `reference_no` (`reference_no`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8
            ");
        }

        // Journal Entry Lines
        if (!$this->db->table_exists('journal_entry_lines')) {
            $this->db->query("
                CREATE TABLE `journal_entry_lines` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `journal_entry_id` int(11) NOT NULL,
                    `account_id` int(11) NOT NULL,
                    `description` text,
                    `debit_amount` decimal(15,2) DEFAULT 0.00,
                    `credit_amount` decimal(15,2) DEFAULT 0.00,
                    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    KEY `journal_entry_id` (`journal_entry_id`),
                    KEY `account_id` (`account_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8
            ");
        }

        // Insert default chart of accounts
        $this->insertDefaultAccounts();
    }

    private function insertDefaultAccounts() {
        $accounts = array(
            // Assets
            array('account_code' => '1001', 'account_name' => 'Cash', 'account_type' => 'asset'),
            array('account_code' => '1002', 'account_name' => 'Bank Account', 'account_type' => 'asset'),
            array('account_code' => '1200', 'account_name' => 'Accounts Receivable', 'account_type' => 'asset'),
            array('account_code' => '1300', 'account_name' => 'Medical Equipment', 'account_type' => 'asset'),
            
            // Liabilities
            array('account_code' => '2001', 'account_name' => 'Accounts Payable', 'account_type' => 'liability'),
            array('account_code' => '2100', 'account_name' => 'Equipment Loan', 'account_type' => 'liability'),
            
            // Equity
            array('account_code' => '3001', 'account_name' => 'Owner Equity', 'account_type' => 'equity'),
            array('account_code' => '3100', 'account_name' => 'Retained Earnings', 'account_type' => 'equity'),
            
            // Revenue
            array('account_code' => '4001', 'account_name' => 'OPD Revenue', 'account_type' => 'revenue'),
            array('account_code' => '4002', 'account_name' => 'IPD Revenue', 'account_type' => 'revenue'),
            array('account_code' => '4003', 'account_name' => 'Pathology Revenue', 'account_type' => 'revenue'),
            array('account_code' => '4004', 'account_name' => 'Radiology Revenue', 'account_type' => 'revenue'),
            array('account_code' => '4005', 'account_name' => 'Pharmacy Revenue', 'account_type' => 'revenue'),
            
            // Expenses
            array('account_code' => '5001', 'account_name' => 'Staff Salaries', 'account_type' => 'expense'),
            array('account_code' => '5002', 'account_name' => 'Medical Supplies', 'account_type' => 'expense'),
            array('account_code' => '5003', 'account_name' => 'Utilities', 'account_type' => 'expense'),
            array('account_code' => '5004', 'account_name' => 'Equipment Maintenance', 'account_type' => 'expense'),
            array('account_code' => '5005', 'account_name' => 'Office Expenses', 'account_type' => 'expense')
        );

        foreach ($accounts as $account) {
            $this->db->where('account_code', $account['account_code']);
            $existing = $this->db->get('chart_of_accounts')->row();
            
            if (!$existing) {
                $this->db->insert('chart_of_accounts', $account);
            }
        }
    }

    private function syncExistingBillingData() {
        // Sync OPD billing data
        $this->syncOPDBilling();
        
        // Sync IPD billing data
        $this->syncIPDBilling();
    }

    private function syncOPDBilling() {
        // Get OPD billing records that don't have journal entries
        $this->db->select('*');
        $this->db->from('opd_billing');
        $this->db->where('(journal_entry_id IS NULL OR journal_entry_id = 0)');
        $this->db->limit(100); // Process in batches
        $opd_bills = $this->db->get()->result_array();

        foreach ($opd_bills as $bill) {
            if ($bill['amount'] > 0) {
                $journal_id = $this->createJournalEntry(
                    'OPD-' . $bill['id'],
                    $bill['date'],
                    'OPD Consultation Revenue',
                    $bill['amount'],
                    'opd',
                    $bill['id']
                );
                
                // Update billing record with journal entry ID
                if ($journal_id) {
                    $this->db->where('id', $bill['id']);
                    $this->db->update('opd_billing', array('journal_entry_id' => $journal_id));
                }
            }
        }
    }

    private function syncIPDBilling() {
        // Get IPD billing records that don't have journal entries
        $this->db->select('*');
        $this->db->from('ipd_billing');
        $this->db->where('(journal_entry_id IS NULL OR journal_entry_id = 0)');
        $this->db->limit(100); // Process in batches
        $ipd_bills = $this->db->get()->result_array();

        foreach ($ipd_bills as $bill) {
            if ($bill['amount'] > 0) {
                $journal_id = $this->createJournalEntry(
                    'IPD-' . $bill['id'],
                    $bill['date'],
                    'IPD Treatment Revenue',
                    $bill['amount'],
                    'ipd',
                    $bill['id']
                );
                
                // Update billing record with journal entry ID
                if ($journal_id) {
                    $this->db->where('id', $bill['id']);
                    $this->db->update('ipd_billing', array('journal_entry_id' => $journal_id));
                }
            }
        }
    }

    private function createJournalEntry($reference, $date, $description, $amount, $type, $source_id) {
        // Create journal entry header
        $journal_data = array(
            'reference_no' => $reference,
            'date' => $date,
            'description' => $description,
            'total_amount' => $amount,
            'source_module' => $type,
            'source_id' => $source_id,
            'status' => 'approved'
        );
        
        $this->db->insert('journal_entries', $journal_data);
        $journal_entry_id = $this->db->insert_id();

        if (!$journal_entry_id) {
            return false;
        }

        // Get account IDs
        $cash_account = $this->getAccountByCode('1001'); // Cash
        $revenue_account = $this->getAccountByCode($type == 'opd' ? '4001' : '4002'); // OPD or IPD Revenue
        
        if (!$cash_account || !$revenue_account) {
            return false;
        }

        // Create journal entry lines - Debit Cash
        $this->db->insert('journal_entry_lines', array(
            'journal_entry_id' => $journal_entry_id,
            'account_id' => $cash_account['id'],
            'description' => 'Cash received for ' . $description,
            'debit_amount' => $amount,
            'credit_amount' => 0
        ));
        
        // Create journal entry lines - Credit Revenue
        $this->db->insert('journal_entry_lines', array(
            'journal_entry_id' => $journal_entry_id,
            'account_id' => $revenue_account['id'],
            'description' => $description,
            'debit_amount' => 0,
            'credit_amount' => $amount
        ));

        return $journal_entry_id;
    }

    private function getAccountByCode($code) {
        $this->db->where('account_code', $code);
        return $this->db->get('chart_of_accounts')->row_array();
    }

    private function checkSystemStatus() {
        $status = array();
        
        // Check if tables exist
        $status['chart_of_accounts'] = $this->db->table_exists('chart_of_accounts');
        $status['journal_entries'] = $this->db->table_exists('journal_entries');
        $status['journal_entry_lines'] = $this->db->table_exists('journal_entry_lines');
        
        // Check if default accounts exist
        if ($status['chart_of_accounts']) {
            $this->db->from('chart_of_accounts');
            $status['accounts_count'] = $this->db->count_all_results();
        } else {
            $status['accounts_count'] = 0;
        }
        
        // Check if billing data is synced
        if ($status['journal_entries']) {
            $this->db->from('journal_entries');
            $status['journal_entries_count'] = $this->db->count_all_results();
        } else {
            $status['journal_entries_count'] = 0;
        }
        
        // Check if OPD billing has journal_entry_id column
        $status['opd_billing_synced'] = $this->db->field_exists('journal_entry_id', 'opd_billing');
        
        // Check if IPD billing has journal_entry_id column
        $status['ipd_billing_synced'] = $this->db->field_exists('journal_entry_id', 'ipd_billing');
        
        return $status;
    }

    /**
     * Add journal_entry_id columns to billing tables
     */
    public function updateBillingTables() {
        if (!$this->rbac->hasPrivilege('system_settings', 'can_edit')) {
            access_denied();
        }

        try {
            // Add journal_entry_id to opd_billing if it doesn't exist
            if (!$this->db->field_exists('journal_entry_id', 'opd_billing')) {
                $this->db->query("ALTER TABLE `opd_billing` ADD COLUMN `journal_entry_id` int(11) DEFAULT NULL");
                $this->db->query("ALTER TABLE `opd_billing` ADD INDEX `idx_journal_entry` (`journal_entry_id`)");
            }
            
            // Add journal_entry_id to ipd_billing if it doesn't exist
            if (!$this->db->field_exists('journal_entry_id', 'ipd_billing')) {
                $this->db->query("ALTER TABLE `ipd_billing` ADD COLUMN `journal_entry_id` int(11) DEFAULT NULL");
                $this->db->query("ALTER TABLE `ipd_billing` ADD INDEX `idx_journal_entry` (`journal_entry_id`)");
            }
            
            echo json_encode(array('status' => 'success', 'message' => 'Billing tables updated successfully'));
            
        } catch (Exception $e) {
            echo json_encode(array('status' => 'error', 'message' => $e->getMessage()));
        }
    }

    /**
     * Test financial system
     */
    public function test() {
        header('Content-Type: application/json');
        
        try {
            $test_results = array();
            
            // Test database connectivity
            $test_results['database'] = $this->db->conn_id ? 'Connected' : 'Not Connected';
            
            // Test if financial tables exist
            $test_results['tables'] = array(
                'chart_of_accounts' => $this->db->table_exists('chart_of_accounts'),
                'journal_entries' => $this->db->table_exists('journal_entries'),
                'journal_entry_lines' => $this->db->table_exists('journal_entry_lines')
            );
            
            // Test account counts
            if ($test_results['tables']['chart_of_accounts']) {
                $this->db->from('chart_of_accounts');
                $test_results['account_count'] = $this->db->count_all_results();
            }
            
            // Test recent transactions
            if ($test_results['tables']['journal_entries']) {
                $this->db->select('COUNT(*) as count');
                $this->db->from('journal_entries');
                $this->db->where('date >=', date('Y-m-01'));
                $result = $this->db->get()->row();
                $test_results['monthly_transactions'] = $result ? $result->count : 0;
            }
            
            echo json_encode(array(
                'status' => 'success',
                'message' => 'Financial system test completed',
                'results' => $test_results
            ));
            
        } catch (Exception $e) {
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Test failed: ' . $e->getMessage()
            ));
        }
    }
}