<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Control Center Controller - Enhanced Version (Fixed)
 * Hospital Management System - Unified Admin Dashboard
 * Comprehensive system management and monitoring
 */
class Control_center extends Admin_Controller {

    private $cache_time = 300; // 5 minutes cache
    
    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            // Load models first
            $this->load->model([
                'control_center_model',
                'patient_model',
                'staff_model'
            ]);
            
            // Load libraries (removed duplicate customlib)
            $this->load->library([
                'datatables',
                'customlib',
                'form_validation',
                'upload',
                'email'
            ]);
            
            $this->load->helper(['url', 'file', 'date', 'download', 'security']);
            
            // Set session data for navigation
            $this->session->set_userdata('top_menu', 'Control Center');
            
        } catch (Exception $e) {
            log_message('error', 'Control Center initialization error: ' . $e->getMessage());
            show_error('Failed to initialize Control Center: ' . $e->getMessage());
        }
    }

    // =============================================
    // MAIN DASHBOARD
    // =============================================

    public function index() {
        $this->session->set_userdata('sub_menu', 'dashboard');
        
        $data['title'] = 'Hospital Control Center Dashboard';
        $data['page_title'] = 'Control Center';
        
        try {
            // Get cached or fresh data
            $cache_key = 'control_center_dashboard_' . date('Y-m-d-H');
            
            if (!$dashboard_data = $this->get_cache($cache_key)) {
                $dashboard_data = $this->_prepare_dashboard_data();
                $this->set_cache($cache_key, $dashboard_data, 300); // 5 min cache
            }
            
            $data = array_merge($data, $dashboard_data);
            
        } catch (Exception $e) {
            log_message('error', 'Dashboard data error: ' . $e->getMessage());
            $data['error'] = 'Unable to load dashboard data. Please try again.';
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/control_center/dashboard', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Prepare comprehensive dashboard data
     */
    private function _prepare_dashboard_data() {
        return [
            'system_overview' => $this->control_center_model->getSystemOverview(),
            'real_time_metrics' => $this->control_center_model->getRealTimeMetrics(),
            'queue_overview' => $this->control_center_model->getCurrentQueueStatus(),
            'department_stats' => $this->control_center_model->getDepartmentStats(),
            'financial_summary' => $this->control_center_model->getFinancialSummary(),
            'recent_activities' => $this->control_center_model->getRecentActivities(15),
            'system_alerts' => $this->control_center_model->getSystemAlerts(),
            'performance_metrics' => $this->control_center_model->getPerformanceMetrics(),
            'critical_notifications' => $this->control_center_model->getCriticalNotifications()
        ];
    }

    // =============================================
    // REAL-TIME DATA ENDPOINTS (AJAX)
    // =============================================

    /**
     * Get real-time dashboard updates
     */
    public function get_dashboard_data() {
        $this->_ensure_ajax();
        
        try {
            $data = [
                'timestamp' => date('Y-m-d H:i:s'),
                'system_status' => $this->control_center_model->getSystemStatus(),
                'queue_updates' => $this->control_center_model->getCurrentQueueStatus(),
                'alert_updates' => $this->control_center_model->getSystemAlerts(),
                'performance_summary' => $this->control_center_model->getPerformanceMetrics(),
                'live_metrics' => [
                    'active_users' => $this->_getActiveUsersCount(),
                    'current_queues' => $this->_getCurrentQueuesCount(),
                    'today_revenue' => $this->control_center_model->getTodayRevenue(),
                    'system_load' => $this->_getSystemLoad()
                ]
            ];
            
            $this->_json_success($data);
            
        } catch (Exception $e) {
            log_message('error', 'Dashboard data AJAX error: ' . $e->getMessage());
            $this->_json_error('Unable to fetch dashboard data');
        }
    }

    /**
     * Get system health metrics
     */
    public function get_system_health() {
        $this->_ensure_ajax();
        
        try {
            $health = $this->control_center_model->getSystemHealth();
            $this->_json_success($health);
            
        } catch (Exception $e) {
            log_message('error', 'System health error: ' . $e->getMessage());
            $this->_json_error('Unable to fetch system health data');
        }
    }

    /**
     * Get department-specific statistics
     */
    public function get_department_stats($department = null) {
        $this->_ensure_ajax();
        
        try {
            $department = $department ?: $this->input->post('department', TRUE);
            $stats = $this->control_center_model->getDepartmentDetailedStats($department);
            
            $this->_json_success($stats);
            
        } catch (Exception $e) {
            log_message('error', 'Department stats error: ' . $e->getMessage());
            $this->_json_error('Unable to fetch department statistics');
        }
    }

    // =============================================
    // REPORTS MANAGEMENT
    // =============================================

    public function reports() {
        $this->session->set_userdata('sub_menu', 'reports');
        
        $data['title'] = 'System Reports';
        $data['page_title'] = 'Control Center Reports';
        
        try {
            $data['report_categories'] = $this->control_center_model->getReportCategories();
            $data['recent_reports'] = $this->control_center_model->getRecentReports();
            $data['scheduled_reports'] = $this->control_center_model->getScheduledReports();
            $data['report_stats'] = $this->control_center_model->getReportStats();
            
        } catch (Exception $e) {
            log_message('error', 'Reports page error: ' . $e->getMessage());
            $data['error'] = 'Unable to load reports data';
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/control_center/reports', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Generate report
     */
    public function generate_report() {
        $this->_ensure_ajax();
        
        try {
            $report_type = $this->input->post('report_type', TRUE);
            $date_from = $this->input->post('date_from', TRUE);
            $date_to = $this->input->post('date_to', TRUE);
            $department = $this->input->post('department', TRUE);
            $format = $this->input->post('format', TRUE) ?: 'html';
            
            // Validate inputs
            if (empty($report_type)) {
                $this->_json_error('Report type is required');
                return;
            }
            
            $report_data = $this->control_center_model->generateReport([
                'type' => $report_type,
                'date_from' => $date_from,
                'date_to' => $date_to,
                'department' => $department,
                'format' => $format
            ]);
            
            if ($format === 'download') {
                $this->_download_report($report_data, $report_type);
            } else {
                $this->_json_success($report_data);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Report generation error: ' . $e->getMessage());
            $this->_json_error('Unable to generate report: ' . $e->getMessage());
        }
    }

    /**
     * Download report as PDF/Excel
     */
    private function _download_report($data, $type) {
        try {
            $filename = $type . '_report_' . date('Y_m_d_H_i_s');
            
            // Create simple HTML report (since PDF library may not be available)
            $html = $this->load->view('admin/control_center/report_template', ['data' => $data], true);
            
            // Force download as HTML file for now
            $this->load->helper('download');
            force_download($filename . '.html', $html);
            
        } catch (Exception $e) {
            log_message('error', 'Download report error: ' . $e->getMessage());
            $this->_json_error('Unable to download report');
        }
    }

    // =============================================
    // TICKET MANAGEMENT SYSTEM
    // =============================================

    public function tickets() {
        $this->session->set_userdata('sub_menu', 'tickets');
        
        $data['title'] = 'Ticket Management Hub';
        $data['page_title'] = 'Support Tickets';
        
        try {
            $data['ticket_stats'] = $this->control_center_model->getTicketStats();
            $data['priority_breakdown'] = $this->control_center_model->getTicketPriorityBreakdown();
            $data['staff_list'] = $this->_getActiveStaffList();
            $data['ticket_categories'] = $this->control_center_model->getTicketCategories();
            
        } catch (Exception $e) {
            log_message('error', 'Tickets page error: ' . $e->getMessage());
            $data['error'] = 'Unable to load tickets data';
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/control_center/tickets', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Get tickets data for DataTable
     */
    public function get_tickets_data() {
        $this->_ensure_ajax();
        
        try {
            $request = $this->input->post();
            $data = $this->control_center_model->getTicketsDataTable($request);
            
            echo json_encode($data);
            
        } catch (Exception $e) {
            log_message('error', 'Tickets DataTable error: ' . $e->getMessage());
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Create new ticket
     */
    public function create_ticket() {
        try {
            if ($this->input->method() === 'post') {
                // Validation rules
                $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
                $this->form_validation->set_rules('description', 'Description', 'required');
                $this->form_validation->set_rules('priority', 'Priority', 'required|in_list[low,medium,high,critical]');
                $this->form_validation->set_rules('category', 'Category', 'required');
                
                if ($this->form_validation->run()) {
                    $ticket_data = [
                        'title' => $this->input->post('title', TRUE),
                        'description' => $this->input->post('description', TRUE),
                        'priority' => $this->input->post('priority', TRUE),
                        'category' => $this->input->post('category', TRUE),
                        'assigned_to' => $this->input->post('assigned_to', TRUE),
                        'created_by' => $this->_getCurrentStaffId(),
                        'attachments' => $this->_handle_ticket_attachments()
                    ];
                    
                    $ticket_id = $this->control_center_model->createTicket($ticket_data);
                    
                    if ($ticket_id) {
                        // Send notification if assigned
                        if ($ticket_data['assigned_to']) {
                            $this->_send_ticket_notification($ticket_id, 'assigned');
                        }
                        
                        $this->session->set_flashdata('success', 'Ticket created successfully');
                        redirect('admin/control_center/tickets');
                    } else {
                        $this->session->set_flashdata('error', 'Failed to create ticket');
                    }
                }
            }
            
            // Load create ticket view
            $data['title'] = 'Create New Ticket';
            $data['staff_list'] = $this->_getActiveStaffList();
            $data['categories'] = $this->control_center_model->getTicketCategories();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/control_center/create_ticket', $data);
            $this->load->view('layout/footer');
            
        } catch (Exception $e) {
            log_message('error', 'Create ticket error: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'Unable to create ticket');
            redirect('admin/control_center/tickets');
        }
    }

    /**
     * Update ticket status
     */
    public function update_ticket_status() {
        $this->_ensure_ajax();
        
        try {
            $ticket_id = $this->input->post('ticket_id', TRUE);
            $new_status = $this->input->post('status', TRUE);
            $notes = $this->input->post('notes', TRUE);
            
            if (empty($ticket_id) || empty($new_status)) {
                $this->_json_error('Ticket ID and status are required');
                return;
            }
            
            $update_data = [
                'status' => $new_status,
                'resolution_notes' => $notes,
                'updated_by' => $this->_getCurrentStaffId()
            ];
            
            if ($new_status === 'resolved' || $new_status === 'closed') {
                $update_data['resolved_at'] = date('Y-m-d H:i:s');
                $update_data['resolved_by'] = $this->_getCurrentStaffId();
            }
            
            $result = $this->control_center_model->updateTicket($ticket_id, $update_data);
            
            if ($result) {
                $this->_send_ticket_notification($ticket_id, 'status_updated');
                $this->_json_success(['message' => 'Ticket updated successfully']);
            } else {
                $this->_json_error('Failed to update ticket');
            }
            
        } catch (Exception $e) {
            log_message('error', 'Update ticket error: ' . $e->getMessage());
            $this->_json_error('Unable to update ticket');
        }
    }

    // =============================================
    // SYSTEM MAINTENANCE
    // =============================================

    public function maintenance() {
        $this->session->set_userdata('sub_menu', 'maintenance');
        
        $data['title'] = 'System Maintenance';
        $data['page_title'] = 'System Maintenance & Administration';
        
        try {
            $data['system_health'] = $this->control_center_model->getSystemHealth();
            $data['backup_status'] = $this->control_center_model->getBackupStatus();
            $data['cache_info'] = $this->control_center_model->getCacheInfo();
            $data['log_files'] = $this->control_center_model->getLogFiles();
            $data['database_info'] = $this->control_center_model->getDatabaseInfo();
            $data['server_info'] = $this->control_center_model->getServerInfo();
            $data['maintenance_tasks'] = $this->control_center_model->getMaintenanceTasks();
            
        } catch (Exception $e) {
            log_message('error', 'Maintenance page error: ' . $e->getMessage());
            $data['error'] = 'Unable to load maintenance data';
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/control_center/maintenance', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Clear system cache
     */
    public function clear_cache() {
        $this->_ensure_ajax();
        
        try {
            $cache_types = $this->input->post('cache_types') ?: ['file', 'db'];
            $cleared = [];
            
            foreach ($cache_types as $type) {
                switch ($type) {
                    case 'file':
                        $this->_clear_file_cache();
                        $cleared[] = 'File Cache';
                        break;
                    case 'db':
                        $this->_clear_db_cache();
                        $cleared[] = 'Database Cache';
                        break;
                    case 'session':
                        $this->_clear_session_cache();
                        $cleared[] = 'Session Cache';
                        break;
                    case 'view':
                        $this->_clear_view_cache();
                        $cleared[] = 'View Cache';
                        break;
                }
            }
            
            $message = 'Successfully cleared: ' . implode(', ', $cleared);
            $this->control_center_model->logMaintenanceAction('cache_clear', $message);
            
            $this->_json_success(['message' => $message]);
            
        } catch (Exception $e) {
            log_message('error', 'Clear cache error: ' . $e->getMessage());
            $this->_json_error('Unable to clear cache');
        }
    }

    /**
     * Optimize database
     */
    public function optimize_database() {
        $this->_ensure_ajax();
        
        try {
            $tables = $this->input->post('tables') ?: 'all';
            $operations = $this->input->post('operations') ?: ['optimize'];
            
            $results = $this->control_center_model->optimizeDatabase($tables, $operations);
            
            if ($results['success']) {
                $message = "Database optimization completed. {$results['tables_processed']} tables processed.";
                $this->control_center_model->logMaintenanceAction('db_optimize', $message);
                $this->_json_success(['message' => $message, 'details' => $results]);
            } else {
                $this->_json_error('Database optimization failed: ' . $results['error']);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Database optimization error: ' . $e->getMessage());
            $this->_json_error('Unable to optimize database');
        }
    }

    /**
     * Create system backup
     */
    public function create_backup() {
        $this->_ensure_ajax();
        
        try {
            $backup_type = $this->input->post('backup_type', TRUE) ?: 'database';
            $include_files = $this->input->post('include_files') === 'true';
            
            $result = $this->control_center_model->createBackup([
                'type' => $backup_type,
                'include_files' => $include_files,
                'created_by' => $this->_getCurrentStaffId()
            ]);
            
            if ($result['success']) {
                $this->control_center_model->logMaintenanceAction('backup_created', 'Backup created: ' . $result['filename']);
                $this->_json_success($result);
            } else {
                $this->_json_error('Backup creation failed: ' . $result['error']);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Backup creation error: ' . $e->getMessage());
            $this->_json_error('Unable to create backup');
        }
    }

    /**
     * Download log file
     */
    public function download_log($filename = null) {
        try {
            if (!$filename) {
                show_404();
            }
            
            $log_path = APPPATH . 'logs/' . $filename;
            
            if (!file_exists($log_path)) {
                show_404();
            }
            
            // Security check
            if (strpos($filename, '..') !== false || strpos($filename, '/') !== false) {
                show_404();
            }
            
            $this->load->helper('download');
            force_download($filename, file_get_contents($log_path));
            
        } catch (Exception $e) {
            log_message('error', 'Log download error: ' . $e->getMessage());
            show_404();
        }
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Get current staff ID safely
     */
    private function _getCurrentStaffId() {
        try {
            if (method_exists($this->customlib, 'getStaffID')) {
                return $this->customlib->getStaffID();
            } else {
                return $this->session->userdata('admin_id') ?: 1;
            }
        } catch (Exception $e) {
            log_message('error', 'Get staff ID error: ' . $e->getMessage());
            return 1;
        }
    }

    /**
     * Get active staff list
     */
    private function _getActiveStaffList() {
        try {
            if (method_exists($this->staff_model, 'getActiveStaff')) {
                return $this->staff_model->getActiveStaff();
            } else {
                $this->db->select('id, name, surname, employee_id');
                $this->db->where('is_active', 'yes');
                $query = $this->db->get('staff');
                return $query ? $query->result_array() : [];
            }
        } catch (Exception $e) {
            log_message('error', 'Get staff list error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get active users count
     */
    private function _getActiveUsersCount() {
        try {
            if ($this->db->table_exists('ci_sessions')) {
                $this->db->where('timestamp >', time() - 1800); // 30 minutes
                return $this->db->count_all_results('ci_sessions');
            }
            return 0;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Get current queues count
     */
    private function _getCurrentQueuesCount() {
        try {
            if ($this->db->table_exists('patient_queue')) {
                $this->db->where_in('status', ['waiting', 'in_service']);
                return $this->db->count_all_results('patient_queue');
            }
            return 0;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Get system load
     */
    private function _getSystemLoad() {
        try {
            return [
                'memory_usage' => memory_get_usage(true),
                'memory_peak' => memory_get_peak_usage(true),
                'execution_time' => round((microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) * 1000, 2)
            ];
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Clear file cache
     */
    private function _clear_file_cache() {
        $cache_path = APPPATH . 'cache/';
        if (is_dir($cache_path)) {
            $files = glob($cache_path . '*');
            foreach ($files as $file) {
                if (is_file($file) && basename($file) !== 'index.html') {
                    unlink($file);
                }
            }
        }
    }

    /**
     * Clear database cache
     */
    private function _clear_db_cache() {
        try {
            $this->db->cache_delete_all();
        } catch (Exception $e) {
            log_message('debug', 'DB cache clear error: ' . $e->getMessage());
        }
    }

    /**
     * Clear session cache
     */
    private function _clear_session_cache() {
        // Don't destroy current session, just clear session data cache if exists
        $session_cache_path = APPPATH . 'cache/sessions/';
        if (is_dir($session_cache_path)) {
            $files = glob($session_cache_path . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
    }

    /**
     * Clear view cache
     */
    private function _clear_view_cache() {
        $view_cache_path = APPPATH . 'cache/views/';
        if (is_dir($view_cache_path)) {
            $files = glob($view_cache_path . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
    }

    /**
     * Handle ticket file attachments
     */
    private function _handle_ticket_attachments() {
        $attachments = [];
        
        if (!empty($_FILES['attachments']['name'][0])) {
            $config['upload_path'] = './uploads/tickets/';
            $config['allowed_types'] = 'gif|jpg|png|pdf|doc|docx|txt';
            $config['max_size'] = 2048; // 2MB
            $config['encrypt_name'] = TRUE;
            
            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0755, true);
            }
            
            $this->load->library('upload', $config);
            
            foreach ($_FILES['attachments']['name'] as $key => $name) {
                if (!empty($name)) {
                    $_FILES['file']['name'] = $_FILES['attachments']['name'][$key];
                    $_FILES['file']['type'] = $_FILES['attachments']['type'][$key];
                    $_FILES['file']['tmp_name'] = $_FILES['attachments']['tmp_name'][$key];
                    $_FILES['file']['error'] = $_FILES['attachments']['error'][$key];
                    $_FILES['file']['size'] = $_FILES['attachments']['size'][$key];
                    
                    if ($this->upload->do_upload('file')) {
                        $uploadData = $this->upload->data();
                        $attachments[] = [
                            'original_name' => $name,
                            'file_name' => $uploadData['file_name'],
                            'file_size' => $uploadData['file_size'],
                            'file_type' => $uploadData['file_type']
                        ];
                    }
                }
            }
        }
        
        return json_encode($attachments);
    }

    /**
     * Send ticket notification
     */
    private function _send_ticket_notification($ticket_id, $action) {
        try {
            $ticket = $this->control_center_model->getTicketById($ticket_id);
            if (!$ticket || !$ticket['assigned_to']) {
                return;
            }
            
            // Get staff email
            $this->db->select('email');
            $this->db->where('id', $ticket['assigned_to']);
            $query = $this->db->get('staff');
            
            if ($query && $query->num_rows() > 0) {
                $staff_email = $query->row()->email;
                
                if ($staff_email && filter_var($staff_email, FILTER_VALIDATE_EMAIL)) {
                    $this->email->from('noreply@hospital.com', 'Hospital Management System');
                    $this->email->to($staff_email);
                    $this->email->subject('Ticket Notification - ' . $ticket['title']);
                    
                    $message = "A ticket has been " . str_replace('_', ' ', $action) . ".\n\n";
                    $message .= "Ticket Number: " . $ticket['ticket_number'] . "\n";
                    $message .= "Title: " . $ticket['title'] . "\n";
                    $message .= "Priority: " . ucfirst($ticket['priority']) . "\n";
                    $message .= "Status: " . ucfirst($ticket['status']) . "\n\n";
                    $message .= "Please log in to the system to view more details.";
                    
                    $this->email->message($message);
                    $this->email->send();
                }
            }
        } catch (Exception $e) {
            log_message('error', 'Ticket notification error: ' . $e->getMessage());
        }
    }

    /**
     * Set cache data
     */
    private function set_cache($key, $data, $ttl = 300) {
        try {
            $this->load->driver('cache');
            $this->cache->save($key, $data, $ttl);
        } catch (Exception $e) {
            log_message('debug', 'Cache set error: ' . $e->getMessage());
        }
    }

    /**
     * Get cache data
     */
    private function get_cache($key) {
        try {
            $this->load->driver('cache');
            return $this->cache->get($key);
        } catch (Exception $e) {
            log_message('debug', 'Cache get error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Ensure request is AJAX
     */
    private function _ensure_ajax() {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
    }

    /**
     * Send JSON success response
     */
    private function _json_success($data = null, $message = 'Success') {
        $response = [
            'status' => 'success',
            'message' => $message,
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        if ($data !== null) {
            $response['data'] = $data;
        }
        
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
    
    /**
     * Send JSON error response
     */
    private function _json_error($message = 'Error occurred', $code = 400) {
        $response = [
            'status' => 'error',
            'message' => $message,
            'timestamp' => date('Y-m-d H:i:s'),
            'code' => $code
        ];
        
        header('Content-Type: application/json');
        echo json_encode($response);
        
        exit;
    }
}

?>